﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vfx.h>
#include <Viewer.h>
#include <Grid.h>

#include <nns/nac/nac_MemoryAllocator.h>
#include <nns/nac/nac_File.h>

#include <nn/vfx/vfx_System.h>
#include <nn/vfx/vfx_EmitterSet.h>
#include <nn/vfx/viewer/vfx_Viewer.h>
#include "EffectBinaryPlayer.h"
#include "G3dModelPreview.h"

#include "detail/util_DrawShapes.h"
#include "detail/util_DrawVfxDebugInfo.h"
#define NN_PERF_PROFILE_ENABLED
#include <nn/perf/perf_Profile.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include "detail/util_EmulateTestCustomBehavior.h"
#endif

//---------------------------------------------------------------------------
//! @brief  ヒープ
//---------------------------------------------------------------------------
class NacHeap : public nn::vfx::Heap
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    NacHeap() NN_NOEXCEPT
    {
        m_pAllocator  = nullptr;
        m_IsCreated  = false;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //! @param[in]  pAllocator  TBD
    //---------------------------------------------------------------------------
    explicit NacHeap( nns::nac::MemoryAllocator* pAllocator ) NN_NOEXCEPT
    {
        m_pAllocator  = pAllocator;
        m_IsCreated  = false;
    }

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    ~NacHeap() NN_NOEXCEPT
    {
        Finalize();
    }

    //---------------------------------------------------------------------------
    //! @brief      nns::nac::MemoryAllocatorインスタンスを設定します。
    //! @param[in]  pAllocator  TBD
    //---------------------------------------------------------------------------
    void SetNacAllocator( nns::nac::MemoryAllocator* pAllocator ) NN_NOEXCEPT
    {
        m_pAllocator  = pAllocator;
        m_IsCreated  = false;
    }

    //---------------------------------------------------------------------------
    //! @brief      ヒープを生成する
    //! @param[in]  buffer  TBD
    //! @param[in]  size    TBD
    //---------------------------------------------------------------------------
    void Initialize( void* buffer, size_t size ) NN_NOEXCEPT
    {
        m_CreatedAllocator.Initialize( buffer, size );
        SetNacAllocator( &m_CreatedAllocator );
        m_IsCreated  = true;
    }

    //---------------------------------------------------------------------------
    //! @brief      メモリを確保します。
    //! @param[in]  size        TBD
    //! @param[in]  alignment   TBD
    //---------------------------------------------------------------------------
    virtual void* Alloc( size_t size, size_t alignment = nn::vfx::Heap::Alignment_Default ) NN_NOEXCEPT
    {
        void* ptr = m_pAllocator->Alloc( size, static_cast<int>(alignment) );
        return ptr;
    }

    //---------------------------------------------------------------------------
    //! @brief      メモリを解放します。
    //! @param[in]  ptr TBD
    //---------------------------------------------------------------------------
    virtual void Free( void* ptr ) NN_NOEXCEPT
    {
        return m_pAllocator->Free( ptr );
    }

    //---------------------------------------------------------------------------
    //! @brief  終了処理
    //---------------------------------------------------------------------------
    void Finalize() NN_NOEXCEPT
    {
        if ( m_IsCreated )
        {
            m_CreatedAllocator.Finalize();
            m_IsCreated = false;
        }
    }

private:
    nns::nac::MemoryAllocator*  m_pAllocator;                       //!< アロケータへのポインタ
    nns::nac::MemoryAllocator   m_CreatedAllocator;                 //!< メモリアロケータ
    bool                        m_IsCreated;                        //!< 初期化済みチェック
};


nn::gfx::Device*                g_pGfxDevice            = nullptr;
NacHeap                         g_Heap;                                 //!< エフェクトヒープ
NacHeap                         g_VfxDynamicHeap;                       //!< Vfx動的ヒープ
void*                           g_VfxDynamicHeapBuffer  = nullptr;      //!< Vfx動的ヒープバッファ
NacHeap                         g_ConnectionHeap;                       //!< EffectMakerとの通信用ヒープ
void*                           g_ConnectionHeapBuffer  = nullptr;      //!< EffectMakerとの通信用ヒープ先メモリ
void*                           g_pEffectViewerAddr     = nullptr;      //!< Viewer用領域のアドレス
nn::vfx::System*                g_pEffectSystem         = nullptr;      //!< ＶＦＸシステム
nn::vfx::viewer::ViewerSystem*  g_pEffectViewer         = nullptr;      //!< ＶＦＸビューアシステム
EffectBinaryPlayer*             g_pEffectBinaryPlayer   = nullptr;      //!< エフェクトバイナリプレイヤー
int                             g_SamplerBaseIndex      = 0;        //!< TBD
int                             g_TextureBaseIndex      = 0;        //!< TBD

vfxdemo::TextureDescriptorIndexAllocator* g_pTextureDescriptorPoolAllocator;
vfxdemo::SamplerDescriptorIndexAllocator* g_pSamplerDescriptorPoolAllocator;

detail::DrawShapes              g_DrawShapes;

nn::util::Float3                g_PreviousResourceCameraPosition;
nn::util::Float3                g_PreviousResourceCameraTarget;
CameraData                      g_CameraData;

#if defined( NN_BUILD_CONFIG_OS_WIN )
detail::EmulateTestCustomBehavior g_EmulateTest;
#endif

//#define _USE_PRIVATE_CUSTOM_FEATURE
#ifdef  _USE_PRIVATE_CUSTOM_FEATURE
#include "../../../../../../../Samples/Sources/Applications/VfxPrivateCustomFeature/callback.h"
#endif

static void OutputWarningEmpty(const char* format, va_list args ) NN_NOEXCEPT
{
    NN_UNUSED(format);
    NN_UNUSED(args);
    return;
}

static void HandleViewerData( nn::vfx::Heap* pHeap, void* pThis, void* pBinaryData, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback ) NN_NOEXCEPT
{
    NN_UNUSED( pHeap );
    NN_UNUSED( pThis );
    NN_UNUSED( pBinaryData );

    Viewer* pVfxDemo = reinterpret_cast< Viewer* >( pThis );

    nn::vfx::detail::BinaryData* pNodeBinary = static_cast<nn::vfx::detail::BinaryData*>( pBinaryData );
    nn::vfx::detail::BinaryData* pTextureBinary = pNodeBinary->GetNextData();
    NN_UNUSED( pTextureBinary );

    if ( pNodeBinary->GetBinaryTag() == VFX_MAKE_TAG( 'V', 'I', 'W', 'D' ) )
    {
        pNodeBinary = static_cast<nn::vfx::detail::BinaryData*>( pBinaryData );
        nn::vfx::viewer::detail::ResCamera* pCamera = pNodeBinary->GetBinaryDataWithFlip<nn::vfx::viewer::detail::ResCamera>();

        // 基本設定
        pVfxDemo->SetPlaySpeed( pCamera->frameRate );
        pVfxDemo->SetGridType( pCamera->gridType );
        pVfxDemo->SetDoDebugDraw( pCamera->displayDebugInformation != 0 );
        pVfxDemo->SetDoGpuProfileDraw( pCamera->displayGpuProfile != 0 );
        pVfxDemo->SetDoGridDraw( pCamera->displayGrid != 0 );
        pVfxDemo->SetGridSize( static_cast<float>( pCamera->gridSize ) );
        pVfxDemo->SetDoDebugEmitterShapeDraw( pCamera->displayParentEmitterShape != 0 );
        pVfxDemo->SetDoDebugEmitterShapeChildDraw(pCamera->displayChildEmitterShape != 0);
        pVfxDemo->SetDoDebugShapeFieldDraw(pCamera->displayFieldShape != 0);
        pVfxDemo->SetPadState( pCamera->enablePad != 0 );
        pVfxDemo->SetDoOverDrawMode( pCamera->enableOverDrawMode );
        pVfxDemo->SetGlareLevel( pCamera->glareLevel );
        pVfxDemo->SetGridDivCount( pCamera->gridDivisionNumber );

        // カメラ位置を設定  ビューアデータのCamera情報に変更がない場合は、EffectMakerに送信したCamera情報設定する。
        nn::util::Float3 position;
        nn::util::Float3 target;

        if ( ( pCamera->position.x == g_PreviousResourceCameraPosition.x ) &&
             ( pCamera->position.y == g_PreviousResourceCameraPosition.y ) &&
             ( pCamera->position.z == g_PreviousResourceCameraPosition.z ) )
        {
            position.x = nn::util::VectorGetX(g_CameraData.position );
            position.y = nn::util::VectorGetY(g_CameraData.position );
            position.z = nn::util::VectorGetZ(g_CameraData.position );
        }
        else
        {
             position =  pCamera->position;
             g_PreviousResourceCameraPosition = pCamera->position;
             nn::util::VectorLoad( &g_CameraData.position, pCamera->position );
        }

        // 注視点を設定
        if ( ( pCamera->target.x == g_PreviousResourceCameraTarget.x ) &&
             ( pCamera->target.y == g_PreviousResourceCameraTarget.y ) &&
             ( pCamera->target.z == g_PreviousResourceCameraTarget.z ) )
        {
            target.x = nn::util::VectorGetX(g_CameraData.target );
            target.y = nn::util::VectorGetY(g_CameraData.target );
            target.z = nn::util::VectorGetZ(g_CameraData.target );
        }
        else
        {
            target = pCamera->target;
            g_PreviousResourceCameraTarget = pCamera->target;
            nn::util::VectorLoad( &g_CameraData.target, pCamera->target );
        }
        // カメラ設定
        pVfxDemo->SetCameraSettings(
            position.v[ 0 ], position.v[ 1 ], position.v[ 2 ],
            target.v[ 0 ], target.v[ 1 ], target.v[ 2 ],
            pCamera->rotation,
            pCamera->projectionType,
            pCamera->aspectRatio,
            pCamera->fieldOfView,
            pCamera->imageHeight,
            pCamera->nearClip,
            pCamera->farClip );

        // 背景カラー
        pVfxDemo->SetBackgroundColor( reinterpret_cast<float*>( &pCamera->bgColor ) );

        // 背景の乗算カラー
        nn::util::Vector4fType iColor;
        nn::util::VectorSet( &iColor, pCamera->imageColor.v[0], pCamera->imageColor.v[1], pCamera->imageColor.v[2], pCamera->imageColor.v[3] );
        pVfxDemo->SetBackgroundMultiplyColor( iColor );

        // 警告の抑制
        if ( pCamera->restrainViewerWarnings )
        {
            nn::vfx::detail::SetOutputWarningCallBack( OutputWarningEmpty );
        }
        else
        {
            nn::vfx::detail::SetOutputWarningCallBack( nullptr );
        }

        // リアルタイム編集だったらここで終了
        if ( pCamera->isRealtimeEditing == 1 ) return;

        // 背景画像
        if ( pVfxDemo->IsBackgroundTextureInitialized() == true )
        {
            pVfxDemo->FinalizeBackgroundTexture( g_pGfxDevice, pHeap );
        }

        if ( pTextureBinary != nullptr && pTextureBinary->GetBinarySize() > 32 )
        {
            nn::util::BinaryFileHeader *header = reinterpret_cast<nn::util::BinaryFileHeader *>( pTextureBinary->GetBinaryData() );
            pVfxDemo->InitializeBackgroundTexture( g_pGfxDevice, pHeap, header, header->GetFileSize(), header->GetAlignment() );
        }

        //-----------------------------------------
        // エフェクトバイナリ再生機能
        //-----------------------------------------
        nn::vfx::detail::BinaryData* pSubBinary = pNodeBinary->GetSubData();
        if( pSubBinary != nullptr )
        {
            // バイナリ転送対応
            if ( pSubBinary->GetBinaryTag() == VFX_MAKE_TAG( 'P', 'T', 'C', 'L' ) )
            {
                nn::vfx::viewer::detail::ResPtclBinary* pResParticle = pSubBinary->GetBinaryDataWithFlip<nn::vfx::viewer::detail::ResPtclBinary>();

                // 強制リロードのOn/Offを取得
                static int currentReloadCtrl = 0;
                bool forceReload = pResParticle->reloadCtrl != currentReloadCtrl;
                currentReloadCtrl = pResParticle->reloadCtrl;

                // エフェクトバイナリプレイヤークラスにパスを登録
                bool resSetup = g_pEffectBinaryPlayer->SetupEffectBinary( reinterpret_cast<const char*>( pResParticle->filePath), pRegisterTextureSlotCallback, forceReload );

                // 再生するエミッタセットのインデックス送り/戻し処理
                if ( !resSetup )
                {
                    static int currentIndexCtrl = 0;
                    int subIndex = pResParticle->esetIndexCtrl - currentIndexCtrl;
                    currentIndexCtrl = pResParticle->esetIndexCtrl;

                    if (subIndex > 0)
                    {
                        // インデックス送り
                        g_pEffectBinaryPlayer->Send();
                    }
                    else if (subIndex < 0)
                    {
                        // インデックス戻し
                        g_pEffectBinaryPlayer->Prev();
                    }
                }
            }
        }
    }
}

//------------------------------------------------------------------------------
//      ビューアメッセージの処理
//------------------------------------------------------------------------------
static void HandleViewerMessage( nn::vfx::Heap* pHeap, void* pThis, void* pBinaryData ) NN_NOEXCEPT
{
    NN_UNUSED( pHeap );
    NN_UNUSED( pThis );

    struct Config
    {
        int                 frameRate;          //!< フレームレート
        int                 resolution;         //!< 解像度
        nn::util::Vector4fType  clearColor;         //!< 背景カラー
        float               worldScaleRange;    //!< ワールドスケール範囲
        int                 worldScaleTime;     //!< ワールドスケール時間
        float               gridScale;          //!< グリッドスケール
        int                 isLinearEditMode;   //!< リニア編集モード
    };

    Config* pConfig = reinterpret_cast< Config* >( pBinaryData );
    Viewer* pVfxDemo = reinterpret_cast< Viewer* >( pThis );
    pVfxDemo->SetFrameRate( static_cast< INIT_SETTINGS_FRAME_RATE >( pConfig->frameRate ) );
}


//------------------------------------------------------------------------------
//      ビューアキャプチャメッセージの処理
//------------------------------------------------------------------------------
void HandleViewerCaptureMessage( void* pThis, const char* pCaptureFilePath, size_t captureFilePathLength ) NN_NOEXCEPT
{
    NN_UNUSED( pThis );
    NN_UNUSED( pCaptureFilePath );
    NN_UNUSED( captureFilePathLength );

    Viewer* pViewerApp = reinterpret_cast< Viewer* >( pThis );
    pViewerApp->SetCaptureFilePath( pCaptureFilePath );
}


// 動的バッファのサイズ
size_t g_MemSizeForDynamicBuffer = 5 * 1024 * 1024;     // Vfx 用メモリサイズ


//------------------------------------------------------------------------------
//  ビューアクラス初期化
//------------------------------------------------------------------------------
void Viewer::Initialize( nn::gfx::Device* pDevice,
                         nns::nac::MemoryAllocator* pAllocator,
                         vfxdemo::TextureDescriptorIndexAllocator* pTextureDescriptorPoolAllocator,
                         vfxdemo::SamplerDescriptorIndexAllocator* pSamplerDescriptorPoolAllocator, int width, int height,
                         nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback,
                         nn::vfx::RegisterSamplerSlot     pRegisterSamplerSlotCallback,
                         ViewerLaunchArg &launchArg ) NN_NOEXCEPT
{
    m_Width  = width;
    m_Height = height;
    m_GridSize = launchArg.GetGridSize();

    g_pGfxDevice = pDevice;

    g_pTextureDescriptorPoolAllocator = pTextureDescriptorPoolAllocator;
    g_pSamplerDescriptorPoolAllocator = pSamplerDescriptorPoolAllocator;

    // グレア関連の初期化
    m_GlareEffect.Initialize( pDevice, pAllocator, g_pTextureDescriptorPoolAllocator, g_pSamplerDescriptorPoolAllocator, width, height );

    // Heap
    g_Heap.SetNacAllocator( pAllocator );

    // Vfx動的ヒープ
    // GetGpuBufferSize は、動的ヒープから確保される
    g_MemSizeForDynamicBuffer = launchArg.GetDynamicBufferSize();
    g_VfxDynamicHeapBuffer = g_Heap.Alloc( g_MemSizeForDynamicBuffer + launchArg.GetGpuBufferSize() );
    g_VfxDynamicHeap.Initialize( g_VfxDynamicHeapBuffer, g_MemSizeForDynamicBuffer + launchArg.GetGpuBufferSize() );

    // System 初期化
    nn::vfx::Config  config;
    config.SetEffectHeap( &g_Heap );
    config.SetEffectDynamicHeap( &g_VfxDynamicHeap );
    config.SetEmitterSetCount( 256 );
    config.SetEmitterCount( 1024 );
    config.SetTemporaryBufferSize( launchArg.GetTemporaryBufferSize() );
    config.SetGpuBufferSize( launchArg.GetGpuBufferSize() );
    config.SetStripeCount( launchArg.GetStripeCount() );
    config.SetSuperStripeCount( launchArg.GetSuperStripeCount() );
    config.SetGfxDevice( pDevice );
    config.SetEnableComputeShaderBatchProcess();

    g_pEffectSystem = new nn::vfx::System( config );
    g_pEffectSystem->RegisterTextureViewToDescriptorPool( pRegisterTextureSlotCallback, NULL );
    g_pEffectSystem->RegisterSamplerToDescriptorPool( pRegisterSamplerSlotCallback, NULL );

    // [Experimental]厳密なGPUカウンタモードを使用するかどうか
    if( launchArg.IsEnabledPreciseGpuCounterMode() )
    {
        // MEMO: オプション指定があったら、初回に1発ONにしておく。
        g_pEffectSystem->EnablePreciseGpuCounterMode();
    }

#ifdef  _USE_PRIVATE_CUSTOM_FEATURE
    // カスタム機能のコールバックをセット
    ::SetCallbacks( g_pEffectSystem );
#endif

    g_ConnectionHeapBuffer = g_Heap.Alloc( 1024 * 1024 * 128 );
    g_ConnectionHeap.Initialize( g_ConnectionHeapBuffer, 1024 * 1024 * 128 );

    // Viewer 初期化
    g_pEffectViewerAddr = g_Heap.Alloc( sizeof( nn::vfx::viewer::ViewerSystem ), 16 );
    g_pEffectViewer = new ( g_pEffectViewerAddr )nn::vfx::viewer::ViewerSystem( &g_Heap, &g_ConnectionHeap, g_pEffectSystem );

    // ViewerData用のコールバックを登録
    g_pEffectViewer->SetViewerDataCallBack( this, HandleViewerData );

    // ViewerMessage用のコールバックを登録します。
    g_pEffectViewer->SetViewerMessageCallBack( this, HandleViewerMessage );

    // ViewerCaptureMessage用のコールバックを登録します。
    g_pEffectViewer->SetViewerCaptureMessageCallBack( this, HandleViewerCaptureMessage );

    // エフェクトバイナリプレイヤー
    g_pEffectBinaryPlayer = new EffectBinaryPlayer( g_pEffectSystem, &g_Heap, 0, 1, 0 );

    nw::eftdemo::InitializeG3dShaderResource( pDevice, &g_Heap );
    g_pEffectViewer->SetCreateModelPreviewCallback( G3dModelPreview::CreateModelPreview );
    g_pEffectViewer->SetDestroyModelPreviewCallback( G3dModelPreview::DestroyModelPreview );

    // パーティクル重なりチェック用の初期化
    detail::VisualizeParticleFillHotSpot::Initialize( pDevice, &g_Heap );
    detail::VisualizeParticleFillHotSpot::RegisterTextureViewToDescriptorPool( pRegisterTextureSlotCallback, NULL );
    detail::VisualizeParticleFillHotSpot::RegisterSamplerToDescriptorPool( pRegisterSamplerSlotCallback, NULL );

    // モデルプレビューを初期化
    G3dModelPreview::SetDevice( pDevice );
    G3dModelPreview::SetTextureDescriptorPoolAllocator( g_pTextureDescriptorPoolAllocator );
    G3dModelPreview::SetSamplerDescriptorPoolAllocator( g_pSamplerDescriptorPoolAllocator );

    // モデル表示用ラスタライザステートを初期化
    {
        nn::gfx::RasterizerState::InfoType info;
        info.SetDefault();
        info.SetCullMode(nn::gfx::CullMode_None);
        info.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
        info.SetScissorEnabled(true);
        info.SetDepthClipEnabled(false);
        m_ModelRasterizerState.Initialize(pDevice, info);
    }

    // 背景画像
    {
        m_BackgroundTexRepresentedId = 0;

        nn::gfx::Sampler::InfoType info;
        info.SetDefault();
        info.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
        info.SetAddressU(nn::gfx::TextureAddressMode_Mirror);
        info.SetAddressV(nn::gfx::TextureAddressMode_Mirror);
        info.SetAddressW(nn::gfx::TextureAddressMode_Mirror);
        m_BackgroundSampler.Initialize(pDevice, info);

        // ディスクリプタスロットを確保
        g_pSamplerDescriptorPoolAllocator->AllocateStaticSamplerDescriptorSlot(&m_BackgroundSamplerDesSlot, m_BackgroundSampler, NULL);
    }

    // デバッグ表示の初期状態
    m_DoDebugDraw = true;
    m_DoGridDraw = true;
    m_DoGpuProfileDraw = false;
    m_DoDebugEmitterShapeDraw = false;
    m_DoDebugEmitterShapeDrawChild = false;
    m_DoDebugShapeDrawField = false;
    m_DoOverDrawMode = false;

#if defined( NN_BUILD_CONFIG_OS_WIN )
    m_IsMouseLButtonHold = false;
    m_IsMouseRButtonHold = false;
    m_IsMouseMButtonHold = false;
    m_MouseWheel = 0;;
#endif
    m_LStickX = 0.0f;
    m_LStickY = 0.0f;
}

//------------------------------------------------------------------------------
//  ビューアクラス破棄
//------------------------------------------------------------------------------
void Viewer::Finalize(nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator,
    nn::vfx::UnregisterTextureViewSlot pUnregisterTextureSlotCallback,
    nn::vfx::UnregisterSamplerSlot pUnregisterSamplerSlotCallback) NN_NOEXCEPT
{
    detail::VisualizeParticleFillHotSpot::UnregisterTextureViewFromDescriptorPool( pUnregisterTextureSlotCallback, NULL );
    detail::VisualizeParticleFillHotSpot::UnregisterSamplerFromDescriptorPool( pUnregisterSamplerSlotCallback, NULL );
    detail::VisualizeParticleFillHotSpot::Finalize( pDevice, &g_Heap );

    if (IsBackgroundTextureInitialized() == true)
    {
        FinalizeBackgroundTexture(pDevice, &g_Heap);
    }

    g_pEffectViewer->Finalize( pUnregisterTextureSlotCallback, NULL );
    g_Heap.Free( g_pEffectViewerAddr );

    g_pEffectSystem->UnregisterTextureViewFromDescriptorPool(pUnregisterTextureSlotCallback, NULL);
    g_pEffectSystem->UnregisterSamplerFromDescriptorPool(pUnregisterSamplerSlotCallback, NULL);
    nw::eftdemo::FinalizeG3dShaderResource(pDevice, &g_Heap);
    delete g_pEffectSystem;
    g_VfxDynamicHeap.Finalize();
    g_Heap.Free( g_VfxDynamicHeapBuffer );
    g_ConnectionHeap.Finalize();
    g_Heap.Free(g_ConnectionHeapBuffer);

    g_Heap.Finalize();

    m_GlareEffect.Finalize(pDevice, pAllocator);
    m_BackgroundSampler.Finalize(pDevice);
    m_ModelRasterizerState.Finalize(pDevice);
}

#if defined( NN_BUILD_CONFIG_OS_WIN )
//------------------------------------------------------------------------------
//  テストエミュレートモード時のカメラ位置を更新
//------------------------------------------------------------------------------
void Viewer::AdjustCameraFromUserData( nns::nac::Pad* pPad, vfxdemo::DrawParam* pDrawParam ) NN_NOEXCEPT
{
    // Shift + AキーでテストのCustom Behaviorをエミュレート
    if ( pPad->IsTrigger( nns::nac::Pad::MASK_A ) )  // Aキーが押されている
    {
        // 右左どちらかのShiftキーが押されている
        if (pPad->IsHold(nns::nac::Pad::MASK_LS_SHIFT) || pPad->IsHold(nns::nac::Pad::MASK_RS_SHIFT))
        {
            m_DoEmulatingTest = !m_DoEmulatingTest;
        }
    }

    if ( m_DoEmulatingTest )
    {
        g_EmulateTest.Update(g_pEffectSystem, m_Width, m_Height, g_pEffectViewer->GetTime() );
        pDrawParam->m_ViewMtx   = *g_EmulateTest.GetView();
        pDrawParam->m_ProjMtx   = *g_EmulateTest.GetProjMatrix();
        pDrawParam->m_CameraPos = *g_EmulateTest.GetCurrentCameraPos();
    }
}
#endif
//------------------------------------------------------------------------------
//  計算処理
//------------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
void Viewer::ProcCalc( nns::nac::Mouse* pMouse, nns::nac::Pad* pPad, const vfxdemo::DrawParam* pDrawParam, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlot ) NN_NOEXCEPT
#else
void Viewer::ProcCalc( nns::nac::Pad* pPad, const vfxdemo::DrawParam* pDrawParam, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlot ) NN_NOEXCEPT
#endif
{
    // Lボタンでデバッグ描画の有効/無効を切り替える
    if (pPad->IsTrigger(nns::nac::Pad::MASK_L))
    {
        m_DoDebugDraw = !m_DoDebugDraw;
    }

    // 3Dモデルプレビューモデルの計算処理
    {
        nn::vfx::viewer::detail::Preview* preview = g_pEffectViewer->GetPreviewHead();
        while (preview)
        {
            if (preview->GetPreviewType() == nn::vfx::viewer::detail::Preview::VfxPreviewType_ModelPreview)
            {
                G3dModelPreview* modelPreview = reinterpret_cast<G3dModelPreview*>(preview);
                modelPreview->CalcBlock(&pDrawParam->m_ProjMtx, &pDrawParam->m_ViewMtx, &pDrawParam->m_CameraPos);
            }
            preview = preview->GetNextPreview();
        }
    }

    bool isTransportCameraData = false;

#if defined( NN_BUILD_CONFIG_OS_WIN )
    // マウスボタンを押して離した瞬間またはホイールの操作時に、カメラ情報をEffectMakerへ送信する。
    if ( (!pMouse->IsHold( nns::nac::Mouse::MASK_LBUTTON ) && m_IsMouseLButtonHold) ||
         (!pMouse->IsHold( nns::nac::Mouse::MASK_RBUTTON ) && m_IsMouseRButtonHold) ||
         (!pMouse->IsHold( nns::nac::Mouse::MASK_MBUTTON ) && m_IsMouseMButtonHold) ||
         ( pMouse->GetWheel() == 0                         && m_MouseWheel != 0   ) )
    {
        isTransportCameraData = true;
    }

    // マウスボタンとホイールの状態を更新
    m_IsMouseLButtonHold = pMouse->IsHold( nns::nac::Mouse::MASK_LBUTTON );
    m_IsMouseRButtonHold = pMouse->IsHold( nns::nac::Mouse::MASK_RBUTTON );
    m_IsMouseMButtonHold = pMouse->IsHold( nns::nac::Mouse::MASK_MBUTTON );
    m_MouseWheel = pMouse->GetWheel();
#endif

    // パッド操作によるカメラデータ転送
    {
        if ( pPad->IsRelease(0xFFFFFFFF) ||
             ( (m_LStickX != 0.0f && pPad->GetLeftStickX() == 0.0f ) ||
               (m_LStickY != 0.0f && pPad->GetLeftStickY() == 0.0f ) )
           )
        {
            isTransportCameraData = true;
        }

        m_LStickX = pPad->GetLeftStickX();
        m_LStickY = pPad->GetLeftStickY();
    }

    // ビューアにカメラ情報を転送する
    if ( isTransportCameraData )
    {
        g_CameraData.position = *m_pCamera->GetPos();
        g_CameraData.target   = *m_pCamera->GetLookAtPos();
        g_pEffectViewer->SendBinaryData(static_cast<void*>(&g_CameraData), sizeof(g_CameraData));
    }

#if defined( NN_BUILD_CONFIG_OS_WIN )

    // グリッド表示のOn/Off
    if ( pPad->IsTrigger( nns::nac::Pad::MASK_G ) ) SetDoGridDraw( !DoGridDraw() );

    // デバックエミッタ表示のOn/Off
    if ( pPad->IsTrigger( nns::nac::Pad::MASK_E ) ) SetDoDebugEmitterShapeDraw( !DoDebugEmitterShapeDraw() );

    // デバックチャイルドエミッタ表示のOn/Off
    if ( pPad->IsTrigger( nns::nac::Pad::MASK_C ) ) SetDoDebugEmitterShapeChildDraw( !DoDebugEmitterShapeChildDraw() );

    // デバックフィールド表示のOn/Off
    if ( pPad->IsTrigger( nns::nac::Pad::MASK_F ) ) SetDoDebugShapeFieldDraw( !DoDebugShapeFieldDraw() );
#endif
    // ビューア計算処理
    ProcVfxCalc(pDrawParam, pRegisterTextureSlot);
}


//------------------------------------------------------------------------------
//  ビューア計算処理
//------------------------------------------------------------------------------
void Viewer::ProcVfxCalc(const vfxdemo::DrawParam* pDrawParam, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlot) NN_NOEXCEPT
{
    // ビューア計算処理を行い返り値をみて、ビューアグループのエフェクト計算処理を分岐
    const bool processResult = g_pEffectViewer->ProcessCalculation( GetPlaySpeed(), pDrawParam->m_ViewMtx, NULL, NULL );

    // リソース変更が発生した場合は、ディスクリプタスロットの振り直しを行う
    if ( g_pEffectViewer->IsReloadBinary() )
    {
        g_pTextureDescriptorPoolAllocator->BeginAllocateTransientDescriptorSlot();
        g_pSamplerDescriptorPoolAllocator->BeginAllocateTransientDescriptorSlot();

        // ・エフェクトビューア内エフェクトリソース
        // ・エフェクトバイナリプレイヤーのリソース
        // のスロット振り直し
        for (int i = 0; i < g_pEffectSystem->GetResourceCountMax(); i++)
        {
            nn::vfx::Resource* res = g_pEffectSystem->GetResource(i);
            if (res)
            {
                res->RegisterTextureViewToDescriptorPool(pRegisterTextureSlot, NULL);
            }
        }

        // 背景画像
        if ( IsBackgroundTextureInitialized() == true )
        {
            const nn::gfx::TextureView *backgroundTextureView = GetBackgroundTexture()->GetTextureView();
            g_pTextureDescriptorPoolAllocator->AllocateTransientTextureDescriptorSlot(&m_BackgroundTextureDesSlot, *backgroundTextureView, nullptr);
        }

        // 3Dモデル
        nn::vfx::viewer::detail::Preview* preview = g_pEffectViewer->GetPreviewHead();
        while (preview)
        {
            if (preview->GetPreviewType() == nn::vfx::viewer::detail::Preview::VfxPreviewType_ModelPreview)
            {
                G3dModelPreview* modelPreview = reinterpret_cast<G3dModelPreview*>(preview);
                modelPreview->UpdateDescriptorSlot(g_pTextureDescriptorPoolAllocator, g_pSamplerDescriptorPoolAllocator);
            }
            preview = preview->GetNextPreview();
        }

        g_pSamplerDescriptorPoolAllocator->EndAllocateTransientDescriptorSlot();
        g_pTextureDescriptorPoolAllocator->EndAllocateTransientDescriptorSlot();
    }

    // エフェクトの計算処理
    NN_PERF_SET_COLOR( nn::util::Color4u8::Green() );
    NN_PERF_BEGIN_MEASURE_NAME( "ProcVfxCalc" );
    {
        g_pEffectSystem->BeginFrame();

        if( !processResult )
        {
            g_pEffectSystem->Calculate( 0, 1.0f, nn::vfx::BufferSwapMode_Auto );                                                    // EffectBinaryPlayer用
            g_pEffectSystem->Calculate( nn::vfx::SystemParameters_ViewerGroupId, GetPlaySpeed(), nn::vfx::BufferSwapMode_Auto );    // 再生中
        }
        else
        {
            g_pEffectSystem->Calculate( 0, 1.0f, nn::vfx::BufferSwapMode_Auto );                                                    // EffectBinaryPlayer用
            g_pEffectSystem->Calculate( nn::vfx::SystemParameters_ViewerGroupId, 0.0f, nn::vfx::BufferSwapMode_Auto );              // 一時停止
        }
    }
    NN_PERF_END_MEASURE();
}

//------------------------------------------------------------------------------
//  描画処理
//------------------------------------------------------------------------------
void Viewer::ProcDraw( const vfxdemo::DrawParam* pDrawParam, GpuStatistics* gpuStatistics, const ViewerLaunchArg* pViewerLaunchArg ) NN_NOEXCEPT
{
    // ビューア描画処理
    ProcViewerDraw( pDrawParam );

    // レンダーターゲットを戻す
    nn::gfx::ColorTargetView* pTarget = pDrawParam->m_ColorTargetView;
    pDrawParam->m_CommandBuffer->SetRenderTargets( 1, &pTarget, pDrawParam->m_DepthStencilView );

    // エフェクト描画処理
    NN_PERF_SET_COLOR( nn::util::Color4u8::Red() );
    NN_PERF_BEGIN_MEASURE_NAME( "ProcVfxDraw" );
    {
        ProcVfxDraw( pDrawParam, gpuStatistics, pViewerLaunchArg->GetTextureOriginMode() );
    }
    NN_PERF_END_MEASURE();

    // DebugEmitterShapeDraw
    if ( DoDebugEmitterShapeDraw() || DoDebugEmitterShapeChildDraw() || DoDebugShapeFieldDraw() )
    {
        DebugEmitterShapeDraw( pDrawParam->m_CommandBuffer,
                               pDrawParam->m_PrimitiveRenderer,
                               &pDrawParam->m_ProjMtx,
                               &pDrawParam->m_ViewMtx  );
    }

    // グレア描画処理
    if ( m_GlareBufferNum != 0 )
    {
        m_GlareEffect.Draw(pDrawParam, m_GlareBufferNum );
    }
}


//------------------------------------------------------------------------------
//  ビューア描画処理
//------------------------------------------------------------------------------
void Viewer::ProcViewerDraw( const vfxdemo::DrawParam* pDrawParam ) NN_NOEXCEPT
{
    // 背景テクスチャが存在する場合
    if ( IsBackgroundTextureInitialized() == true  )
    {
        pDrawParam->m_PrimitiveRenderer->SetBlendState(pDrawParam->m_CommandBuffer,
            nns::gfx::PrimitiveRenderer::BlendType::BlendType_Opacity );
        pDrawParam->m_PrimitiveRenderer->SetDepthStencilState( pDrawParam->m_CommandBuffer,
            nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest );

        pDrawParam->m_PrimitiveRenderer->SetProjectionMatrix( &m_BackgroundProjMtx );
        pDrawParam->m_PrimitiveRenderer->SetViewMatrix( &m_BackgroundViewMtx );

        pDrawParam->m_PrimitiveRenderer->SetColor(m_BackgroundMultiplyColor);
        pDrawParam->m_PrimitiveRenderer->DrawScreenQuad(pDrawParam->m_CommandBuffer, m_BackgroundTextureDesSlot, m_BackgroundSamplerDesSlot );
    }

    // 背景画像の遅延開放
    {
        m_BackgroundTextures[0].DelayedRelease();
        m_BackgroundTextures[1].DelayedRelease();
    }

    pDrawParam->m_PrimitiveRenderer->SetBlendState( pDrawParam->m_CommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Opacity );
    pDrawParam->m_PrimitiveRenderer->SetDepthStencilState( pDrawParam->m_CommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthWriteTest );

    // グリッド処理
    if ( DoGridDraw() )
    {
        vfxdemo::DefaultGrid          defaultGrid;
        vfxdemo::CubeGrid             cubeGrid;

        switch( GetGridType() )
        {
            case 0: defaultGrid.ProcDraw( *pDrawParam, m_GridSize, m_GridDivCount ); break;
            case 2: cubeGrid.ProcDraw( *pDrawParam, m_GridSize, m_GridDivCount ); break;
        }
    }

    // モデル表示用ラスタライザステートを設定
    pDrawParam->m_CommandBuffer->SetRasterizerState( &m_ModelRasterizerState );

    // モデル描画
    ProcModelDraw( pDrawParam->m_CommandBuffer );
}


//------------------------------------------------------------------------------
//  モデル描画処理
//------------------------------------------------------------------------------
void Viewer::ProcModelDraw( nn::gfx::CommandBuffer* pCommnadBuffer ) NN_NOEXCEPT
{
    nn::vfx::viewer::detail::Preview* preview = g_pEffectViewer->GetPreviewHead();
    while ( preview )
    {
        if ( preview->GetPreviewType() == nn::vfx::viewer::detail::Preview::VfxPreviewType_ModelPreview )
        {
            preview->Draw( pCommnadBuffer );
        }
        preview = preview->GetNextPreview();
    }
}


//------------------------------------------------------------------------------
//  描画グループと描画パスを指定してエミッタセットを描画する
//------------------------------------------------------------------------------
void Viewer::DrawEmitterSet( nn::gfx::CommandBuffer* pCommnadBuffer, int groupId, uint32_t drawPathFlag ) NN_NOEXCEPT
{
    int processingIndex = 0;

    if ( g_pEffectSystem->IsHasRenderingEmitter( groupId, drawPathFlag ) )
    {
        g_pEffectSystem->Draw( processingIndex, pCommnadBuffer, groupId, drawPathFlag, true, true, nullptr );
    }
}

//------------------------------------------------------------------------------
//  エフェクト描画処理
//------------------------------------------------------------------------------
void Viewer::ProcVfxDraw( const vfxdemo::DrawParam* pDrawParam, GpuStatistics* gpuStatistics, ViewerLaunchArg::TextureOriginMode textureOriginMode ) NN_NOEXCEPT
{
    bool yFlip = false;
    if ( textureOriginMode == ViewerLaunchArg::TextureOriginMode_Lower_Left )
    {
        yFlip = true;
    }

    // パーティクル重なりチェックの設定
    if ( DoOverDrawMode() )
    {
        g_pEffectSystem->SetDrawPathRenderStateSetCallback( nn::vfx::DrawPathFlag_All, detail::VisualizeParticleFillHotSpot::DrawPathRenderStateSetCallback );
    }
    else
    {
        g_pEffectSystem->SetDrawPathRenderStateSetCallback( nn::vfx::DrawPathFlag_All, nullptr );
    }

    // エミッタ単位 GPU処理負荷計測
    if ( DoGpuProfileDraw() )
    {
        g_pEffectSystem->SetEmitterDrawProfilerCallback( DrawEmitterProfileCallback );
    }
    else
    {
        g_pEffectSystem->SetEmitterDrawProfilerCallback( nullptr );
    }

    nn::vfx::ViewParam view;
    view.Set( pDrawParam->m_ViewMtx, pDrawParam->m_ProjMtx, pDrawParam->m_CameraPos, pDrawParam->m_NearZ, pDrawParam->m_FarZ, pDrawParam->m_BaseFovy, pDrawParam->m_CurrentFovy );

    int processingIndex = 0;
    NN_PERF_SET_COLOR_GPU( nn::util::Color4u8::Red() );
    {
        g_pEffectSystem->SwapBuffer();
        g_pEffectSystem->SetViewParam( processingIndex, pDrawParam->m_CommandBuffer, &view );

        NN_PERF_BEGIN_MEASURE_NAME_GPU(pDrawParam->m_CommandBuffer, "Compute");
        g_pEffectSystem->BatchCalculationComputeShaderEmitter( pDrawParam->m_CommandBuffer, NULL );
        NN_PERF_END_MEASURE_GPU(pDrawParam->m_CommandBuffer);

        // カラーバッファをコピー
        m_GlareEffect.CopyTexture( pDrawParam->m_CommandBuffer, pDrawParam->m_PrimitiveRenderer, pDrawParam->m_ColorCopyTargetView, pDrawParam->m_ColorDescSlot, yFlip );

        // 深度ステンシルバッファをコピー
        m_GlareEffect.CopyTexture( pDrawParam->m_CommandBuffer, pDrawParam->m_PrimitiveRenderer, pDrawParam->m_DepthStencilCopyTargetView, pDrawParam->m_DepthStencilDescSlot, yFlip );

        pDrawParam->m_CommandBuffer->FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
        pDrawParam->m_CommandBuffer->FlushMemory( nn::gfx::GpuAccess_DepthStencil );

        // フレームバッファ/デプスバッファをvfxに設定
        g_pEffectSystem->SetFrameBufferTexture( 0, pDrawParam->m_ColorCopyDescSlot );
        g_pEffectSystem->SetDepthBufferTexture( 0, pDrawParam->m_DepthStencilCopyDescSlot );

        // レンダーターゲットを戻す
        nn::gfx::ColorTargetView* pTarget = pDrawParam->m_ColorTargetView;
        pDrawParam->m_CommandBuffer->SetRenderTargets( 1, &pTarget, pDrawParam->m_DepthStencilView );

        // エフェクト描画 後半
        NN_PERF_BEGIN_MEASURE_NAME_GPU( pDrawParam->m_CommandBuffer, "VfxGpu" );

        DrawEmitterSet( pDrawParam->m_CommandBuffer, 0, static_cast<uint32_t>( nn::vfx::DrawPathFlag_All ) );     // EffectBinaryPlayer用

        if ( gpuStatistics )
        {
            gpuStatistics->BeginQuery( pDrawParam->m_CommandBuffer );
            DrawEmitterSet( pDrawParam->m_CommandBuffer, nn::vfx::SystemParameters_ViewerGroupId, static_cast<uint32_t>( nn::vfx::DrawPathFlag_All ) );
            gpuStatistics->EndQuery( pDrawParam->m_CommandBuffer );
        }
        else
        {
            DrawEmitterSet( pDrawParam->m_CommandBuffer, nn::vfx::SystemParameters_ViewerGroupId, static_cast<uint32_t>( nn::vfx::DrawPathFlag_All ) );
        }

        NN_PERF_END_MEASURE_GPU( pDrawParam->m_CommandBuffer );
    }
}


//---------------------------------------------------------------------------
//! @brief                   背景画像の乗算カラーを設定します。
//---------------------------------------------------------------------------
void Viewer::SetBackgroundMultiplyColor( const nn::util::Vector4fType& backgroundMultiplyColor ) NN_NOEXCEPT
{
    nn::util::Vector4fType v;
    nn::util::VectorMultiply( &v, backgroundMultiplyColor, 255.0f );
    nn::util::VectorStore( &m_BackgroundMultiplyColor, v );
}

//---------------------------------------------------------------------------
//! @brief                   背景画像を初期化します.
//---------------------------------------------------------------------------
void Viewer::InitializeBackgroundTexture( nn::gfx::Device* pDevice, nn::vfx::Heap *pHeap , void* pResource, size_t resourceSize, size_t alignSize ) NN_NOEXCEPT
{
    BackgroundTexture *pBgTex = GetCurrentBackgroundTexture();
    pBgTex->Initialize( pDevice, pHeap, pResource, resourceSize, alignSize );
}

//---------------------------------------------------------------------------
//! @brief                   背景画像を解放します.
//---------------------------------------------------------------------------
void Viewer::FinalizeBackgroundTexture( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    BackgroundTexture *pBgTex = GetCurrentBackgroundTexture();
    if ( pBgTex->m_Initialized )
    {
        pBgTex->ReserveFinalize( pDevice, pHeap );

        m_BackgroundTexRepresentedId = m_BackgroundTexRepresentedId == 0 ? 1 : 0;
    }
}

//---------------------------------------------------------------------------
//      カメラの姿勢を設定します。
//---------------------------------------------------------------------------
void Viewer::SetCameraSettings(
    float posX, float posY, float posZ,
    float lookAtX, float lookAtY, float lookAtZ,
    float rotation,
    int projType,
    float aspect,
    float fov,
    float height,
    float nearZ, float farZ ) NN_NOEXCEPT
{
    NN_UNUSED(aspect);

    if ( m_pCamera == nullptr || m_pProjMatrix == nullptr ) return;

    nn::util::Vector3fType zaxis = NN_UTIL_VECTOR_3F_INITIALIZER( lookAtX - posX, lookAtY - posY, lookAtZ - posZ );
    if ( nn::util::VectorLength( zaxis ) == 0.0f ) return;

    m_Fovy = fov;

    m_pCamera->SetPos( posX, posY, posZ );
    m_pCamera->SetLookAtPos( lookAtX, lookAtY, lookAtZ );

    // アップベクトルを計算
    {
        nn::util::Vector3fType lookUp;

        if ( nn::util::VectorGetX( zaxis ) * nn::util::VectorGetX( zaxis ) + nn::util::VectorGetZ( zaxis ) * nn::util::VectorGetZ( zaxis ) != 0.0f )
        {
            nn::util::VectorSet( &lookUp, 0.0f, 1.0f, 0.0f );
        }
        else
        {
            nn::util::VectorSet( &lookUp, 0.0f, 0.0f, 1.0f );
        }

        nn::util::Matrix4x3fType rotMtx;
        nn::util::MatrixIdentity( &rotMtx );
        nn::util::Vector3fType rotRad;
        nn::util::VectorSet(&rotRad, 0.0f, 0.0f, rotation);
        nn::util::MatrixSetRotateXyz( &rotMtx, rotRad );

        nn::util::Vector3fType rotLookUp;
        nn::util::VectorTransform( &rotLookUp, lookUp, rotMtx );

        m_pCamera->GetMatrix();
        m_pCamera->SetUpVec( rotLookUp );
        m_pCamera->SetTwist( rotation );
    }

    if ( nearZ <= farZ * 0.000001f )
    {
        nearZ = farZ * 0.000001f;
    }

    SetNearZ( nearZ );
    SetFarZ( farZ );
    m_OrthHeight = height;
    m_ProjectionType = projType;

}

//---------------------------------------------------------------------------
//      プロジェクションマトリックスを更新します。
//---------------------------------------------------------------------------
void Viewer::UpdateProjection( float aspect ) NN_NOEXCEPT
{
    if ( m_ProjectionType == 0 )
    {
        // パースペクティブ
        nn::util::MatrixPerspectiveFieldOfViewRightHanded( m_pProjMatrix, m_Fovy, aspect, m_NearZ, m_FarZ );
    }
    else
    {
        // オルソ
        float h = m_OrthHeight / 2.0f;
        float w = h * aspect;
        nn::util::MatrixOrthographicOffCenterRightHanded( m_pProjMatrix, -w, w, -h, h, m_NearZ, m_FarZ );
    }
}

#if NN_VFX_VIEWER_USE_FONT
//---------------------------------------------------------------------------
//  デバッグ表示
//---------------------------------------------------------------------------
void Viewer::DebugDraw( nn::gfx::util::DebugFontTextWriter* pTextWriter, nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, GpuStatistics* pGpuStatistics ) NN_NOEXCEPT
{
    pTextWriter->SetCursor( 8, 8 );

    float viewerEndTime = 0.0f;
    if ( g_pEffectViewer->IsLoop() )
    {
        viewerEndTime = g_pEffectViewer->GetEndTime();
    }
    DrawViewerInfo( pCommandBuffer, pTextWriter, pPrimitiveRenderer, g_pEffectSystem, 8, pTextWriter->GetCursorY(), g_MemSizeForDynamicBuffer, g_pEffectViewer->GetTime(), viewerEndTime );

    // 画面右端に時間を表示
    pTextWriter->SetCursor( m_Width - 150.f, m_Height - 100.f );
    pTextWriter->SetCursor( 8, 300 );

    if ( DoGpuProfileDraw() )
    {
        pGpuStatistics->DrawDebugInfo( pCommandBuffer, pTextWriter, pPrimitiveRenderer, 220, 8 );
    }
}

//---------------------------------------------------------------------------
//  指定したグループのエミッタセット～エミッタの内訳を描画
//---------------------------------------------------------------------------
void Viewer::DebugDrawEmitterDetail( nn::gfx::util::DebugFontTextWriter* pTextWriter, int groupId ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pTextWriter );

    //---------------------------------------------------------------------------
    //  こんな感じで表示されます。
    //  ※ vfx::System に登録された順番に表示されるので、
    //     EffectMaker での登録順とは必ずしも一致しないことに注意。
    //---------------------------------------------------------------------------
    //
    //  [ESet]:    ESetExample
    //  +-[CPU]:   EmitterA     :   20[p]
    //    +-[GPU]: ChildEmitter1:   40[p],   1[e]
    //    +-[SO ]: ChildEmitter2:   80[p],  40[e]
    //
    //  [ESet]:    ESetTest
    //  +-[CPU]:   EmitterB     :    5[p]
    //    +-[CPU]: ChildEmitter3:  100[p],  10[e]
    //  +-[GPU]:   EmitterC     :   60[p]
    //
    //---------------------------------------------------------------------------
    // [p]: パーティクル数
    // [e]: エミッタ数
    //---------------------------------------------------------------------------

    pTextWriter->Print( "\n" );

    // グループに存在する EmitterSet について走査
    const nn::vfx::EmitterSet* pEmitterSet = g_pEffectSystem->GetEmitterSetHead( groupId );
    while( pEmitterSet )
    {
        // エミッタセット名
        pTextWriter->Print( "[ESet]    %s\n", pEmitterSet->GetName() );

        // EmitterSet が持つ Emitter について走査
        const int emitterCount = pEmitterSet->GetEmitterCount();
        for( int idxEmitter = 0; idxEmitter < emitterCount; ++idxEmitter )
        {
            const nn::vfx::Emitter* pEmitter = pEmitterSet->GetEmitter( idxEmitter );
            if( pEmitter == NULL )
            {
                continue;
            }

            const char* emitterName   = pEmitter->GetName();
            const char* calcType      = nn::vfx::detail::ToString( pEmitter->GetCalculationType() );
            const int   particleCount = pEmitter->CalculatePreciseParticleCount();

            // エミッタ計算タイプ／エミッタ名／パーティクル数
            pTextWriter->Print( "+-[%-3s]   %-13s: %4d[p]\n", calcType, emitterName, particleCount );

            // 子エミッタについての処理（インスタンスごとにまとめる）
            for( int idxChildRes = 0; idxChildRes < nn::vfx::SystemParameters_MaxEmitterInclusionCount; ++idxChildRes )
            {
                const nn::vfx::Emitter* pChildEmitter = pEmitter->GetChildEmitterHead( idxChildRes );
                if( pChildEmitter == NULL )
                {
                    // これ以降チャイルドは無いのでここで抜ける
                    break;
                }

                const char* childEmitterName = pChildEmitter->GetName();
                const char* childCalcType    = nn::vfx::detail::ToString( pChildEmitter->GetCalculationType() );

                // チャイルドエミッタの数とチャイルドパーティクルの数を計測
                int numChildEmitter = 0;
                int numChildParticle = 0;
                while( pChildEmitter )
                {
                    if( pChildEmitter->IsAlive() )
                    {
                        // チャイルドエミッタ毎にパーティクル数を問い合わせて合計
                        numChildParticle += pChildEmitter->CalculatePreciseParticleCount();
                        // エミッタ数は生きているものをカウント
                        numChildEmitter++;
                    }
                    pChildEmitter = pChildEmitter->GetNextEmitter();
                }

                // （チャイルド）エミッタ計算タイプ／エミッタ名／パーティクル数／チャイルドエミッタ数
                pTextWriter->Print( "  +-[%-3s] %-13s: %4d[p],%4d[e]\n", childCalcType, childEmitterName, numChildParticle, numChildEmitter );
            }
        }

        pEmitterSet = pEmitterSet->GetNext();
        pTextWriter->Print( "\n" );
    }
}
#endif

//---------------------------------------------------------------------------
//      垂直同期の待つ回数を取得
//---------------------------------------------------------------------------
int Viewer::GetSwapInterval() NN_NOEXCEPT
{
    switch ( m_FrameRate )
    {
        case INIT_SETTINGS_FRAME_RATE_60:
            return PRESENT_INTERVAL_60FPS;
            break;
        case INIT_SETTINGS_FRAME_RATE_30:
            return PRESENT_INTERVAL_30FPS;
            break;
        case INIT_SETTINGS_FRAME_RATE_20:
            return PRESENT_INTERVAL_20FPS;
            break;
        case INIT_SETTINGS_FRAME_RATE_15:
            return PRESENT_INTERVAL_15FPS;
            break;
    }

    return PRESENT_INTERVAL_60FPS;
}

//---------------------------------------------------------------------------
//! @brief  背景画像を遅延開放します.
//---------------------------------------------------------------------------
void BackgroundTexture::DelayedRelease() NN_NOEXCEPT
{
    if ( m_ReleaseCount == 0 ) return;
    --m_ReleaseCount;
    if ( m_ReleaseCount > 0 ) return;

    if ( m_pTextureFile )
    {
        m_pTexture->Finalize( m_pDevice );
        m_pTexture = nullptr;
        m_pTextureFile->Finalize( m_pDevice );
        m_pTextureFile = nullptr;
        m_pDevice = nullptr;
    }
    if ( m_pBuffer )
    {
        m_pHeap->Free( m_pBuffer );
        m_pBuffer = nullptr;
        m_pHeap = nullptr;
    }
}

//---------------------------------------------------------------------------
//! @brief  背景画像を初期化します.
//---------------------------------------------------------------------------
void BackgroundTexture::Initialize( nn::gfx::Device* pDevice, nn::vfx::Heap *pHeap , void* pResource, size_t resourceSize, size_t alignSize ) NN_NOEXCEPT
{
    if ( m_Initialized || pResource == nullptr || m_ReleaseCount != 0 ) return;

    // アライン済みのバッファを確保
    m_pBuffer = pHeap->Alloc( resourceSize, alignSize );
    NN_SDK_ASSERT_NOT_NULL( m_pBuffer );

    // コピー
    memcpy( m_pBuffer, pResource, resourceSize );

    // テクスチャリソースとしてキャスト＆初期化
    m_pTextureFile = nn::gfx::ResTextureFile::ResCast( m_pBuffer );
    m_pTextureFile->Initialize( pDevice );
    m_pTexture = m_pTextureFile->GetResTexture( 0 );
    m_pTexture->Initialize( pDevice );

    m_Initialized = true;
}

//---------------------------------------------------------------------------
//! @brief  背景画像の解放を指示します.
//---------------------------------------------------------------------------
void BackgroundTexture::ReserveFinalize( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    if ( m_Initialized == false ) return;

    m_pDevice = pDevice;
    m_pHeap = pHeap;
    m_ReleaseCount = 10;
    m_Initialized = false;
}

//---------------------------------------------------------------------------
//! @brief     TBD
//---------------------------------------------------------------------------
void Viewer::DebugEmitterShapeDraw( nn::gfx::CommandBuffer*                pCommandBuffer,
                                    nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                    const nn::util::Matrix4x4fType*        pProjectionMatrix,
                                    const nn::util::Matrix4x3fType*        pViewMatrix ) NN_NOEXCEPT
{
    pRenderer->SetDefaultParameters();
    pRenderer->SetProjectionMatrix( pProjectionMatrix );
    pRenderer->SetViewMatrix( pViewMatrix );
    pRenderer->SetDepthStencilState( pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWrite );
    pRenderer->SetBlendState( pCommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Normal );
    const nn::gfx::RasterizerState* pRasterizerState = pRenderer->GetRasterizerState( nn::gfx::PrimitiveTopologyType_Triangle,
                                                                                      nn::gfx::CullMode_None,
                                                                                      nn::gfx::FillMode_Solid );
    pCommandBuffer->SetRasterizerState( pRasterizerState );

    for ( int i = 0; i < nn::vfx::SystemParameters_MaxGroupCount; i++ )
    {
        nn::vfx::EmitterSet* pEmitterSet = g_pEffectSystem->GetEmitterSetHead( i );

        while ( pEmitterSet )
        {
            // エミッタセットが非表示のとき表示しない
            if ( pEmitterSet->GetEmitterSetResource()->isVisible == false || pEmitterSet->IsVisible() == false )
            {
                pEmitterSet = pEmitterSet->GetNext();
                continue;
            }

            for ( int j = 0 ; j < pEmitterSet->GetEmitterCount(); j++ )
            {
                const nn::vfx::Emitter* pEmitter = pEmitterSet->GetEmitter( j );
                if ( pEmitter )
                {
                    // エミッタが非表示のとき表示しない
                    if ( pEmitter->GetEmitterResource()->m_IsVisible == false || pEmitter->IsAlive() == false )
                    {
                        continue;
                    }

                    // 親エミッタ形状の描画
                    if ( DoDebugEmitterShapeDraw() || DoDebugShapeFieldDraw() )
                    {
                        g_DrawShapes.DrawMain(pCommandBuffer, pRenderer, pEmitter, *m_pCamera->GetPos(), DoDebugEmitterShapeDraw(), DoDebugShapeFieldDraw());
                    }

                    // 子エミッタの処理
                    if ( DoDebugEmitterShapeChildDraw() || DoDebugShapeFieldDraw() )
                    {
                        for ( int k = 0; k < pEmitter->GetEmitterResource()->m_ChildEmitterResCount; k++ )
                        {
                            const nn::vfx::Emitter* childEmitter = pEmitter->GetChildEmitterHead( k );
                            while ( childEmitter )
                            {
                                // 子エミッタが非表示のとき表示しない
                                if ( childEmitter->GetEmitterResource()->m_IsVisible == false || childEmitter->IsAlive() == false )
                                {
                                    childEmitter = childEmitter->GetNextEmitter();
                                    continue;
                                }
                                g_DrawShapes.DrawMain(  pCommandBuffer, pRenderer, childEmitter, *m_pCamera->GetPos(), DoDebugEmitterShapeChildDraw(), DoDebugShapeFieldDraw() );
                                childEmitter = childEmitter->GetNextEmitter();
                            }
                        }
                    }
                }
            }
            pEmitterSet = pEmitterSet->GetNext();
        }
    }
}

//---------------------------------------------------------------------------
//  キャプチャ・フレーム・ファイルパスを設定する。
//---------------------------------------------------------------------------
void Viewer::SetCaptureFilePath( const char* captureFilePath ) NN_NOEXCEPT
{
    NN_UNUSED( captureFilePath );
#if defined( NN_BUILD_CONFIG_OS_WIN )
    memset( m_CaptureFilePath, 0, _MAX_PATH );
    if ( captureFilePath != nullptr )
    {
        strcpy_s( m_CaptureFilePath, _MAX_PATH, captureFilePath );
    }
#endif
}
