﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <G3dModelPreview.h>
#include <nn/vfx/viewer/vfx_Viewer.h>

// 使用するデバイス・アロケータ
nn::gfx::Device* G3dModelPreview::mpDevice = NULL;
vfxdemo::TextureDescriptorIndexAllocator* G3dModelPreview::mpTextureDescriptorPoolAllocator = NULL;
vfxdemo::SamplerDescriptorIndexAllocator* G3dModelPreview::mpSamplerDescriptorPoolAllocator = NULL;

//---------------------------------------------------------------------------
//  EffectMakerへのモデル情報転送クラス
//---------------------------------------------------------------------------
class ModelEnumerator : public nn::vfx::viewer::ModelEnumerator
{
    enum
    {
        ModelMaxCount = 1
    };
public:
    ModelEnumerator() NN_NOEXCEPT : m_ModelCount(0) {}

    virtual ~ModelEnumerator() NN_NOEXCEPT {}

public:
    // モデル名からインデックスを取得する
    int GetIndex(const char* modelName) const NN_NOEXCEPT
    {
        for ( int i = 0; i < m_ModelCount; i++ )
        {
            if (strcmp(m_Model[i]->GetModelName(), modelName) == 0)
            {
                return i;
            }
        }
        return -1;
    }

    // モデル数を取得する
    int GetModelCount() const NN_NOEXCEPT
    {
        return m_ModelCount;
    }

    // モデル数を取得する
    const char* GetModelName(int index) const NN_NOEXCEPT
    {
        if ( (index < 0 ) || (index > GetModelCount() ) ) return NULL;
        return m_Model[index]->GetModelName();
    }

    // ボーン名を取得する
    int GetBoneCount(const char* modelName) const NN_NOEXCEPT
    {
        int index = GetIndex(modelName);
        if ( (index < 0 ) || ( index > GetModelCount() ) ) return 0;
        return m_Model[index]->GetBoneNum();
    }

    // ボーン名を取得する
    virtual const char* GetBoneName(const char* modelName, int boneIndex) const NN_NOEXCEPT
    {
        int index = GetIndex(modelName);
        if ( (index < 0 ) || ( index > GetModelCount() ) ) return NULL;
        return m_Model[index]->GetBoneName( boneIndex );
    }

    // ボーンマトリクスを取得する
    virtual void GetBoneMatrix(nn::util::Matrix4x3fType* pOutBoneMatrix, const char* modelName, int boneIndex) const NN_NOEXCEPT
    {
        int index = GetIndex(modelName);
        if ( (index < 0 ) || ( index > GetModelCount() ) ) return;
        *pOutBoneMatrix = *m_Model[index]->GetBoneWorldMatrix(boneIndex);
    }

    // モデルの表示位置を取得する
    void GetModelRootMatrix(nn::util::Matrix4x3fType* pOutRootMatrix, const char* modelName) const NN_NOEXCEPT
    {
        NN_UNUSED(modelName);
        nn::util::MatrixIdentity( pOutRootMatrix );
    }

    // モデルオブジェクトをセットする
    int SetModel( nw::eftdemo::SimpleModel* model ) NN_NOEXCEPT
    {
        int modelIndex = 0;
        if ( m_ModelCount < ModelMaxCount)
        {
            modelIndex = m_ModelCount;
            m_Model[m_ModelCount++] = model;
        }
        return modelIndex;
    }

    // モデル中のボーン名のIDを探す
    int SearchBoneId( int modelIndex, const char* boneName ) NN_NOEXCEPT
    {
        for ( int i = 0; i < m_Model[modelIndex]->GetBoneNum(); i++ )
        {
            if ( strcmp( m_Model[modelIndex]->GetBoneName( i ), boneName ) == 0 )
            {
                return i;
            }
        }
        return 0;
    }

private:
    nw::eftdemo::SimpleModel*  m_Model[ModelMaxCount];
    int                        m_ModelCount;
};

//---------------------------------------------------------------------------
//  EftVwrModelクラスのインスタンスを生成します。
//---------------------------------------------------------------------------
nn::vfx::viewer::detail::Preview* G3dModelPreview::CreateModelPreview( nn::vfx::Heap* pHeap, nn::vfx::viewer::ViewerSystem* pViewerSystem, void* pBinaryData, size_t binarySize, nn::vfx::viewer::detail::Guid guid ) NN_NOEXCEPT
{
    // モデルプレビュークラス生成
    void* buffer = pHeap->Alloc( sizeof( G3dModelPreview ) );
    if ( !buffer ) { return NULL; }
    G3dModelPreview* modelPreview = new (buffer) G3dModelPreview();

    // プレビュー初期化
    modelPreview->Initialize( pHeap, Preview::VfxPreviewType_ModelPreview, guid );
    if ( !modelPreview->InitializeModelPreview( pHeap, pBinaryData, binarySize ) )
    {
        pHeap->Free( modelPreview );
        return NULL;
    }

    // 破棄コールバックをセット
    modelPreview->SetDestroyPreviewCallback( G3dModelPreview::DestroyModelPreview );

    // ボーン情報バッファを生成します
    modelPreview->mpAllocator = pHeap;
    nn::util::Matrix4x3fType identity;
    nn::util::MatrixIdentity(&identity);
    modelPreview->Calculate( false, 1.0f, identity, identity );

    // モデル内の情報を送信
    ModelEnumerator modelInfo;
    modelInfo.SetModel( modelPreview->GetModelHandle() );
    nn::vfx::viewer::detail::Guid modeGuid = modelPreview->GetGuid();
    pViewerSystem->AddModelInfoInternal( &modelInfo, modelInfo.GetModelName(0), &modeGuid );

    return modelPreview;
}

//---------------------------------------------------------------------------
//  EftVwrModelクラスのインスタンスを破棄します。
//---------------------------------------------------------------------------
void G3dModelPreview::DestroyModelPreview( nn::vfx::Heap* pHeap, nn::vfx::viewer::detail::Preview* pPreview ) NN_NOEXCEPT
{
    G3dModelPreview* modelPreview = reinterpret_cast<G3dModelPreview*>( pPreview );

    // モデルの終了処理
    modelPreview->FinalizeModelPreview( pHeap );

    // プレビューの終了処理を行います。
    modelPreview->Finalize();

    // メモリを解放します。
    pHeap->Free( modelPreview );
}

//---------------------------------------------------------------------------
//  モデルプレビューで使用するデバイスを登録します。
//---------------------------------------------------------------------------
void G3dModelPreview::SetDevice( nn::gfx::Device* pDevice ) NN_NOEXCEPT
{
    mpDevice = pDevice;
}

//---------------------------------------------------------------------------
//  モデルプレビューに使用するでディスクリプタアロケータを登録します。
//---------------------------------------------------------------------------
void G3dModelPreview::SetTextureDescriptorPoolAllocator( vfxdemo::TextureDescriptorIndexAllocator* pAllocator ) NN_NOEXCEPT
{
    mpTextureDescriptorPoolAllocator = pAllocator;
}

//---------------------------------------------------------------------------
//  モデルプレビューに使用するでディスクリプタアロケータを登録します。
//---------------------------------------------------------------------------
void G3dModelPreview::SetSamplerDescriptorPoolAllocator( vfxdemo::SamplerDescriptorIndexAllocator* pAllocator ) NN_NOEXCEPT
{
    mpSamplerDescriptorPoolAllocator = pAllocator;
}

//---------------------------------------------------------------------------
//  モデルプレビューに使用するディスクリプタアロケータを取得します。
//---------------------------------------------------------------------------
vfxdemo::TextureDescriptorIndexAllocator* G3dModelPreview::GetTextureDescriptorPoolAllocator() NN_NOEXCEPT
{
    return mpTextureDescriptorPoolAllocator;
}

//---------------------------------------------------------------------------
//  モデルプレビューに使用するでディスクリプタアロケータを取得します。
//---------------------------------------------------------------------------
vfxdemo::SamplerDescriptorIndexAllocator* G3dModelPreview::GetSamplerDescriptorPoolAllocator() NN_NOEXCEPT
{
    return mpSamplerDescriptorPoolAllocator;
}

//---------------------------------------------------------------------------
//  モデルプレビューに使用するディスクリプタプールを取得します。
//---------------------------------------------------------------------------
nn::gfx::DescriptorPool* G3dModelPreview::GetTextureDescriptorPool() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(mpTextureDescriptorPoolAllocator);
    return mpTextureDescriptorPoolAllocator->GetDescriptorPool();
}

//---------------------------------------------------------------------------
//  モデルプレビューに使用するでディスクリプタプールを取得します。
//---------------------------------------------------------------------------
nn::gfx::DescriptorPool* G3dModelPreview::GetSamplerDescriptorPool() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(mpSamplerDescriptorPoolAllocator);
    return mpSamplerDescriptorPoolAllocator->GetDescriptorPool();
}

//---------------------------------------------------------------------------
//  モデルプレビュークラスの初期化します。
//---------------------------------------------------------------------------
bool G3dModelPreview::InitializeModelPreview( nn::vfx::Heap* pHeap, void* pBinary, size_t binarySize ) NN_NOEXCEPT
{
    // モデルバイナリを確保する
    mModelBinary = pHeap->Alloc( binarySize, 64 * 1024 );
    if ( !mModelBinary ) return false;
    memcpy( mModelBinary, pBinary, binarySize );

    nn::os::FlushDataCache( mModelBinary, binarySize );

    // モデルの初期化
    mModel.Initialize( mModelBinary, binarySize, pHeap, mpDevice );
    return true;
}

//---------------------------------------------------------------------------
//  モデルプレビュークラスの終了処理をします。
//---------------------------------------------------------------------------
void G3dModelPreview::FinalizeModelPreview( nn::vfx::Heap* pHeap ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( mModelBinary );
    mModel.Finalize( mpDevice, pHeap );
    pHeap->Free( mModelBinary );
}

//---------------------------------------------------------------------------
//  コンストラクタです。
//---------------------------------------------------------------------------
G3dModelPreview::G3dModelPreview() NN_NOEXCEPT
{
    mModelBinary            = NULL;
    mpAllocator             = NULL;
    mAnimFrame              = 0.0f;
}

//---------------------------------------------------------------------------
//  プレビュー計算処理です。
//---------------------------------------------------------------------------
bool G3dModelPreview::Calculate( bool bPause, float frameRate, const nn::util::Matrix4x3fType& centerMatrix, const nn::util::Matrix4x3fType& viewMatrix ) NN_NOEXCEPT
{
    // 規定クラスの計算処理
    bool ret = Preview::Calculate( bPause, frameRate, centerMatrix, viewMatrix );
    if ( !ret )
    {
        return ret;
    }

    if ( !bPause )
    {
        mAnimFrame += frameRate;
    }

    mModel.SetAnimationFrame( mAnimFrame );
    mModel.SetAnimationIndex( 0 );
    mModel.SetDrawMatrix(GetDrawMatrix());
    mModel.Calc();

    return true;
}

//------------------------------------------------------------------------------
//  モデルブロック計算処理
//------------------------------------------------------------------------------
void G3dModelPreview::CalcBlock( const nn::util::Matrix4x4fType* projMatrix,
                                 const nn::util::Matrix4x3fType* viewMatrix,
                                 const nn::util::Vector3fType* lightPos )
{
    mModel.CalcBlock( projMatrix, viewMatrix, lightPos );
}

//------------------------------------------------------------------------------
//  プレビュー描画処理
//------------------------------------------------------------------------------
void G3dModelPreview::Draw( nn::gfx::CommandBuffer* pCommandBuffer ) NN_NOEXCEPT
{
    if ( IsVisible() && IsAlive() )
    {
        mModel.Draw( pCommandBuffer, mpAllocator );
    }
}

//---------------------------------------------------------------------------
//  プレビューをリセットします。
//---------------------------------------------------------------------------
void G3dModelPreview::ResetPreview( bool bFade ) NN_NOEXCEPT
{
    NN_UNUSED( bFade );
    mAnimFrame = 0.0f;
}

//---------------------------------------------------------------------------
//  マトリクスを取得します。
//---------------------------------------------------------------------------
void G3dModelPreview::GetMatrix( nn::util::Matrix4x3fType* matrix, int idx ) NN_NOEXCEPT
{
    const nn::util::Matrix4x3fType* world = mModel.GetBoneWorldMatrix( idx );
    if (world)
    {
        *matrix = *world;
    }
}
