﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "Config.h"

#include <nn/vfx/vfx_Data.h>
#include <nn/vfx/vfx_Emitter.h>
#include <nn/vfx/vfx_Heap.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

namespace detail {

//------------------------------------------------------------------------------
//  形状描画用
//------------------------------------------------------------------------------
enum Shapes
{
    Shapes_TriangleVertexCount = 3,
    Shapes_NumLines   = 18,
    Shapes_NumShapes  = 12,
    Shapes_MaxMeshBuf = 64
};

enum ShapeBufferType
{
    ShapeBufferType_Line = 0,
    ShapeBufferType_Triangle,
    ShapeBufferType_Quad
};

//------------------------------------------------------------------------------
//  形状描画クラス
//------------------------------------------------------------------------------
class DrawShapes
{
public:
    //------------------------------------------------------------------------------
    //  コンストラクタ
    //------------------------------------------------------------------------------
    DrawShapes() NN_NOEXCEPT
    {
        m_Color.v[0] = 0x80;
        m_Color.v[1] = 0x80;
        m_Color.v[2] = 0x80;
        m_Color.v[3] = 0x50;
    }

    //------------------------------------------------------------------------------
    //  デストラクタ
    //------------------------------------------------------------------------------
    ~DrawShapes() NN_NOEXCEPT {};

    //------------------------------------------------------------------------------
    //  形状描画メイン処理
    //------------------------------------------------------------------------------
    void DrawMain( nn::gfx::CommandBuffer* pCommandBuffer,
                   nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                   const nn::vfx::Emitter* pEmitter,
                   const nn::util::Vector3fType& cameraPos,
                   bool isShapes,
                   bool isField ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  円形状描画処理
    //---------------------------------------------------------------------------
    void DrawCircle( nn::gfx::CommandBuffer* pCommandBuffer,
                     nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                     const nn::vfx::Emitter* pEmitter,
                     const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  円ポリゴン形状描画処理
    //---------------------------------------------------------------------------
    void DrawCircleFill( nn::gfx::CommandBuffer* pCommandBuffer,
                         nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                         const nn::vfx::Emitter* pEmitter,
                         const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  球形状描画処理を行います。
    //---------------------------------------------------------------------------
    void DrawSphere( nn::gfx::CommandBuffer* pCommandBuffer,
                     nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                     const nn::vfx::Emitter* pEmitter,
                     const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  シリンダー形状描画処理
    //---------------------------------------------------------------------------
    void DrawCylinder( nn::gfx::CommandBuffer* pCommandBuffer,
                       nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                       const nn::vfx::Emitter* pEmitter,
                       const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  キューブ形状描画処理
    //---------------------------------------------------------------------------
    void DrawCube( nn::gfx::CommandBuffer* pCommandBuffer,
                   nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                   const nn::vfx::Emitter* pEmitter,
                   const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  ライン形状描画処理
    //---------------------------------------------------------------------------
    void DrawLine( nn::gfx::CommandBuffer* pCommandBuffer,
                   nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                   const nn::vfx::Emitter* pEmitter,
                   const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  キューブ形状描画処理を行います。
    //---------------------------------------------------------------------------
    void DrawRectangle( nn::gfx::CommandBuffer* pCommandBuffer,
                        nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                        const nn::vfx::Emitter* pEmitter,
                        const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

   //---------------------------------------------------------------------------
    //  プリミティブ形状描画処理
    //---------------------------------------------------------------------------
    void DrawPrimitive( nn::gfx::CommandBuffer* pCommandBuffer,
                        nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                        const nn::vfx::Emitter* pEmitter,
                        const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  フィールドの収束点描画処理
    //---------------------------------------------------------------------------
    void DrawFieldConvergencePosition( nn::gfx::CommandBuffer* pCommandBuffer,
                                       nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                       const nn::vfx::Emitter* pEmitter,
                                       const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  フィールドの磁力描画処理
    //---------------------------------------------------------------------------
    void DrawFieldMagnetPosition( nn::gfx::CommandBuffer* pCommandBuffer,
                                  nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                  const nn::vfx::Emitter* pEmitter,
                                  const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  フィールドのスピン形状描画処理
    //---------------------------------------------------------------------------
    void DrawFieldSpin( nn::gfx::CommandBuffer* pCommandBuffer,
                        nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                        const nn::vfx::Emitter* pEmitter,
                        const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  フィールドのコリジョン面描画処理
    //---------------------------------------------------------------------------
    void DrawCollision( nn::gfx::CommandBuffer* pCommandBuffer,
                        nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                        const nn::vfx::Emitter* pEmitter,
                        const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  範囲内ループ描画処理
    //---------------------------------------------------------------------------
    void DrawAreaLoop( nn::gfx::CommandBuffer* pCommandBuffer,
                       nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                       const nn::vfx::Emitter* pEmitter,
                       const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT;

private:
    //------------------------------------------------------------------------------
    //  注視点を描画
    //------------------------------------------------------------------------------
    void DrawAxis( nn::gfx::CommandBuffer* pCommandBuffer,
                   nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                   const nn::util::Matrix4x3fType& esetSrt,
                   const nn::util::Vector3fType& cameraPos ) NN_NOEXCEPT;

    //------------------------------------------------------------------------------
    //   最小角度幅、開始角度の計算。
    //------------------------------------------------------------------------------
    void CalculateVolumeSweepParams( const nn::vfx::Emitter* pEmitter,
                                                 const nn::vfx::detail::ResEmitterVolume* pEmitterVolume,
                                                 float* minLatitude,
                                                 float* sweepStart,
                                                 int numFace ) NN_NOEXCEPT;

    //------------------------------------------------------------------------------
    //  位置の計算。
    //------------------------------------------------------------------------------
    void CalculatePos( nn::util::Float3* pos, float rot,
                                   nn::util::Vector3fType& scale ) NN_NOEXCEPT;

    void CalculatePos( nn::util::Vector3fType* pos, float rot,
                                   nn::util::Vector3fType& scale ) NN_NOEXCEPT;

    //------------------------------------------------------------------------------
    //  球を緯度の軸に合うように回転します。
    //------------------------------------------------------------------------------
    void RotateDirection( const nn::vfx::detail::ResEmitterVolume* pEmitterVolume,
                                      nn::util::Vector3fType* pOutDir ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  空洞率のスケールの計算
    //---------------------------------------------------------------------------
    void CalculateHoleScale( const nn::vfx::Emitter* pEmitter,
                             nn::util::Matrix4x3fType *esetSrt,
                             bool isYscaleFix ) NN_NOEXCEPT;

    //------------------------------------------------------------------------------
    //  球状の位置の計算。
    //------------------------------------------------------------------------------
    void PrimitiveSphere( const nn::vfx::Emitter* pEmitter,
                                      nn::util::Float3* pos, float minRot,
                                      float volumeSweepStart, int vLine, int hLine,
                                      nn::util::Vector3fType& scale ) NN_NOEXCEPT;
    //---------------------------------------------------------------------------
    //  スクリーンのサイズに変換
    //---------------------------------------------------------------------------
    float AjustScreenScaleSize( nn::util::Vector3fType& pos,
                                            const nn::util::Vector3fType& cameraPos) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //  頂点、インデックスバッファサイズの計算
    //---------------------------------------------------------------------------
    void CalculateVertexIndexBufferSize( int* numVertices,
                                                     int* numIndices, int numFace,
                                                     ShapeBufferType bufferType ) NN_NOEXCEPT;

    //------------------------------------------------------------------------------
    //  マトリックスを取得する。
    //------------------------------------------------------------------------------
    void GetMatrix( const nn::vfx::Emitter* pEmitter,
                                nn::util::Matrix4x3fType* matrixSrt) NN_NOEXCEPT;

private:
    nn::util::Uint8x4                        m_Color;
};

}
