﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "Config.h"

#if NN_VFX_VIEWER_USE_FONT
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#endif

#include <nns/nac/nac_MemoryAllocator.h>
#include <nns/nac/nac_Pad.h>
#include <Camera.h>
#include <DrawParam.h>
#include <GlareEffect.h>
#include <ViewerLaunchArg.h>
#include <CaptureFrame.h>
#include "detail/util_GpuStatistics.h"
#include "detail/util_VisualizeParticleFillHotSpot.h"


//---------------------------------------------------------------------------
//! @brief        フレームレート
//---------------------------------------------------------------------------
enum INIT_SETTINGS_FRAME_RATE
{
    INIT_SETTINGS_FRAME_RATE_60 = 0,
    INIT_SETTINGS_FRAME_RATE_30,
    INIT_SETTINGS_FRAME_RATE_20,
    INIT_SETTINGS_FRAME_RATE_15
};

//---------------------------------------------------------------------------
//! @brief        垂直同期数
//---------------------------------------------------------------------------
enum PRESENT_INTERVAL
{
    PRESENT_INTERVAL_60FPS = 1,
    PRESENT_INTERVAL_30FPS,
    PRESENT_INTERVAL_20FPS,
    PRESENT_INTERVAL_15FPS
};


struct BackgroundTexture
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    BackgroundTexture() NN_NOEXCEPT
        : m_pTexture( nullptr )
        , m_Initialized( false )
        , m_pTextureFile( nullptr )
        , m_pBuffer( nullptr )
        , m_ReleaseCount( 0 )
        , m_pDevice( nullptr )
        , m_pHeap( nullptr )
    {
    }

    //---------------------------------------------------------------------------
    //! @brief  背景画像を遅延開放します.
    //---------------------------------------------------------------------------
    void DelayedRelease() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  背景画像を初期化します.
    //---------------------------------------------------------------------------
    void Initialize( nn::gfx::Device* pDevice, nn::vfx::Heap *pHeap, void* pResource, size_t resourceSize, size_t alignSize ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  背景画像の解放を指示します.
    //---------------------------------------------------------------------------
    void ReserveFinalize( nn::gfx::Device *pDevice, nn::vfx::Heap *pHeap ) NN_NOEXCEPT;

    nn::gfx::ResTexture*        m_pTexture;     //!< 背景画像
    bool                        m_Initialized;  //!< 背景画像初期化済みフラグ

private:
    nn::gfx::ResTextureFile*    m_pTextureFile; //!< 背景画像ファイル
    void*                       m_pBuffer;      //!< 背景画像用バッファ
    int                         m_ReleaseCount; //!< 遅延開放カウンタ
    nn::gfx::Device*            m_pDevice;      //!< 遅延開放用GFXデバイス
    nn::vfx::Heap*              m_pHeap;        //!< 遅延開放用ヒープ
};

//--------------------------------------------------------------------------------------------
//! @brief      カメラ情報を表します。カメラ情報をEffectMakerへ送信するときに利用します。
//--------------------------------------------------------------------------------------------
struct CameraData
{
    NN_DISALLOW_COPY( CameraData );
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    CameraData() NN_NOEXCEPT
    {
    }

    nn::util::Vector3fType      position;   //!< カメラ位置
    nn::util::Vector3fType      target;     //!< ターゲット位置
};

class Viewer
{
public:
    Viewer() NN_NOEXCEPT
        : m_DoDebugDraw( true )
        , m_DoGpuProfileDraw( true )
        , m_DoDebugEmitterShapeDraw( false )
        , m_DoDebugEmitterShapeDrawChild(false)
        , m_DoDebugShapeDrawField(false)
        , m_DoGridDraw( true )
#if defined( NN_BUILD_CONFIG_OS_WIN )
        , m_DoEmulatingTest( false )
#endif
        , m_DoOverDrawMode( false )
        , m_PadState( true )
        , m_IsLinearMode( false )
        , m_PlaySpeed( 1.0f )
        , m_GridType( 0 )
        , m_pCamera( nullptr )
        , m_pProjMatrix( nullptr )
        , m_NearZ( 0.1f )
        , m_FarZ( 10000.0f )
        , m_OrthHeight( 0.0f )
        , m_ProjectionType( 0 )
        , m_BackgroundTexRepresentedId( 0 )
        , m_BackgroundProjMtx()
        , m_BackgroundViewMtx()
        , m_BackgroundSampler()
        , m_BackgroundMultiplyColor()
        , m_GlareBufferNum( 5 )
        , m_GridSize( 10.f )
        , m_GridDivCount( 1 )
    {
        for( int i = 0; i < 4; ++i )
        {
            m_BackgroundColor[ i ] = 0.2f;
        }
        nn::util::MatrixIdentity( &m_BackgroundProjMtx );
        nn::util::MatrixIdentity( &m_BackgroundViewMtx );
    }


    void Initialize( nn::gfx::Device* pDevice,
                     nns::nac::MemoryAllocator* pAllocator,
                     vfxdemo::TextureDescriptorIndexAllocator* pTextureDescriptorPoolAllocator,
                     vfxdemo::SamplerDescriptorIndexAllocator* pSamplerDescriptorPoolAllocator,
                     int width, int height,
                     nn::vfx::RegisterTextureViewSlot pRegisterTextureSlotCallback,
                     nn::vfx::RegisterSamplerSlot     pRegisterSamplerSlotCallback,
                     ViewerLaunchArg &launchArg ) NN_NOEXCEPT;
    void Finalize( nn::gfx::Device* pDevice,
                   nns::nac::MemoryAllocator* pAllocator,
                   nn::vfx::UnregisterTextureViewSlot pUnregisterTextureSlotCallback,
                   nn::vfx::UnregisterSamplerSlot pUnregisterSamplerSlotCallback ) NN_NOEXCEPT;
#if defined( NN_BUILD_CONFIG_OS_WIN )
    void ProcCalc( nns::nac::Mouse* pMouse, nns::nac::Pad* pPad, const vfxdemo::DrawParam* pDrawParam, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlot ) NN_NOEXCEPT;
#else
    void ProcCalc( nns::nac::Pad* pPad, const vfxdemo::DrawParam* pDrawParam, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlot ) NN_NOEXCEPT;
#endif
    void ProcVfxCalc( const vfxdemo::DrawParam* pDrawParam, nn::vfx::RegisterTextureViewSlot pRegisterTextureSlot ) NN_NOEXCEPT;
    void ProcDraw( const vfxdemo::DrawParam* pDrawParam, GpuStatistics* gpuStatistics, const ViewerLaunchArg* pViewerLaunchArg ) NN_NOEXCEPT;
    void ProcViewerDraw( const vfxdemo::DrawParam* pDrawParam ) NN_NOEXCEPT;
    void ProcVfxDraw( const vfxdemo::DrawParam* pDrawParam, GpuStatistics* gpuStatistics, ViewerLaunchArg::TextureOriginMode textureOriginMode ) NN_NOEXCEPT;


    void ProcModelDraw( nn::gfx::CommandBuffer* pCommnadBuffer ) NN_NOEXCEPT;
    void DrawEmitterSet( nn::gfx::CommandBuffer* pCommnadBuffer, int groupId, uint32_t drawPathFlag ) NN_NOEXCEPT;

    BackgroundTexture* GetCurrentBackgroundTexture() NN_NOEXCEPT
    {
        return &m_BackgroundTextures[m_BackgroundTexRepresentedId];
    }

    nn::gfx::ResTexture* GetBackgroundTexture() NN_NOEXCEPT
    {
        return m_BackgroundTextures[m_BackgroundTexRepresentedId].m_pTexture;
    }

    //---------------------------------------------------------------------------
    //! @brief                   デバッグ描画の有効/無効を取得します。
    //!
    //! @return                  デバッグ描画の有効/無効を返します。
    //---------------------------------------------------------------------------
    bool DoDebugDraw() const NN_NOEXCEPT { return m_DoDebugDraw; }

    //---------------------------------------------------------------------------
    //! @brief                   Gpuの計測結果の描画の有効/無効を取得します。
    //!
    //! @return                  Gpuの計測結果の描画の有効/無効を返します。
    //---------------------------------------------------------------------------
    bool DoGpuProfileDraw() const NN_NOEXCEPT { return m_DoGpuProfileDraw; }

    //---------------------------------------------------------------------------
    //! @brief                   エミッタ形状デバッグ描画の有効/無効を取得します。
    //!
    //! @return                  エミッタ形状デバッグ描画の有効/無効を返します。
    //---------------------------------------------------------------------------
    bool DoDebugEmitterShapeDraw() const NN_NOEXCEPT {  return m_DoDebugEmitterShapeDraw; }
    bool DoDebugEmitterShapeChildDraw() const NN_NOEXCEPT { return m_DoDebugEmitterShapeDrawChild; }
    bool DoDebugShapeFieldDraw() const NN_NOEXCEPT { return m_DoDebugShapeDrawField; }

    //---------------------------------------------------------------------------
    //! @brief                   グリッド描画の有効/無効を取得します。
    //!
    //! @return                  グリッド描画の有効/無効を返します。
    //---------------------------------------------------------------------------
    bool DoGridDraw() const NN_NOEXCEPT { return m_DoGridDraw; }

    //---------------------------------------------------------------------------
    //! @brief                   OverDrawモードの有効/無効を取得します。
    //!
    //! @return                  OverDrawモードの有効/無効を返します。
    //---------------------------------------------------------------------------
    bool DoOverDrawMode() const NN_NOEXCEPT { return m_DoOverDrawMode; }

    //---------------------------------------------------------------------------
    //! @brief                   リニアモードの有効/無効を設定します。
    //!
    //! @param[in] isLinearMode  リニアモードの有効/無効
    //---------------------------------------------------------------------------
    void SetLinearMode( bool isLinearMode ) NN_NOEXCEPT { m_IsLinearMode = isLinearMode; }

    //---------------------------------------------------------------------------
    //! @brief                   リニアモードの有効/無効を取得します。
    //!
    //! @return                  リニアモードの有効/無効を返します。
    //---------------------------------------------------------------------------
    bool GetLinearMode() NN_NOEXCEPT { return m_IsLinearMode; }

    //---------------------------------------------------------------------------
    //! @brief                   再生速度を設定します。
    //!
    //! @param[in] playSpeed     再生速度
    //---------------------------------------------------------------------------
    void SetPlaySpeed( float playSpeed ) NN_NOEXCEPT { m_PlaySpeed = playSpeed; }

    //---------------------------------------------------------------------------
    //! @brief                   再生速度を取得します。
    //!
    //! @return                  再生速度を返します。
    //---------------------------------------------------------------------------
    float GetPlaySpeed() NN_NOEXCEPT { return m_PlaySpeed; }

    //---------------------------------------------------------------------------
    //! @brief                   グリッドタイプを設定します。
    //!
    //! @param[in] gridType      グリッドタイプ
    //---------------------------------------------------------------------------
    void SetGridType( int gridType ) NN_NOEXCEPT { m_GridType = gridType; }

    //---------------------------------------------------------------------------
    //! @brief                   グリッドタイプを取得します。
    //!
    //! @return                  グリッドタイプを返します。
    //---------------------------------------------------------------------------
    int GetGridType() NN_NOEXCEPT { return m_GridType; }

    //---------------------------------------------------------------------------
    //! @brief                   カメラを設定します。
    //!
    //! @param[in] pCamera       カメラ
    //---------------------------------------------------------------------------
    void SetCamera( Camera* pCamera ) NN_NOEXCEPT { m_pCamera = pCamera; }

    //---------------------------------------------------------------------------
    //! @brief                   プロジェクションマトリクスを設定します。
    //!
    //! @param[in] pProjMatrix   プロジェクションマトリクス
    //---------------------------------------------------------------------------
    void SetProjectionMatrix( nn::util::Matrix4x4fType* pProjMatrix ) NN_NOEXCEPT { m_pProjMatrix = pProjMatrix; }

    //---------------------------------------------------------------------------
    //! @brief                   近クリップ平面への距離を設定します。
    //!
    //! @param[in] nearZ         近クリップ平面への距離
    //---------------------------------------------------------------------------
    void SetNearZ( float nearZ ) NN_NOEXCEPT { m_NearZ = nearZ; }

    //---------------------------------------------------------------------------
    //! @brief                   デバッグ描画の有効/無効を設定します。
    //!
    //! @param[in] doDebugDraw   デバッグ描画の有効/無効
    //---------------------------------------------------------------------------
    void SetDoDebugDraw(bool doDebugDraw) NN_NOEXCEPT { m_DoDebugDraw = doDebugDraw; }

    //---------------------------------------------------------------------------
    //! @brief                   　 OverDrawモードの有効/無効を設定します。
    //!
    //! @param[in] doOverDrawMode   OverDrawモードの有効/無効
    //---------------------------------------------------------------------------
    void SetDoOverDrawMode(bool doOverDrawMode) NN_NOEXCEPT
    {
        m_DoOverDrawMode = doOverDrawMode;
    }

    //---------------------------------------------------------------------------
    //! @brief                        Gpuの計測結果の描画の有効/無効を設定します。
    //!
    //! @param[in] doGpuProfileDraw   Gpuの計測結果の描画の有効/無効
    //---------------------------------------------------------------------------
    void SetDoGpuProfileDraw(bool doGpuProfileDraw) NN_NOEXCEPT { m_DoGpuProfileDraw = doGpuProfileDraw; }


    //---------------------------------------------------------------------------
    //! @brief                               エミッタ形状デバッグ描画の有効/無効を設定します。
    //!
    //! @param[in] doDebugEmitterShapeDraw   エミッタ形状デバッグ描画の描画の有効/無効
    //---------------------------------------------------------------------------
    void SetDoDebugEmitterShapeDraw(bool doDebugEmitterShapeDraw) NN_NOEXCEPT { m_DoDebugEmitterShapeDraw = doDebugEmitterShapeDraw; }
    void SetDoDebugEmitterShapeChildDraw(bool doDebugEmitterShapeChildDraw) NN_NOEXCEPT { m_DoDebugEmitterShapeDrawChild = doDebugEmitterShapeChildDraw; }
    void SetDoDebugShapeFieldDraw(bool doDebugShapeFieldDraw) NN_NOEXCEPT { m_DoDebugShapeDrawField = doDebugShapeFieldDraw; }

    //---------------------------------------------------------------------------
    //! @brief                   グリッド描画の有効/無効を設定します。
    //!
    //! @param[in] doGridDraw    グリッド描画の有効/無効
    //---------------------------------------------------------------------------
    void SetDoGridDraw(bool doGridDraw) NN_NOEXCEPT { m_DoGridDraw = doGridDraw; }

    //---------------------------------------------------------------------------
    //! @brief                   画角を設定します。
    //!
    //! @param[in] fov          画角を
    //---------------------------------------------------------------------------
    void SetFovy( float fov ) NN_NOEXCEPT { m_Fovy = fov; }

    //---------------------------------------------------------------------------
    //! @brief                   近クリップ平面への距離を取得します。
    //!
    //! @return                  近クリップ平面への距離を返します。
    //---------------------------------------------------------------------------
    float GetNearZ() NN_NOEXCEPT { return m_NearZ; }

    //---------------------------------------------------------------------------
    //! @brief                   遠クリップ平面への距離を設定します。
    //!
    //! @param[in] farZ          遠クリップ平面への距離
    //---------------------------------------------------------------------------
    void SetFarZ( float farZ ) NN_NOEXCEPT { m_FarZ = farZ; }

    //---------------------------------------------------------------------------
    //! @brief                   遠クリップ平面への距離を取得します。
    //!
    //! @return                  遠クリップ平面への距離を返します。
    //---------------------------------------------------------------------------
    float GetFarZ() NN_NOEXCEPT { return m_FarZ; }

    //---------------------------------------------------------------------------
    //! @brief                   カメラの姿勢を設定します。
    //!
    //! @param[in] posX          X座標
    //! @param[in] posY          Y座標
    //! @param[in] posZ          Z座標
    //! @param[in] lookAtX       注視点のX座標
    //! @param[in] lookAtY       注視点のY座標
    //! @param[in] lookAtZ       注視点のZ座標
    //! @param[in] rotation      回転核
    //! @param[in] projType      プロジェクションタイプ
    //! @param[in] aspect        アスペクト比
    //! @param[in] fov           パースペクティブ時の垂直画角
    //! @param[in] height        オルソ時の画像の高さ
    //! @param[in] nearZ         近クリップ平面への距離
    //! @param[in] farZ          遠クリップ平面への距離
    //---------------------------------------------------------------------------
    void SetCameraSettings(
        float posX,    float posY,    float posZ,
        float lookAtX, float lookAtY, float lookAtZ,
        float rotation,
        int projType,
        float aspect,
        float fov,
        float height,
        float nearZ, float farZ ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief                      背景カラーを設定します。
    //!
    //! @param[in] pBackgroundColor 背景カラー
    //---------------------------------------------------------------------------
    void SetBackgroundColor( const float* pBackgroundColor ) NN_NOEXCEPT { memcpy(m_BackgroundColor, pBackgroundColor, sizeof(m_BackgroundColor) ); }

    //---------------------------------------------------------------------------
    //! @brief                   背景カラーを取得します。
    //!
    //! @return                  背景カラーを返します。
    //---------------------------------------------------------------------------
    const float* GetBackgroundColor() NN_NOEXCEPT { return m_BackgroundColor; }

    //---------------------------------------------------------------------------
    //! @brief                   背景画像の乗算カラーを設定します。
    //---------------------------------------------------------------------------
    void SetBackgroundMultiplyColor( const nn::util::Vector4fType& backgroundMultiplyColor ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief                   背景画像を初期化します.
    //---------------------------------------------------------------------------
    void InitializeBackgroundTexture( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap, void* pResource, size_t resourceSize, size_t alignSize ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief                   背景画像を解放します.
    //---------------------------------------------------------------------------
    void FinalizeBackgroundTexture( nn::gfx::Device* pDevice, nn::vfx::Heap* pHeap ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief                   背景画像が初期化されているか取得します。
    //!
    //! @return                  背景画像が初期化されているか
    //---------------------------------------------------------------------------
    bool IsBackgroundTextureInitialized() const NN_NOEXCEPT { return m_BackgroundTextures[m_BackgroundTexRepresentedId].m_Initialized; }

    //---------------------------------------------------------------------------
    //! @brief                   Fovy を取得します。
    //! @return                  Fovy値
    //---------------------------------------------------------------------------
    float GetFovy() const NN_NOEXCEPT { return m_Fovy; }

    //---------------------------------------------------------------------------
    //! @brief                   垂直同期の待つ回数を取得します
    //!
    //! @return                  垂直同期の待つ回数
    //---------------------------------------------------------------------------
    int GetSwapInterval() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief                   パッドの入力受け付けるかどうかの状態を取得します
    //!
    //! @return                  パッドの入力可否状態
    //---------------------------------------------------------------------------
    bool GetPadState() NN_NOEXCEPT
    {
        return m_PadState;
    }

    //---------------------------------------------------------------------------
    //! @brief                   パッドの入力受け付けるかどうかの状態を設定します
    //!
    //! @param[in] padState      パッドの入力可否状態
    //---------------------------------------------------------------------------
    void SetPadState(bool padState) NN_NOEXCEPT
    {
        m_PadState = padState;
    }

    //---------------------------------------------------------------------------
    //! @brief                   フレームレートを設定します。
    //! @param[in] frameRate     フレームレート
    //---------------------------------------------------------------------------
    void SetFrameRate( INIT_SETTINGS_FRAME_RATE frameRate ) NN_NOEXCEPT
    {
        m_FrameRate = frameRate;
    }

    //---------------------------------------------------------------------------
    //! @brief                   ビューア背景画像用 テクスチャサンプラを取得します。
    //---------------------------------------------------------------------------
    const nn::gfx::DescriptorSlot& GetTextureSampler() const NN_NOEXCEPT
    {
        return m_BackgroundSamplerDesSlot;
    }

#if NN_VFX_VIEWER_USE_FONT
    //---------------------------------------------------------------------------
    //! @brief      エフェクトデモのデバッグ文字描画
    //! @param[in]  pTextWriter     テキストライターへのポインタ
    //---------------------------------------------------------------------------
    void DebugDraw( nn::gfx::util::DebugFontTextWriter* pTextWriter, nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, GpuStatistics* gpuStatistics ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      Gpu処理 詳細情報 デバッグ文字描画
    //---------------------------------------------------------------------------
    void DebugGpuStatisticsDraw(nn::gfx::util::DebugFontTextWriter* pTextWriter, nn::gfx::CommandBuffer* pCommandBuffer, nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, GpuStatistics* gpuStatistics ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      エフェクトデモのエミッタ内訳描画
    //! @param[in]  pTextWriter     テキストライターへのポインタ
    //! @param[in]  groupId         グループ ID
    //---------------------------------------------------------------------------
    void DebugDrawEmitterDetail( nn::gfx::util::DebugFontTextWriter* pTextWriter, int groupId ) NN_NOEXCEPT;
#endif

    //---------------------------------------------------------------------------
    //! @brief      グレアを使用してテクスチャをコピーします。
    //! @param[in]  commandBuffer      TBD
    //! @param[in]  primitiveRenderer  TBD
    //! @param[in]  bufferIndex        TBD
    //! @param[in]  pDstColorTarget    TBD
    //! @param[in]  srcTextureDescSlot TBD
    //---------------------------------------------------------------------------
    void GlareCopyTexture( nn::gfx::CommandBuffer*        commandBuffer,
                           nns::gfx::PrimitiveRenderer::Renderer* primitiveRenderer,
                           nn::gfx::ColorTargetView*      pDstColorTarget,
                           nn::gfx::DescriptorSlot        srcTextureDescSlot ) NN_NOEXCEPT
    {
        m_GlareEffect.CopyTexture( commandBuffer, primitiveRenderer, pDstColorTarget, srcTextureDescSlot );
    }

    //---------------------------------------------------------------------------
    //! @brief      OverDrawモード用テクスチャの作成
    //---------------------------------------------------------------------------
    void GenerateOverDrawTexture( nn::gfx::Device* pDevice, nn::vfx::Heap* heap ) NN_NOEXCEPT;

#if defined( NN_BUILD_CONFIG_OS_WIN )
    //---------------------------------------------------------------------------
    //! @brief      テストエミュレートモード時のカメラ位置を更新
    //! @param[in]  pPad        パッド
    //! @param[in]  pDrawParam  描画パラメータ
    //---------------------------------------------------------------------------
    void AdjustCameraFromUserData( nns::nac::Pad* pPad, vfxdemo::DrawParam* pDrawParam ) NN_NOEXCEPT;
#endif

    //---------------------------------------------------------------------------
    //! @brief      プロジェクションマトリックスを更新
    //! @param[in]  aspect        アスペクト比パッド
    //---------------------------------------------------------------------------
    void UpdateProjection( float aspect ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      グレア強度を設定します。
    //! @param[in]  glareLevel        グレア強度
    //---------------------------------------------------------------------------
    void SetGlareLevel( int glareLevel ) NN_NOEXCEPT
    {
        m_GlareBufferNum = glareLevel;
    }

    //---------------------------------------------------------------------------
    //! @brief      グリッドサイズを設定します。
    //! @param[in]  gridSize        グリッドサイズ
    //---------------------------------------------------------------------------
    void SetGridSize( float gridSize ) NN_NOEXCEPT
    {
        m_GridSize = gridSize;
    }

    //---------------------------------------------------------------------------
    //! @brief      グリッド分割数を設定します。
    //! @param[in]  gridSize        グリッド分割数
    //---------------------------------------------------------------------------
    void SetGridDivCount( int divCount ) NN_NOEXCEPT
    {
        m_GridDivCount = divCount;
    }

    //---------------------------------------------------------------------------
    //! @brief      キャプチャ・フレーム・ファイルパスを設定する。
    //! @param[in]  captureFilePath        キャプチャファイルパス
    //---------------------------------------------------------------------------
    void SetCaptureFilePath( const char* captureFilePath ) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief      キャプチャ・フレーム・ファイルパスを取得する。
    //---------------------------------------------------------------------------
    const char* GetCaptureFilePath() const NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        return m_CaptureFilePath;
#else
        return nullptr;
#endif
    }

private:
    //---------------------------------------------------------------------------
    //! @brief      エミッタ形状を描画する。
    //---------------------------------------------------------------------------
    void DebugEmitterShapeDraw( nn::gfx::CommandBuffer*                pCommandBuffer,
                                nns::gfx::PrimitiveRenderer::Renderer* pRenderer,
                                const nn::util::Matrix4x4fType*        pProjectionMatrix,
                                const nn::util::Matrix4x3fType*        pViewMatrix ) NN_NOEXCEPT;

    bool                                m_DoDebugDraw;                  //!< デバッグ描画の有効/無効
    bool                                m_DoGpuProfileDraw;             //!< GPUの計測結果の描画の有効/無効
    bool                                m_DoDebugEmitterShapeDraw;      //!< エミッタ形状デバッグ描画の有効/無効
    bool                                m_DoDebugEmitterShapeDrawChild; //!< チャイルドエミッタ形状デバッグ描画の有効/無効
    bool                                m_DoDebugShapeDrawField;        //!< フィールド形状デバッグ描画の有効/無効
    bool                                m_DoGridDraw;                   //!< グリッド表示のon/off
#if defined( NN_BUILD_CONFIG_OS_WIN )
    bool                                m_DoEmulatingTest;
#endif
    bool                                m_DoOverDrawMode;               //!< OverDrawモードのon/off
    bool                                m_PadState;                     //!< パッド入力の有効/無効
    bool                                m_IsLinearMode;                 //!< リニアモードの有効/無効
    float                               m_PlaySpeed;                    //!< 再生速度
    int                                 m_GridType;                     //!< グリッドタイプ
    Camera*                             m_pCamera;                      //!< カメラ
    nn::util::Matrix4x4fType*           m_pProjMatrix;                  //!< プロジェクションマトリクス
    float                               m_Fovy;                         //!< fovy
    float                               m_NearZ;                        //!< 近クリップ平面への距離
    float                               m_FarZ;                         //!< 遠クリップ平面への距離
    float                               m_OrthHeight;                   //!< オルソカメラの高さ
    int                                 m_ProjectionType;               //!< プロジェクションマトリックスタイプ
    float                               m_BackgroundColor[4];           //!< 背景カラー
    BackgroundTexture                   m_BackgroundTextures[2];        //!< 背景画像
    int                                 m_BackgroundTexRepresentedId;   //!< 現在使用している背景画像ID
    nn::util::Matrix4x4fType            m_BackgroundProjMtx;            //!< 背景画像向けプロジェクションマトリクス
    nn::util::Matrix4x3fType            m_BackgroundViewMtx;            //!< 背景画像向けビューマトリクス
    nn::gfx::Sampler                    m_BackgroundSampler;            //!< 背景画像向けサンプラ
    nn::gfx::DescriptorSlot             m_BackgroundSamplerDesSlot;     //!< 背景画像向けサンプラデスクリプタスロット
    nn::gfx::DescriptorSlot             m_BackgroundTextureDesSlot;     //!< 背景画像向けテクスチャデスクリプタスロット
    nn::util::Uint8x4                   m_BackgroundMultiplyColor;      //!< 背景画像向け乗算カラー
    int                                 m_Width;                        //!< バッファの基準となる大きさ
    int                                 m_Height;                       //!< バッファの基準となる大きさ
    GlareEffect                         m_GlareEffect;                  //!< グレア
    nn::gfx::RasterizerState            m_ModelRasterizerState;         //!< モデル表示用ラスタライザステート
#if defined( NN_BUILD_CONFIG_OS_WIN )
    bool                                m_IsMouseLButtonHold;           //!< マウスの左ボタンが押されているか
    bool                                m_IsMouseRButtonHold;           //!< マウスの右ボタンが押されているか
    bool                                m_IsMouseMButtonHold;           //!< マウスの中央ボタンが押されているか
    int                                 m_MouseWheel;                   //!< マウスのホイールの移動量
#endif
    float                               m_LStickX;                      //!< LスティックXの状態
    float                               m_LStickY;                      //!< LスティックYの状態
    int                                 m_GlareBufferNum;               //!< 利用するグレアバッファの数
    float                               m_GridSize;                     //!< グリッドサイズ
    int                                 m_GridDivCount;                 //!< グリッド分割数
    INIT_SETTINGS_FRAME_RATE            m_FrameRate;                    //!< フレームレート
#if defined( NN_BUILD_CONFIG_OS_WIN )
    char                                m_CaptureFilePath[_MAX_PATH];   //!< キャプチャ・ファイルパス
#endif
};
