﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/gfx/util/gfx_TransientDescriptorAllocatorHolder.h>

namespace vfxdemo {

//---------------------------------------------------------------------------
//! @brief        描画パラメータ
//---------------------------------------------------------------------------
class DrawParam
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    DrawParam() NN_NOEXCEPT
    {
        nn::util::VectorSet( &m_CameraPos,   0.f, 100.f, 100.f );
        nn::util::VectorSet( &m_LightTarget, 0.f,   0.f,   0.f );
        m_NearZ         = 0.1f;
        m_FarZ          = 1000.f;
        m_FrameRate     = 1.0f;
        nn::util::MatrixLookAtRightHanded( &m_ViewMtx, m_CameraPos, m_LightTarget, 0.0f );
        nn::util::MatrixPerspectiveFieldOfViewRightHanded( &m_ProjMtx, nn::util::FloatPi/3, 1.333f, m_NearZ, m_FarZ );
        m_CommandBuffer         = nullptr;
        m_PrimitiveRenderer    = nullptr;
    }

    nn::util::Matrix4x4fType                 m_ProjMtx;                      //!< プロジェクションマトリクス
    nn::util::Matrix4x3fType                 m_ViewMtx;                      //!< ビューマトリクス
    nn::util::Vector3fType                   m_CameraPos;                    //!< カメラ位置
    nn::util::Vector3fType                   m_LightTarget;                  //!< ライト視点
    float                                    m_NearZ;                        //!< スクリーンNear値
    float                                    m_FarZ;                         //!< スクリーンFar値
    float                                    m_FrameRate;                    //!< フレームレート
    nn::gfx::CommandBuffer*                  m_CommandBuffer;                //!< コマンドバッファ
    uint32_t                                 m_BufferIndex;                  //!< マルチバッファインデックス
    int                                      m_BufferWidth;                  //!< フレームバッファの幅
    int                                      m_BufferHeight;                 //!< フレームバッファの高さ
    float                                    m_BaseFovy;                     //!< 基準となる画角
    float                                    m_CurrentFovy;                  //!< 現在の画角
    nns::gfx::PrimitiveRenderer::Renderer*   m_PrimitiveRenderer;            //!< プリミティブレンダラ

    nn::gfx::Texture*                        m_ColorTexture;                 //!< カラーバッファ
    nn::gfx::ColorTargetView*                m_ColorTargetView;              //!< カラーターゲットビュー
    nn::gfx::TextureView*                    m_ColorTextureView;             //!< カラーテクスチャビュー
    nn::gfx::DescriptorSlot                  m_ColorDescSlot;                //!< カラーデスクリプタスロット

    nn::gfx::Texture*                        m_ColorCopyTexture;             //!< カラーコピーバッファ
    nn::gfx::ColorTargetView*                m_ColorCopyTargetView;          //!< カラーコピーターゲットビュー
    nn::gfx::TextureView*                    m_ColorCopyTextureView;         //!< カラーコピーテクスチャビュー
    nn::gfx::DescriptorSlot                  m_ColorCopyDescSlot;            //!< カラーコピーバッファデスクリプタスロット

    nn::gfx::Texture*                        m_DepthStencilTexture;          //!< 深度ステンシルバッファ
    nn::gfx::DepthStencilView*               m_DepthStencilView;             //!< 深度ステンシルビュー
    nn::gfx::TextureView*                    m_DepthStencilTextureView;      //!< 深度ステンシルテクスチャビュー
    nn::gfx::DescriptorSlot                  m_DepthStencilDescSlot;         //!< 深度ステンシルデスクリプタスロット

    nn::gfx::Texture*                        m_DepthStencilCopyTexture;      //!< 深度ステンシルコピーバッファ
    nn::gfx::ColorTargetView*                m_DepthStencilCopyTargetView;   //!< 深度ステンシルコピーターゲットビュー
    nn::gfx::TextureView*                    m_DepthStencilCopyTextureView;  //!< 深度ステンシルコピーテクスチャビュー
    nn::gfx::DescriptorSlot                  m_DepthStencilCopyDescSlot;     //!< 深度ステンシルコピーデスクリプタスロット
};

const int                       g_HistoryLengthMax  = 2;                 // SwapChainのバッファ数だけ用意(ダブルバッファリングであれば 2、トリプルバッファリングであれば 3 )

class IDescriptorIndexAllocator
{
public:
    IDescriptorIndexAllocator() NN_NOEXCEPT {}

    //---------------------------------------------------------------------------
    //! @brief       デスクリプタスロットを初期化
    //! @param[int]  descriptorPool 　  デスクリプタプール
    //! @param[int]  baseIndex          ディスクリプタスロットベースインデックス
    //! @param[int]  staticSlotCount    固定ディスクリプタスロット数
    //! @param[int]  transientSlotCount トランジェントディスクリプタスロット数
    //---------------------------------------------------------------------------
    void Initialize( nn::gfx::DescriptorPool*  descriptorPool, int baseIndex, int staticSlotCount, int transientSlotCount ) NN_NOEXCEPT
    {
        m_pDescPool = descriptorPool;

        // 固定デスクリプタインデックス アロケータの初期化
        m_baseIndex = baseIndex;
        m_StaticIndex = baseIndex;
        m_StaticIndexMaxCount = staticSlotCount;

        // ダイナミックデスクリプタインデックス アロケータの初期化
        m_TransientDescriptor.Initialize( m_pDescPool, baseIndex + staticSlotCount, transientSlotCount );
        m_TransientDescriptor.Get()->FillHistory();
    }

    //---------------------------------------------------------------------------
    //! @brief      デスクリプタスロット確保を開始
    //---------------------------------------------------------------------------
    void BeginAllocateTransientDescriptorSlot() NN_NOEXCEPT
    {
        nn::gfx::util::TransientDescriptorAllocator* pTransientDescAllocator = m_TransientDescriptor.Get();

        if ( !pTransientDescAllocator->IsRecording() )
        {
            pTransientDescAllocator->Free();
            pTransientDescAllocator->Begin();
        }
    }

    //---------------------------------------------------------------------------
    //! @brief      デスクリプタスロット確保を終了
    //---------------------------------------------------------------------------
    void EndAllocateTransientDescriptorSlot() NN_NOEXCEPT
    {
        nn::gfx::util::TransientDescriptorAllocator* pTransientDescAllocator = m_TransientDescriptor.Get();
        pTransientDescAllocator->End();
    }

    //---------------------------------------------------------------------------
    //! @brief     　固定デスクリプタスロットインデックスを取得
    //! @param[int]  requireCount   要求インデックス数
    //---------------------------------------------------------------------------
    int GetStaticDescriptorPoolIndex( int requireCount ) NN_NOEXCEPT
    {
        return GetDescriptorPoolIndex( requireCount );
    }

    //---------------------------------------------------------------------------
    //! @brief     　デスクリプタスプールを取得
    //---------------------------------------------------------------------------
    nn::gfx::DescriptorPool* GetDescriptorPool() NN_NOEXCEPT
    {
        return m_pDescPool;
    }

    //---------------------------------------------------------------------------
    //! @brief     　トランジェントデスクリプタアロケータを取得
    //---------------------------------------------------------------------------
    nn::gfx::util::TransientDescriptorAllocator* GetTransientDescriptorAllocator() NN_NOEXCEPT
    {
        return m_TransientDescriptor.Get();
    }

    //---------------------------------------------------------------------------
    //! @brief      デスクリプタプールのインデックスを取得
    //! @param[in]  requireCount    TBD
    //---------------------------------------------------------------------------
    int GetDescriptorPoolIndex( int requireCount ) NN_NOEXCEPT
    {
        int ret = m_StaticIndex;
        m_StaticIndex += requireCount;
        NN_SDK_ASSERT( m_StaticIndexMaxCount > m_StaticIndex - m_baseIndex );
        return ret;
    }

private:
    nn::gfx::util::TransientDescriptorAllocatorHolder<g_HistoryLengthMax> m_TransientDescriptor;
    nn::gfx::DescriptorPool*                    m_pDescPool;
    int                                         m_baseIndex;
    int                                         m_StaticIndex;
    int                                         m_StaticIndexMaxCount;
};


class TextureDescriptorIndexAllocator : public IDescriptorIndexAllocator
{
public:
    TextureDescriptorIndexAllocator() NN_NOEXCEPT {}

    //---------------------------------------------------------------------------
    //! @brief     　固定デスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  textureView    テクスチャビュー
    //! @param[int]  pUserData      ユーザデータ
    //---------------------------------------------------------------------------
    bool AllocateStaticTextureDescriptorSlot(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData ) NN_NOEXCEPT
    {
        NN_UNUSED(pUserData);

        // 固定ディスクリプタインデックス取得
        int slot = GetDescriptorPoolIndex( 1 );

        GetDescriptorPool()->BeginUpdate();
        {
            GetDescriptorPool()->SetTextureView( slot, &textureView);
        }
        GetDescriptorPool()->EndUpdate();
        GetDescriptorPool()->GetDescriptorSlot( dstSlot, slot );
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief     　トランジェントデスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  textureView    テクスチャビュー
    //! @param[int]  pUserData      ユーザデータ
    //---------------------------------------------------------------------------
    bool AllocateTransientTextureDescriptorSlot(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData ) NN_NOEXCEPT
    {
        NN_UNUSED(pUserData);

        // トランジェントディスクリプタインデックス取得
        int slot = GetTransientDescriptorAllocator()->Allocate();

        GetDescriptorPool()->BeginUpdate();
        {
            GetDescriptorPool()->SetTextureView( slot, &textureView);
        }
        GetDescriptorPool()->EndUpdate();
        GetDescriptorPool()->GetDescriptorSlot( dstSlot, slot );
        return true;
    }
};

class SamplerDescriptorIndexAllocator : public IDescriptorIndexAllocator
{
public:

    //---------------------------------------------------------------------------
    //! @brief      固定デスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  sampler    　　サンプラビュー
    //! @param[int]  pUserData      ユーザデータ
    //---------------------------------------------------------------------------
    bool AllocateStaticSamplerDescriptorSlot(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData ) NN_NOEXCEPT
    {
        NN_UNUSED(pUserData);

        // 固定ディスクリプタインデックス取得
        int slot = GetDescriptorPoolIndex( 1 );

        GetDescriptorPool()->BeginUpdate();
        {
            GetDescriptorPool()->SetSampler( slot, &sampler);
        }
        GetDescriptorPool()->EndUpdate();
        GetDescriptorPool()->GetDescriptorSlot( dstSlot, slot );
        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief       トランジェントデスクリプタスロットを確保
    //! @param[out]  dstSlot        確保したデスクリプタスロットを出力
    //! @param[int]  sampler    　　サンプラビュー
    //! @param[int]  pUserData      ユーザデータ
    //---------------------------------------------------------------------------
    bool AllocateTransientSamplerDescriptorSlot(nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData ) NN_NOEXCEPT
    {
        NN_UNUSED(pUserData);

        // トランジェントディスクリプタインデックス取得
        int slot = GetTransientDescriptorAllocator()->Allocate();

        GetDescriptorPool()->BeginUpdate();
        {
            GetDescriptorPool()->SetSampler( slot, &sampler);
        }
        GetDescriptorPool()->EndUpdate();
        GetDescriptorPool()->GetDescriptorSlot( dstSlot, slot );
        return true;
    }
};

} // namespace vfxdemo

