﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.BusinessLogic.Options;

namespace EffectMaker.UIControls.EffectBrowser.Utilities
{
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;

    using EffectMaker.UIControls.EffectBrowser.Data;
    using EffectMaker.UIControls.EffectBrowser.Data.FileCache;

    using FileInfo = EffectMaker.UIControls.EffectBrowser.Data.FileInfo;

    /// <summary>
    /// The file utility.
    /// </summary>
    public static class FileUtility
    {
        #region Public Methods and Operators

        /// <summary>
        /// The get file kind form ext.
        /// </summary>
        /// <param name="ext">
        /// The ext.
        /// </param>
        /// <returns>
        /// The <see cref="FileKindType"/>.
        /// </returns>
        public static FileKindType GetFileKindFormExt(string ext)
        {
            System.Diagnostics.Debug.Assert(Constants.SupportedExtTypes.ContainsKey(ext), "Constants.SupportedExtTypes.ContainsKey(ext)");

            return Constants.SupportedExtTypes[ext];
        }

        /// <summary>
        /// The make file infos from directory.
        /// </summary>
        /// <param name="dirName">
        /// The dir name.
        /// </param>
        /// <param name="withSubDir">
        /// The with sub dir.
        /// </param>
        /// <param name="isDeepLoad">
        /// The is deep load.
        /// </param>
        /// <returns>
        /// The <see cref="IEnumerable"/>.
        /// </returns>
        public static IEnumerable<FileInfo> MakeFileInfosFromDirectory(string dirName, bool withSubDir, bool isDeepLoad)
        {
            if (PathUtility.IsValidFilepathString(dirName) == false)
            {
                return new FileInfo[0];
            }

            try
            {
                var dirs =
                    PathUtility.EnumerateDirectories(dirName)
                        ////.AsParallel()
                        .Select(x => new FileInfo(FileCacheManager.Instance.Get(x, isDeepLoad)));

                var files =
                    PathUtility.EnumerateFiles(dirName)
                        ////.AsParallel()
                        .Where(x => IsSupportedExt(Path.GetExtension(x)))
                        .Select(x => new FileInfo(FileCacheManager.Instance.Get(x, isDeepLoad)));

                return withSubDir ? dirs.Concat(files) : files;
            }
            catch
            {
                return new FileInfo[0];
            }
        }

        #endregion

        #region Methods

        /// <summary>
        /// The is supported ext.
        /// </summary>
        /// <param name="ext">
        /// The ext.
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        private static bool IsSupportedExt(string ext)
        {
            if (!OptionStore.ProjectConfig.IsEftCombinerEditorEnabled && ext == Constants.ExtCombinerFile)
            {
                return false;
            }

            return Constants.SupportedExtTypes.ContainsKey(ext);
        }

        #endregion
    }
}
