﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing.Text;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.UIControls.EffectBrowser.Controls.AddressBar.DirBreadcrumbList;

namespace EffectMaker.UIControls.EffectBrowser.Data
{
    using System.IO;

    using EffectMaker.BusinessLogic.Serializer;
    using EffectMaker.DataModel.DataModels;
    using EffectMaker.DataModel.Specific.DataModels;
    using EffectMaker.UIControls.EffectBrowser.Data.FileCache;
    using EffectMaker.UIControls.EffectBrowser.Exceptions;

    /// <summary>
    /// The file info.
    /// </summary>
    public partial class FileInfo
    {
        #region Public Methods and Operators

        /// <summary>
        /// The save to file.
        /// </summary>
        /// <param name="isUpdateCacheTimestamp">
        /// The is update cache timestamp.
        /// </param>
        /// <exception cref="InvalidEsetException">
        /// InvalidEsetException
        /// </exception>
        public void SaveToFile(bool isUpdateCacheTimestamp)
        {
            switch (this.FileKind)
            {
                case FileKindType.EsetFile:
                    this.SaveToEset();
                    break;
                case FileKindType.PreviewFile:
                    this.SaveToPrev();
                    break;
                case FileKindType.WorkspaceFile:
                    this.SaveToWork();
                    break;
                case FileKindType.TextureFile:
                case FileKindType.PrimitiveFile:
                    this.SaveToIf();
                    break;
                default:
                    return;
            }

            if (isUpdateCacheTimestamp)
            {
                var cache = FileCacheManager.Instance.Get(this.FileFullPath, true);
                cache.UpdateCacheTimestamp();
            }
        }

        /// <summary>
        /// The get string from label color.
        /// </summary>
        /// <param name="colorType">
        /// The color type.
        /// </param>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        private static string GetStringFromLabelColor(LabelColorType colorType)
        {
            var rgbInt = UIConstants.LabelColorColors[(int)colorType];
            var result = string.Format(
                "{0} {1} {2}",
                (float)(rgbInt.R / 255.0f),
                (float)(rgbInt.G / 255.0f),
                (float)(rgbInt.B / 255.0f));

            return result;
        }

        /// <summary>
        /// ESETファイルにコメントとラベルカラーを書き込む
        /// </summary>
        private void SaveToEset()
        {
            EmitterSetDataSerializer serializer = new EmitterSetDataSerializer();

            // デシリアライズを実行
            var deserializeResult = serializer.DeserializeFromFile(this.FileFullPath);
            EmitterSetData emitterSetData = deserializeResult.DataModel;

            if (deserializeResult.IsSuccess == false)
            {
                throw new InvalidEsetException(this.FileFullPath);
            }

            // ラベルカラー, コメントを編集
            emitterSetData.EmitterSetBasicData.LabelColor = (int)this.LabelColor;
            emitterSetData.EmitterSetBasicData.Comment = this.Comment;

            // シリアライズを実行
            var serializeResult = serializer.SerializeToFile(emitterSetData, this.FileFullPath);

            if (serializeResult.IsSuccess == false)
            {
                throw new InvalidEsetException(this.FileFullPath);
            }
        }

        /// <summary>
        /// PREVファイルにコメントとラベルカラーを書き込む
        /// </summary>
        private void SaveToPrev()
        {
            PreviewDataSerializer serializer = new PreviewDataSerializer();

            // デシリアライズを実行
            var deserializeResult = serializer.DeserializeFromFile(this.FileFullPath);
            PreviewData previewData = deserializeResult.DataModel;

            if (deserializeResult.IsSuccess == false)
            {
                throw new InvalidEsetException(this.FileFullPath);
            }

            // ラベルカラー, コメントを編集
            previewData.BasicSettingData.PreviewBasicFileData.LabelColor = (int)this.LabelColor;
            previewData.BasicSettingData.PreviewBasicFileData.Comment = this.Comment;

            // シリアライズを実行
            var serializeResult = serializer.SerializeToFile(previewData, this.FileFullPath);

            if (serializeResult.IsSuccess == false)
            {
                throw new InvalidEsetException(this.FileFullPath);
            }
        }


        /// <summary>
        /// WORKファイルにコメントとラベルカラーを書き込む
        /// </summary>
        private void SaveToWork()
        {
            WorkspaceDataSerializer serializer = new WorkspaceDataSerializer();

            // デシリアライズを実行
            var deserializeResult = serializer.DeserializeFromFile(this.FileFullPath);
            WorkspaceStorageData workspaceData = deserializeResult.DataModel;

            if (deserializeResult.IsSuccess == false)
            {
                throw new InvalidEsetException(this.FileFullPath);
            }

            // コンストラクタで5個追加されている空のモデルノードを削除する
            workspaceData.ViewerData.ModelList.RemoveRange(0, 5);
            workspaceData.ViewerData.Children.RemoveRange(0, 5);

            // ラベルカラー, コメントを編集
            workspaceData.ViewerData.LabelColor = (int)this.LabelColor;
            workspaceData.ViewerData.Comment = this.Comment;

            // シリアライズを実行
            var serializeResult = serializer.SerializeToFile(deserializeResult.DataModel, this.FileFullPath);

            if (serializeResult.IsSuccess == false)
            {
                throw new InvalidEsetException(this.FileFullPath);
            }
        }

        /// <summary>
        /// 中間ファイルにコメントとラベルカラーを書き込む
        /// </summary>
        private void SaveToIf()
        {
            // 中間ファイルへのアクセスが同じ処理なので、プリミティブの実装を使い回し
            PrimitiveManager.Instance.SaveComment(
                this.FileFullPath,
                this.Comment,
                GetStringFromLabelColor(this.LabelColor));
        }

        #endregion
    }
}
