﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.UIControls.EffectBrowser.Utilities;
using System;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;

namespace EffectMaker.UIControls.EffectBrowser.Controls.Basic
{
    /// <summary>
    /// スプリッタコントロールで区切られたコントロールを縮小、復元します。
    /// </summary>
    public class EBSplitterShutter : UserControl
    {
        /// <summary>
        /// マウスカーソルがコントロール領域内にあるかどうか。
        /// </summary>
        private bool isMouseOver = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public EBSplitterShutter()
        {
        }

        /// <summary>
        /// ターゲットコントロールを取得または設定します。
        /// </summary>
        [Description("ターゲットコントロール")]
        public Control TargetControl { get; set; }

        /// <summary>
        /// ターゲットスプリッタコントロールを取得または設定します。
        /// </summary>
        [Description("ターゲットスプリッタコントロール")]
        public Splitter TargetSplitter { get; set; }

        /// <summary>
        /// 縮小する方向を取得または設定します。
        /// </summary>
        [Description("縮小方向")]
        public Direction ShutterDirection { get; set; } = Direction.Down;

        /// <summary>
        /// 開閉状態を取得または設定します。
        /// </summary>
        public bool TargetVisible
        {
            get
            {
                if (this.TargetControl != null)
                {
                    return this.TargetControl.Visible;
                }

                return false;
            }

            set
            {
                if (this.TargetVisible == value)
                {
                    return;
                }

                bool visibleChanged = false;

                // ターゲットコントロールの表示/非表示を設定
                if (this.TargetControl != null)
                {
                    this.TargetControl.Visible = value;
                    visibleChanged = true;
                }

                // ターゲットスプリッタコントロールの表示/非表示を設定
                if (this.TargetSplitter != null)
                {
                    this.TargetSplitter.Visible = value;
                    visibleChanged = true;
                }

                // VisibleChangedイベントを発生させる
                if (visibleChanged)
                {
                    this.Invalidate();
                    this.TargetVisibleChanged?.Invoke(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool CanRaiseEvents
        {
            get
            {
                return !ControlEventSuppressBlock.Enabled && base.CanRaiseEvents;
            }
        }

        /// <summary>
        /// 描画処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);

            Graphics g = e.Graphics;

            // 背景を描画
            if (this.isMouseOver && this.Enabled)
            {
                Color clearColor = this.BlendColor(this.BackColor, SystemColors.Highlight, 0.25f);
                g.Clear(clearColor);

                Rectangle rect = this.ClientRectangle;
                rect.Width -= 1;
                rect.Height -= 1;

                g.DrawRectangle(SystemPens.Highlight, rect);
            }
            else
            {
                g.Clear(this.BackColor);
            }

            Direction direction = Direction.None;

            // 開閉マークの向きを決定
            {
                bool visible = this.TargetVisible;

                switch (this.ShutterDirection)
                {
                    case Direction.Up:
                        direction = (visible ? Direction.Up : Direction.Down);
                        break;

                    case Direction.Down:
                        direction = (visible ? Direction.Down : Direction.Up);
                        break;

                    case Direction.Left:
                        direction = (visible ? Direction.Left : Direction.Right);
                        break;

                    case Direction.Right:
                        direction = (visible ? Direction.Right : Direction.Left);
                        break;
                }
            }

            // 開閉マークを描画
            {
                int cx = this.Width / 2;
                int cy = this.Height / 2;

                Color penColor = (this.Enabled ? SystemColors.ControlText : SystemColors.ControlDark);

                using (Pen pen = new Pen(penColor))
                {
                    switch (direction)
                    {
                        case Direction.Up:
                            g.DrawLine(pen, cx - 0, cy - 2, cx + 1, cy - 2);
                            g.DrawLine(pen, cx - 1, cy - 1, cx + 2, cy - 1);
                            g.DrawLine(pen, cx - 2, cy + 0, cx + 3, cy + 0);
                            g.DrawLine(pen, cx - 3, cy + 1, cx + 4, cy + 1);
                            break;

                        case Direction.Down:
                            g.DrawLine(pen, cx - 3, cy - 2, cx + 4, cy - 2);
                            g.DrawLine(pen, cx - 2, cy - 1, cx + 3, cy - 1);
                            g.DrawLine(pen, cx - 1, cy + 0, cx + 2, cy + 0);
                            g.DrawLine(pen, cx - 0, cy + 1, cx + 1, cy + 1);
                            break;

                        case Direction.Left:
                            g.DrawLine(pen, cx - 2, cy - 0, cx - 2, cy + 1);
                            g.DrawLine(pen, cx - 1, cy - 1, cx - 1, cy + 2);
                            g.DrawLine(pen, cx + 0, cy - 2, cx + 0, cy + 3);
                            g.DrawLine(pen, cx + 1, cy - 3, cx + 1, cy + 4);
                            break;

                        case Direction.Right:
                            g.DrawLine(pen, cx - 2, cy - 3, cx - 2, cy + 4);
                            g.DrawLine(pen, cx - 1, cy - 2, cx - 1, cy + 3);
                            g.DrawLine(pen, cx + 0, cy - 1, cx + 0, cy + 2);
                            g.DrawLine(pen, cx + 1, cy - 0, cx + 1, cy + 1);
                            break;
                    }
                }
            }
        }

        /// <summary>
        /// マウスカーソルがコントロール領域に入ったときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnMouseEnter(EventArgs e)
        {
            base.OnMouseEnter(e);

            this.isMouseOver = true;

            this.Invalidate();
        }

        /// <summary>
        /// マウスカーソルがコントロール領域から出たときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            this.isMouseOver = false;

            this.Invalidate();
        }

        /// <summary>
        /// クリックしたときの処理を行います。
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnClick(EventArgs e)
        {
            base.OnClick(e);

            this.TargetVisible = !this.TargetVisible;
        }

        /// <summary>
        /// カラーをブレンドします。
        /// </summary>
        private Color BlendColor(Color color1, Color color2, float ratio)
        {
            byte r = (byte)(color1.R * (1.0f - ratio) + color2.R * ratio);
            byte g = (byte)(color1.G * (1.0f - ratio) + color2.G * ratio);
            byte b = (byte)(color1.B * (1.0f - ratio) + color2.B * ratio);

            return Color.FromArgb(r, g, b);
        }

        /// <summary>
        /// 開閉状態が変わったときに発生します。
        /// </summary>
        public EventHandler TargetVisibleChanged;

        /// <summary>
        /// 向きです。
        /// </summary>
        public enum Direction
        {
            /// <summary>
            /// なし。
            /// </summary>
            None = AnchorStyles.None,

            /// <summary>
            /// 上。
            /// </summary>
            Up = AnchorStyles.Top,

            /// <summary>
            /// 下。
            /// </summary>
            Down = AnchorStyles.Bottom,

            /// <summary>
            /// 左。
            /// </summary>
            Left = AnchorStyles.Left,

            /// <summary>
            /// 右。
            /// </summary>
            Right = AnchorStyles.Right
        }
    }
}
