﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

namespace EffectMaker.UIDialogs.SearchDialog
{
    /// <summary>
    /// プログレスダイアログ。
    /// </summary>
    public partial class ProgressForm : Form
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ProgressForm()
        {
            InitializeComponent();
        }

        /// <summary>
        /// 進行状況メーターの現在位置を設定する。
        /// </summary>
        /// <remarks>
        /// 例えば、処理に200の工数があった場合、現在その200の工数の中のどの位置にいるかを示す値。
        /// 既定値は「0」。
        /// </remarks>
        public int ProgressValue
        {
            set { this.progressBar.Value = value; }
        }

        /// <summary>
        /// 進行状況を 0.0 ～ 1.0 で取得する。
        /// </summary>
        public float ProgressRate()
        {
            int length = progressBar.Maximum - progressBar.Minimum;
            if (length == 0)
                return 0.0f;
            return (float)progressBar.Value / (float)length;
        }

        /// <summary>
        /// 進行状況メーターの範囲の最大値を設定する。
        /// </summary>
        /// <remarks>
        /// 処理に200の工数があるなら「200」になる。
        /// 既定値は「100」。
        /// </remarks>
        public int ProgressMax
        {
            set { this.progressBar.Maximum = value; }
        }

        /// <summary>
        /// 進行状況メーターの範囲の最小値を設定する。
        /// </summary>
        /// <remarks>
        /// 既定値は「0」。
        /// </remarks>
        public int ProgressMin
        {
            set { this.progressBar.Minimum = value; }
        }

        /// <summary>
        /// PerformStepメソッドを呼び出したときに、進行状況メーターの現在位置を進める量（ProgressValueの増分値）を設定する。
        /// </summary>
        /// <remarks>
        /// 処理工数が200で、5つの工数が終わった段階で進行状況メーターを更新したい場合は「5」にする。
        /// 既定値は「10」。
        /// </remarks>
        public int ProgressStep
        {
            set { this.progressBar.Step = value; }
        }

        /// <summary>
        /// 進行状況メーターの現在位置（ProgressValue）をProgressStepプロパティの量だけ進める。
        /// </summary>
        public void PerformStep()
        {
            this.progressBar.PerformStep();
            Application.DoEvents();
        }

    }

    /// <summary>
    /// 進行状況ダイアログを表示するためのクラス
    /// </summary>
    public class SearchProgressDialog : IDisposable
    {
        //キャンセルボタンがクリックされたか
        private volatile bool _canceled = false;
        //ダイアログフォーム
        private volatile ProgressForm form;
        //フォームが表示されるまで待機するための待機ハンドル
        private System.Threading.ManualResetEvent _startEvent;
        //フォームが一度表示されたか
        private bool _showed = false;
        //フォームをコードで閉じているか
        private volatile bool closing = false;
        //オーナーフォーム
        private Form _ownerForm;

        //別処理をするためのスレッド
        private System.Threading.Thread _thread;

        //フォームのタイトル
        private volatile string _title = "進行状況";
        //ProgressBarの最小、最大、現在の値
        private volatile int _minimum = 0;
        private volatile int _maximum = 100;
        private volatile int _value = 0;
        //表示するメッセージ
        private volatile string _message = "";

        private DateTime _startDate;

        private int _lastLeftSeconds = -1;

        private int _counterForLeftSeconds = 0;

        /// <summary>
        /// ダイアログのタイトルバーに表示する文字列
        /// </summary>
        public string Title
        {
            set
            {
                _title = value;
                if (form != null)
                    form.Invoke(new MethodInvoker(SetTitle));
            }
            get { return _message; }
        }

        /// <summary>
        /// プログレスバーの最小値
        /// </summary>
        public int Minimum
        {
            set
            {
                _minimum = value;
                if (form != null)
                    form.Invoke(new MethodInvoker(SetProgressMinimum));
            }
            get { return _minimum; }
        }

        /// <summary>
        /// プログレスバーの最大値
        /// </summary>
        public int Maximum
        {
            set
            {
                _maximum = value;
                if (form != null)
                    form.Invoke(new MethodInvoker(SetProgressMaximun));
            }
            get { return _maximum; }
        }

        /// <summary>
        /// プログレスバーの値
        /// </summary>
        public int Value
        {
            set
            {
                _value = value;
                if (form != null)
                {
                    form.Invoke(new MethodInvoker(SetProgressValue));
                }
            }
            get { return _value; }
        }

        /// <summary>
        /// ダイアログに表示するメッセージ
        /// </summary>
        public string Message
        {
            set
            {
                _message = value;
                if (form != null)
                    form.Invoke(new MethodInvoker(SetMessage));
            }
            get { return _message; }
        }

        /// <summary>
        /// キャンセルされたか
        /// </summary>
        public bool Canceled
        {
            get { return _canceled; }
        }

        /// <summary>
        /// ダイアログを表示する
        /// </summary>
        /// <param name="owner">
        /// ownerの中央にダイアログが表示される
        /// </param>
        /// <remarks>
        /// このメソッドは一回しか呼び出せません。
        /// </remarks>
        public void Show(Form owner)
        {
            if (_showed)
                throw new Exception("ダイアログは一度表示されています。");

            _showed = true;

            _canceled = false;
            _startEvent = new System.Threading.ManualResetEvent(false);
            _ownerForm = owner;

            //スレッドを作成
            _thread = new System.Threading.Thread(
                new System.Threading.ThreadStart(Run));
            _thread.IsBackground = true;
            this._thread.SetApartmentState(System.Threading.ApartmentState.STA);
            _thread.Start();

            //フォームが表示されるまで待機する
            _startEvent.WaitOne();
        }

        /// <summary>
        ///
        /// </summary>
        public void Show()
        {
            Show(null);
        }

        /// <summary>
        /// 別スレッドで処理するメソッド
        /// </summary>
        private void Run()
        {
            //フォームの設定
            form = new ProgressForm();
            form.Text = _title;
            form.btnCancel.Click += new EventHandler(Button1_Click);
            form.Closing += new CancelEventHandler(form_Closing);
            form.Activated += new EventHandler(form_Activated);
            form.progressBar.Minimum = _minimum;
            form.progressBar.Maximum = _maximum;
            form.progressBar.Value = _value;
            form.tmrProcess.Tick += Event_TmrProcess_Tick;

            // タイマー起動
            form.tmrProcess.Start();

            // 時間
            _startDate = DateTime.Now;
            UpdateLabel();

            //フォームの表示位置をオーナーの中央へ
            if (_ownerForm != null)
            {
                form.StartPosition = FormStartPosition.Manual;
                form.Left =
                    _ownerForm.Left + (_ownerForm.Width - form.Width) / 2;
                form.Top =
                    _ownerForm.Top + (_ownerForm.Height - form.Height) / 2;
            }
            //フォームの表示
            form.ShowDialog();

            form.Dispose();
        }

        /// <summary>
        /// 現在の進行状況を 0.0 ～ 1.0 で取得する。
        /// </summary>
        private float GetCurrentProgressRateFloat()
        {
            return form.ProgressRate();
        }

        /// <summary>
        /// 現在の進行状況を 0 ～ 100 で取得する。
        /// </summary>
        private int GetCurrentProgressRateAsInt()
        {
            return (int)(Math.Round(GetCurrentProgressRateFloat() * 100));
        }

        /// <summary>
        /// 残り時間を取得する。
        /// </summary>
        private TimeSpan GetLeftTime()
        {
            //TimeSpan span = new TimeSpan(0, 0, 0, 0, GetLeftTimeAsMilliseconds());
            TimeSpan span = new TimeSpan(0, 0, 0, GetLeftTimeAsSeconds(), 0);

            return span;
        }

        /// <summary>
        /// 残り時間をミリ秒で取得する。
        /// </summary>
        private int GetLeftTimeAsMilliseconds()
        {
            int ms = GetPassedTimeAsMilliseconds();
            float rate = GetCurrentProgressRateFloat();

            if (ms == 0)
                return -1;

            float result = (1.0f - rate) * (ms / rate);

            return (int)(Math.Round(result));
        }

        /// <summary>
        /// 残り時間を秒で取得する。
        /// </summary>
        private int GetLeftTimeAsSeconds()
        {
            int interval = 20;

            _counterForLeftSeconds++;
            if (_counterForLeftSeconds % interval != 0 && _lastLeftSeconds != -1)
                return _lastLeftSeconds;

            int s = GetPassedTimeAsSeconds();
            float rate = GetCurrentProgressRateFloat();

            if (s == 0)
                return -1;

            float secondsFloat = (1.0f - rate) * (s / rate);

            int secondsInt = (int)(Math.Round(secondsFloat));

            if (_lastLeftSeconds == -1)
                _lastLeftSeconds = secondsInt;

            float r1 = 0.5f;
            float r2 = 1.0f - r1;

            int result = (int)(Math.Round((secondsInt * r1 + _lastLeftSeconds * r2)));

            _lastLeftSeconds = result;

            return result;
        }

        /// <summary>
        /// 経過時間を取得する。
        /// </summary>
        private TimeSpan GetPassedTime()
        {
            return DateTime.Now - _startDate;
        }

        /// <summary>
        /// 経過時間をミリ秒で取得する。
        /// </summary>
        private int GetPassedTimeAsMilliseconds()
        {
            return (int)(Math.Round(GetPassedTime().TotalMilliseconds));
        }

        /// <summary>
        /// 経過時間をミリ秒で取得する。
        /// </summary>
        private int GetPassedTimeAsSeconds()
        {
            return (int)(Math.Round(GetPassedTime().TotalSeconds));
        }

        /// <summary>
        ///
        /// </summary>
        private string GetTimeInfoString()
        {
            string result = string.Empty;

            TimeSpan timePassed = DateTime.Now - _startDate;

            result += string.Format("開始時刻 {0}\r\n", ToString(_startDate));
            result += string.Format("経過時間 {0}\r\n", ToString(timePassed));
            result += string.Format("進捗率 {0}%\r\n", GetCurrentProgressRateAsInt());

            if (GetPassedTimeAsMilliseconds() < 5000)
            {
                result += string.Format("残り時間 {0}\r\n", "-");
            }
            else
            {
                int ms = GetLeftTimeAsSeconds();
                TimeSpan timeLeft = GetLeftTime();
                result += string.Format("残り時間 {0}\r\n", ToString(timeLeft));
            }

            return result;
        }

        private string ToString(DateTime src)
        {
            return string.Format("{0:00}:{1:00}:{2:00}", src.Hour, src.Minute, src.Second);
        }

        private string ToString(TimeSpan src)
        {
            return string.Format("{0:00}:{1:00}:{2:00}", src.Hours, src.Minutes, src.Seconds);
        }

        /// <summary>
        /// ダイアログを閉じる
        /// </summary>
        public void Close()
        {
            closing = true;
            form.Invoke(new MethodInvoker(form.Close));
        }

        /// <summary>
        ///
        /// </summary>
        public void Dispose()
        {
            // タイマー停止
            form.tmrProcess.Stop();

            form.Invoke(new MethodInvoker(form.Dispose));
        }

        private void SetProgressValue()
        {
            if (form != null && !form.IsDisposed)
                form.progressBar.Value = _value;
        }

        private void SetMessage()
        {
            if (form != null && !form.IsDisposed)
                form.lblState.Text = _message;
        }

        private void SetTitle()
        {
            if (form != null && !form.IsDisposed)
                form.Text = _title;
        }

        private void SetProgressMaximun()
        {
            if (form != null && !form.IsDisposed)
                form.progressBar.Maximum = _maximum;
        }

        private void SetProgressMinimum()
        {
            if (form != null && !form.IsDisposed)
                form.progressBar.Minimum = _minimum;
        }

        private void Button1_Click(object sender, EventArgs e)
        {
            _canceled = true;
        }

        private void form_Closing(object sender, CancelEventArgs e)
        {
            if (!closing)
            {
                e.Cancel = true;
                _canceled = true;
            }
        }

        private void form_Activated(object sender, EventArgs e)
        {
            form.Activated -= new EventHandler(form_Activated);
            _startEvent.Set();
        }

        //---------------------------------------------------------------------
        // タイマー
        private void Event_TmrProcess_Tick(object sender, EventArgs e)
        {
            UpdateLabel();
        }

        private void UpdateLabel()
        {
            form.txbInfo.Text = GetTimeInfoString();
        }

    }


}
