﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace EffectMaker.UIDialogs.ExportDialog
{
    /// <summary>
    /// ソート状態を示すアイコンを表示するためのユーティリティクラス
    /// </summary>
    public static class ColumnUtil
    {
        /// <summary>
        /// The lvm getheader.
        /// </summary>
        private const uint LvmGetheader = 4127;

        /// <summary>
        /// The hdm setimagelist.
        /// </summary>
        private const uint HdmSetimagelist = 4616;

        /// <summary>
        /// The lvm setcolumn.
        /// </summary>
        private const uint LvmSetcolumn = 4122;

        /// <summary>
        /// The lvcf fmt.
        /// </summary>
        private const uint LvcfFmt = 1;

        /// <summary>
        /// The lvcf image.
        /// </summary>
        private const uint LvcfImage = 16;

        /// <summary>
        /// The lvcfmt image.
        /// </summary>
        private const int LvcfmtImage = 2048;

        /// <summary>
        /// Same as LVCFMT_BITMAP_ON_RIGHT
        /// </summary>
        private const int HdfBitmapOnRight = 0x1000;

        /// <summary>
        /// Windows7 Explorer のソート順表示
        /// </summary>
        private const int HdfSortup = 0x0400;

        /// <summary>
        /// Windows7 Explorer のソート順表示.
        /// </summary>
        private const int HdfSortdown = 0x0200;

        /// <summary>
        /// ソート状態を示すアイコンを更新します。
        /// </summary>
        /// <param name="listView">
        /// The list view.
        /// </param>
        /// <param name="columnIndex">
        /// The column index.
        /// </param>
        /// <param name="order">
        /// The order.
        /// </param>
        public static void UpdateSortIcon(ListView listView, int columnIndex, SortOrder order)
        {
            int i;

            // 空のイメージリストを作り、サイズを最小にする
            var imageList = new ImageList { ImageSize = new Size(1, 1) };

            // リストビューのカラム部分のハンドルを取得
            IntPtr colHeader = SendMessage(listView.Handle, LvmGetheader, (uint)0, (uint)0);

            // カラム部のハンドルにイメージリストを割り当て
            SendMessage(colHeader, HdmSetimagelist, (uint)0, (uint)imageList.Handle);

            // カラムの設定
            for (i = 0; i < listView.Columns.Count; i++)
            {
                // Use the LVM_SETCOLUMN message to set the column's image index.
                LVCOLUMN col;
                col.Mask = LvcfFmt | LvcfImage;

                if (i == columnIndex)
                {
                    switch (order)
                    {
                        case SortOrder.Ascending:
                            col.Fmt = LvcfmtImage | HdfSortup | HdfBitmapOnRight;
                            break;
                        case SortOrder.Descending:
                            col.Fmt = LvcfmtImage | HdfSortdown | HdfBitmapOnRight;
                            break;
                        default:
                            col.Fmt = LvcfmtImage | HdfBitmapOnRight;
                            break;
                    }
                }
                else
                {
                    col.Fmt = LvcfmtImage | HdfBitmapOnRight;
                }

                // 隠しリソースを使うのでイメージインデックスは0でOK
                col.IImage = 0;

                // Initialize the rest to zero.
                col.PszText = (IntPtr)0;
                col.CchTextMax = 0;
                col.Cx = 0;
                col.ISubItem = 0;
                col.IOrder = 0;

                // Send the LVM_SETCOLUMN message.
                // The column that we are assigning the image to is defined in the third parameter.
                SendMessage(listView.Handle, LvmSetcolumn, (uint)i, ref col);
            }
        }

        /// <summary>
        /// SendMessageラッパ
        /// </summary>
        /// <param name="hwnd">ハンドル</param>
        /// <param name="msg">メッセージ</param>
        /// <param name="wparam">wParam</param>
        /// <param name="lparam">lParam</param>
        /// <returns>メッセージ処理結果</returns>
        [DllImport("user32.dll")]
        private static extern IntPtr SendMessage(IntPtr hwnd, uint msg, uint wparam, uint lparam);

        /// <summary>
        /// SendMessageラッパ
        /// </summary>
        /// <param name="hwnd">ハンドル</param>
        /// <param name="msg">メッセージ</param>
        /// <param name="wparam">wParam</param>
        /// <param name="lparam">lParam</param>
        /// <returns>メッセージ処理結果</returns>
        [DllImport("User32.dll", CharSet = CharSet.Auto)]
        private static extern IntPtr SendMessage(IntPtr hwnd, uint msg, uint wparam, ref LVCOLUMN lparam);

        /// <summary>
        /// ListView のカラム情報設定用構造体
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 8, CharSet = CharSet.Auto)]
        private struct LVCOLUMN
        {
            /// <summary>
            /// The mask.
            /// </summary>
            public uint Mask;

            /// <summary>
            /// The fmt.
            /// </summary>
            public int Fmt;

            /// <summary>
            /// The cx.
            /// </summary>
            public int Cx;

            /// <summary>
            /// The psz text.
            /// </summary>
            public IntPtr PszText;

            /// <summary>
            /// The cch text max.
            /// </summary>
            public int CchTextMax;

            /// <summary>
            /// The i sub item.
            /// </summary>
            public int ISubItem;

            /// <summary>
            /// The i image.
            /// </summary>
            public int IImage;

            /// <summary>
            /// The i order.
            /// </summary>
            public int IOrder;
        }
    }

    /// <summary>
    /// ListViewの項目の並び替えに使用するクラス
    /// </summary>
    public class ListViewItemComparer : IComparer
    {
        /// <summary>
        /// The _column.
        /// </summary>
        private int column;

        /// <summary>
        /// The _order.
        /// </summary>
        private SortOrder order;

        /// <summary>
        /// The _mode.
        /// </summary>
        private ComparerMode mode;

        /// <summary>
        /// The _column modes.
        /// </summary>
        private ComparerMode[] columnModes;

        /// <summary>
        /// ListViewItemComparerクラスのコンストラクタ
        /// </summary>
        /// <param name="col">並び替える列の番号</param>
        /// <param name="ord">昇順か降順か</param>
        /// <param name="cmod">並び替えの方法</param>
        public ListViewItemComparer(
            int col,
            SortOrder ord,
            ComparerMode cmod)
        {
            this.column = col;
            this.order = ord;
            this.mode = cmod;
        }

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public ListViewItemComparer()
        {
            this.column = 0;
            this.order = SortOrder.None;
            this.mode = ComparerMode.String;
        }

        /// <summary>
        /// 比較する方法
        /// </summary>
        public enum ComparerMode
        {
            /// <summary>
            /// 文字列として比較
            /// </summary>
            String,

            /// <summary>
            /// 数値（Int32型）として比較
            /// </summary>
            Integer,

            /// <summary>
            /// 日時（DataTime型）として比較
            /// </summary>
            DateTime
        }

        /// <summary>
        /// 並び替えるListView列の番号
        /// </summary>
        public int Column
        {
            get
            {
                return this.column;
            }

            set
            {
                // 現在と同じ列の時は、昇順降順を切り替える
                if (this.column == value)
                {
                    if (this.order == SortOrder.None)
                    {
                        this.order = SortOrder.Ascending;
                    }
                    else if (this.order == SortOrder.Ascending)
                    {
                        this.order = SortOrder.Descending;
                    }
                    else if (this.order == SortOrder.Descending)
                    {
                        this.order = SortOrder.None;
                    }
                }
                else
                {
                    this.order = SortOrder.Ascending;
                }

                this.column = value;
            }
        }

        /// <summary>
        /// 昇順か降順か
        /// </summary>
        public SortOrder Order
        {
            get
            {
                return this.order;
            }

            set
            {
                this.order = value;
            }
        }

        /// <summary>
        /// 並び替えの方法
        /// </summary>
        public ComparerMode Mode
        {
            get
            {
                return this.mode;
            }

            set
            {
                this.mode = value;
            }
        }

        /// <summary>
        /// 列ごとの並び替えの方法
        /// </summary>
        public ComparerMode[] ColumnModes
        {
            set
            {
                this.columnModes = value;
            }
        }

        /// <summary>
        /// 比較関数
        /// </summary>
        /// <param name="x">
        /// The x.
        /// </param>
        /// <param name="y">
        /// The y.
        /// </param>
        /// <returns>
        /// xがyより小さいときはマイナスの数、大きいときはプラスの数、
        /// 同じときは0を返す
        /// </returns>
        public int Compare(object x, object y)
        {
            int tmpCol = (this.order == SortOrder.None) ? 4 : this.column;

            int result = 0;

            // ListViewItemの取得
            var itemx = (ListViewItem)x;
            var itemy = (ListViewItem)y;

            // 並べ替えの方法を決定
            if (this.columnModes != null && this.columnModes.Length > tmpCol)
            {
                this.mode = this.columnModes[tmpCol];
            }

            // 並び替えの方法別に、xとyを比較する
            switch (this.mode)
            {
                case ComparerMode.String:
                    // 文字列をとして比較
                    result = string.CompareOrdinal(
                        itemx.SubItems[tmpCol].Text,
                        itemy.SubItems[tmpCol].Text);
                    break;
                case ComparerMode.Integer:
                    // Int32に変換して比較
                    // .NET Framework 2.0からは、TryParseメソッドを使うこともできる
                    result = int.Parse(itemx.SubItems[tmpCol].Text).CompareTo(
                        int.Parse(itemy.SubItems[tmpCol].Text));
                    break;
                case ComparerMode.DateTime:
                    // DateTimeに変換して比較
                    // .NET Framework 2.0からは、TryParseメソッドを使うこともできる
                    result = DateTime.Compare(
                        DateTime.Parse(itemx.SubItems[tmpCol].Text),
                        DateTime.Parse(itemy.SubItems[tmpCol].Text));
                    break;
            }

            // 降順の時は結果を+-逆にする
            if (this.order == SortOrder.Descending)
            {
                result = -result;
            }

            // 結果を返す
            return result;
        }
    }
}
