﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Windows.Forms.Layout;

using EffectMaker.Foundation.Log;

namespace EffectMaker.UIControls.Layout
{
    /// <summary>
    /// Base class for layout engine.
    /// </summary>
    public abstract class LayoutEngineBase : LayoutEngine
    {
        /// <summary>A counter to handle nested calls to suspend layout.</summary>
        private static int layoutSuspendCount = 0;

        /// <summary>The flag indicating whether layout events should be postponed.</summary>
        private static bool isDelayingLayout = false;

        /// <summary>All the created layout engine instances.</summary>
        private static List<LayoutEngineBase> instances = new List<LayoutEngineBase>();

        /// <summary>Stores flag telling whether parent should perform layout or not.</summary>
        private bool requireParentLayout = false;

        /// <summary>
        /// The flag indicating whether there is any layout request while layout is suspended.
        /// </summary>
        private bool shouldPerformLayout = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        protected LayoutEngineBase() : base()
        {
            instances.Add(this);
        }

        /// <summary>
        /// Finalizes an instance of the <see cref="LayoutEngineBase"/> class.
        /// Basically we don't need this, for the controls are not destroyed until
        /// the application exits. The destructor is only made to be safe.
        /// </summary>
        ~LayoutEngineBase()
        {
            instances.Remove(this);
        }

        /// <summary>
        /// Raised when a layout phase must be performed.
        /// </summary>
        public event EventHandler RequestLayout;

        /// <summary>
        /// Get or set the flag indicating whether layout is suspended.
        /// When the flag is true, *ALL* the layout engines stop performing
        /// layout.
        /// </summary>
        public static bool IsLayoutSuspended
        {
            get { return layoutSuspendCount > 0; }
        }

        /// <summary>
        /// Suspend layout for all the layout engines.
        /// </summary>
        public static void SuspendLayout()
        {
            ++layoutSuspendCount;
        }

        /// <summary>
        /// Resume layout for all the layout engines.
        /// </summary>
        public static void ResumeLayout()
        {
            --layoutSuspendCount;
        }

        /// <summary>
        /// Delay layout events.
        /// The layout engine instances that have been requested to perform layout
        /// will be stored in a hash set. After the specified time duration is up,
        /// all these layout engines perform layout all together to prevent redundant
        /// layout operations.
        /// </summary>
        /// <param name="milliseconds">Time to delay.</param>
        public static async void DelayLayout(int milliseconds)
        {
            isDelayingLayout = true;

            // Wait for the specified time duration.
            // All the layout requests will be postponed.
            await System.Threading.Tasks.Task.Delay(milliseconds);

            isDelayingLayout = false;

            // Perform layout for those needed.
            foreach (var instance in instances)
            {
                if (instance.shouldPerformLayout == true)
                {
                    instance.OnRequestLayout(EventArgs.Empty);
                    instance.shouldPerformLayout = false;
                }
            }
        }

        /// <summary>
        ///  Initializes the layout engine.
        /// </summary>
        /// <param name="child">The container on which the layout engine will operate.</param>
        /// <param name="specified">The bounds defining the container's size and position.</param>
        public sealed override void InitLayout(object child, BoundsSpecified specified)
        {
            if (child is Control && child is IControl && child is ILayoutElement)
            {
                // TODO: perform exception handling
                this.OnInitLayout((Control)child, specified);
            }
            else
            {
                base.InitLayout(child, specified);
            }
        }

        /// <summary>
        /// Requests that the layout engine perform a layout operation.
        /// </summary>
        /// <param name="container">The container on which the layout engine will operate.</param>
        /// <param name="layoutEventArgs">An event argument from a Control.Layout event.</param>
        /// <returns>Returns true if layout should be performed again by the parent of container,
        /// false otherwise.</returns>
        public sealed override bool Layout(object container, LayoutEventArgs layoutEventArgs)
        {
            // Bail out if layout requests are suspended.
            if (IsLayoutSuspended == true)
            {
                this.shouldPerformLayout = true;
                return false;
            }

            // If the layout requests should be postponed, set the flag and wait
            // until the delaying duration is up, layout will be updated altogether.
            if (isDelayingLayout == true)
            {
                this.shouldPerformLayout = true;
                return false;
            }

            bool layoutResult;

            if (container is Control &&
                container is IControl &&
                container is ILayoutElement)
            {
                // TODO: perform exception handling
                var control = (Control)container;

                Size parentSize;

                if (control.Parent == null)
                {
                    var form = control.FindForm();

                    if (form == null)
                    {
                        return true;
                    }

                    parentSize = form.ClientSize;
                }
                else
                {
                    parentSize = control.Parent.DisplayRectangle.Size;
                }

                layoutResult = this.OnLayout(control, parentSize, layoutEventArgs);
            }
            else
            {
                layoutResult = base.Layout(container, layoutEventArgs);
            }

            if (this.requireParentLayout)
            {
                this.requireParentLayout = false;
                return true;
            }

            return layoutResult;
        }

        /// <summary>
        /// When overridden, initializes the layout engine.
        /// </summary>
        /// <param name="child">The container on which the layout engine will operate.</param>
        /// <param name="specified">The bounds defining the container's size and position.</param>
        protected virtual void OnInitLayout(Control child, BoundsSpecified specified)
        {
        }

        /// <summary>
        /// When overridden, requests that the layout engine perform a layout operation.
        /// </summary>
        /// <param name="container">The container on which the layout engine will operate.</param>
        /// <param name="containerParentSize">Size of the parent of the current container.</param>
        /// <param name="layoutEventArgs">An event argument from a Control.Layout event.</param>
        /// <returns>Returns true if layout should be performed again by the parent of container,
        /// false otherwise.</returns>
        protected abstract bool OnLayout(
            Control container,
            Size containerParentSize,
            LayoutEventArgs layoutEventArgs);

        /// <summary>
        /// Raises the RequestLayout event.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected virtual void OnRequestLayout(EventArgs e)
        {
            if (IsLayoutSuspended == true)
            {
                return;
            }

            this.requireParentLayout = true;

            var handler = this.RequestLayout;
            if (handler != null)
            {
                handler(this, e);
            }
        }
    }
}
