﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Windows.Markup;

namespace EffectMaker.UIControls.Layout
{
    /// <summary>
    /// An extended UILayoutPanel with grid layout capabilities.
    /// </summary>
    public class Grid : UILayoutPanel<GridLayoutEngine>
    {
        /// <summary>
        /// Stores the cell definition created by attached properties from XAML.
        /// </summary>
        private static Dictionary<Control, CellDefinition> cellDefinitions =
            new Dictionary<Control, CellDefinition>();

        /// <summary>
        /// Gets the row definitions.
        /// </summary>
        public ICollection<RowDefinition> RowDefinitions
        {
            get
            {
                return this.InternalLayoutEngine.RowDefinitions;
            }
        }

        /// <summary>
        /// Gets the column definitions.
        /// </summary>
        public ICollection<ColumnDefinition> ColumnDefinitions
        {
            get
            {
                return this.InternalLayoutEngine.ColumnDefinitions;
            }
        }

        /// <summary>
        /// Gets the cell definitions.
        /// </summary>
        public ICollection<CellDefinition> CellDefinitions
        {
            get
            {
                return this.InternalLayoutEngine.CellDefinitions;
            }
        }

        #region Attached Getters

        /// <summary>
        /// Gets the grid row of a control.
        /// </summary>
        /// <param name="target">The control to get the grid row.</param>
        /// <returns>Returns the grid row, or -1 if unset.</returns>
        public static int GetRow(Control target)
        {
            CellDefinition cellDef;

            if (cellDefinitions.TryGetValue(target, out cellDef))
            {
                return cellDef.Row;
            }

            return -1;
        }

        /// <summary>
        /// Gets the grid row span of a control.
        /// </summary>
        /// <param name="target">The control to get the grid row span.</param>
        /// <returns>Returns the grid row span, or -1 if unset.</returns>
        public static int GetRowSpan(Control target)
        {
            CellDefinition cellDef;

            if (cellDefinitions.TryGetValue(target, out cellDef))
            {
                return cellDef.RowSpan;
            }

            return -1;
        }

        /// <summary>
        /// Gets the grid column of a control.
        /// </summary>
        /// <param name="target">The control to get the grid column.</param>
        /// <returns>Returns the grid column, or -1 if unset.</returns>
        public static int GetColumn(Control target)
        {
            CellDefinition cellDef;

            if (cellDefinitions.TryGetValue(target, out cellDef))
            {
                return cellDef.Column;
            }

            return -1;
        }

        /// <summary>
        /// Gets the grid column span of a control.
        /// </summary>
        /// <param name="target">The control to get the grid column span.</param>
        /// <returns>Returns the grid column span, or -1 if unset.</returns>
        public static int GetColumnSpan(Control target)
        {
            CellDefinition cellDef;

            if (cellDefinitions.TryGetValue(target, out cellDef))
            {
                return cellDef.ColumnSpan;
            }

            return -1;
        }

        #endregion // Attached Getters

        #region Attached Setters

        /// <summary>
        /// Sets the grid row of a control.
        /// </summary>
        /// <param name="target">The control to set the grid row.</param>
        /// <param name="value">The grid row to set.</param>
        public static void SetRow(Control target, int value)
        {
            var cellDef = GetCellDefinition(target);
            if (cellDef == null)
            {
                // should throw an exception ?
                return;
            }

            cellDef.Row = value;
        }

        /// <summary>
        /// Sets the grid row span of a control.
        /// </summary>
        /// <param name="target">The control to set the grid row span.</param>
        /// <param name="value">The grid row span to set.</param>
        public static void SetRowSpan(Control target, int value)
        {
            var cellDef = GetCellDefinition(target);
            if (cellDef == null)
            {
                // should throw an exception ?
                return;
            }

            cellDef.RowSpan = value;
        }

        /// <summary>
        /// Sets the grid column of a control.
        /// </summary>
        /// <param name="target">The control to set the grid column.</param>
        /// <param name="value">The grid column to set.</param>
        public static void SetColumn(Control target, int value)
        {
            var cellDef = GetCellDefinition(target);
            if (cellDef == null)
            {
                // should throw an exception ?
                return;
            }

            cellDef.Column = value;
        }

        /// <summary>
        /// Sets the grid column span of a control.
        /// </summary>
        /// <param name="target">The control to set the grid column span.</param>
        /// <param name="value">The grid column span to set.</param>
        public static void SetColumnSpan(Control target, int value)
        {
            var cellDef = GetCellDefinition(target);
            if (cellDef == null)
            {
                // should throw an exception ?
                return;
            }

            cellDef.ColumnSpan = value;
        }

        #endregion // Attached Setters

        /// <summary>
        /// Called when a layout process is performed.
        /// It populates the layout engine with cell definitions.
        /// </summary>
        /// <param name="levent">Event argument.</param>
        protected override void OnLayout(LayoutEventArgs levent)
        {
            base.OnLayout(levent);

            foreach (var child in this.Controls.OfType<Control>())
            {
                CellDefinition cellDef;
                if (cellDefinitions.TryGetValue(child, out cellDef))
                {
                    cellDefinitions.Remove(child);
                    this.CellDefinitions.Add(cellDef);
                }
            }
        }

        /// <summary>
        /// Produces a GridLayoutEngine instance.
        /// </summary>
        /// <returns>Returns a GridLayoutEngine instance.</returns>
        protected override GridLayoutEngine ProduceLayoutEngine()
        {
            return new GridLayoutEngine();
        }

        /// <summary>
        /// Gets an existing CellDefinition or instance a new one.
        /// </summary>
        /// <param name="target">The target control related to the cell definition.</param>
        /// <returns>Returns a CellDefinition instance.</returns>
        private static CellDefinition GetCellDefinition(Control target)
        {
            CellDefinition cellDef;

            if (cellDefinitions.TryGetValue(target, out cellDef) == false)
            {
                cellDef = new CellDefinition(target);
                cellDefinitions.Add(target, cellDef);
            }

            return cellDef;
        }
    }
}
