﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Win32;

using EffectMaker.UIControls.EventArguments;

namespace EffectMaker.UIControls.Input
{
    /// <summary>
    /// A message filter class to handle keyboard events globally.
    /// </summary>
    public class GlobalKeyEventHandler : IMessageFilter
    {
        /// <summary>A hash set that contains all the keys that is currently held down.</summary>
        private static HashSet<Keys> pressedKeys = new HashSet<Keys>();

        /// <summary>The singleton instance.</summary>
        private static GlobalKeyEventHandler instance = null;

        /// <summary>
        /// ショートカットテーブル
        /// </summary>
        private Dictionary<Keys, Action> shortCuts = new Dictionary<Keys, Action>();

        /// <summary>
        /// Constructor.
        /// </summary>
        private GlobalKeyEventHandler()
        {
        }

        /// <summary>
        /// Pre-filter key event dispatched when a key is pressed down.
        /// </summary>
        public static event EventHandler<PreFilterKeyEventArgs> PreFilterKeyDown = null;

        /// <summary>
        /// Pre-filter key event dispatched when a key is released.
        /// </summary>
        public static event EventHandler<PreFilterKeyEventArgs> PreFilterKeyUp = null;

        /// <summary>
        /// Pre-filter mouse event dispatched when a whell is moving.
        /// </summary>
        public static event EventHandler<PreFilterMouseEventArgs> PreFilterMouseWheel = null;

        /// <summary>
        /// Get the singleton instance.
        /// </summary>
        public static GlobalKeyEventHandler Instance
        {
            get
            {
                if (GlobalKeyEventHandler.instance == null)
                {
                    GlobalKeyEventHandler.instance = new GlobalKeyEventHandler();
                }

                return GlobalKeyEventHandler.instance;
            }
        }

        /// <summary>
        /// Get the number of currently pressed keys.
        /// </summary>
        public static int PressedKeyCount
        {
            get { return GlobalKeyEventHandler.pressedKeys.Count; }
        }

        /// <summary>
        /// Check if the specified key is currently held down.
        /// </summary>
        /// <param name="key">The key.</param>
        /// <returns>True if the key is pressed down.</returns>
        public static bool IsKeyPressed(Keys key)
        {
            return GlobalKeyEventHandler.pressedKeys.Contains(key);
        }

        /// <summary>
        /// Filters out a message before it is dispatched.
        /// </summary>
        /// <param name="m">The message to be dispatched.</param>
        /// <returns>
        /// True to filter the message and stop it from being dispatched;
        /// false to allow the message to continue to the next filter or control.
        /// </returns>
        public bool PreFilterMessage(ref Message m)
        {
            if (m.Msg == (int)WM.WM_KEYDOWN)
            {
                var key = (Keys)m.WParam;
                GlobalKeyEventHandler.pressedKeys.Add(key);

                // Trigger the event.
                if (GlobalKeyEventHandler.PreFilterKeyDown != null)
                {
                    var args = new PreFilterKeyEventArgs(key);
                    GlobalKeyEventHandler.PreFilterKeyDown(null, args);

                    if (args.Filter == true)
                    {
                        return true;
                    }
                }
            }
            else if (m.Msg == (int)WM.WM_KEYUP)
            {
                var key = (Keys)m.WParam;
                GlobalKeyEventHandler.pressedKeys.Remove(key);

                // Trigger the event.
                if (GlobalKeyEventHandler.PreFilterKeyUp != null)
                {
                    var args = new PreFilterKeyEventArgs(key);
                    GlobalKeyEventHandler.PreFilterKeyUp(null, args);

                    if (args.Filter == true)
                    {
                        return true;
                    }
                }
            }
            else if (m.Msg == (int)WM.WM_MOUSEWHEEL)
            {
                int d = Utility.HIWORD(m.WParam);
                d = d < 32768 ? 120 : -120;

                var p = new Point(Utility.LOWORD(m.LParam), Utility.HIWORD(m.LParam));

                // Trigger the event.
                if (GlobalKeyEventHandler.PreFilterMouseWheel != null)
                {
                    var args = new PreFilterMouseEventArgs(MouseButtons.None, 0, p.X, p.Y, d);
                    GlobalKeyEventHandler.PreFilterMouseWheel(null, args);

                    if (args.Filter == true)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// ショートカットのバイパスを設定します。
        /// </summary>
        /// <param name="keys">キーバインド</param>
        /// <param name="action">アクション</param>
        public void PushShortcutAction(Keys keys, Action action)
        {
            if (!this.shortCuts.ContainsKey(keys))
            {
                this.shortCuts.Add(keys, action);
            }
        }

        /// <summary>
        /// ショートカットの処理を呼び出します。
        /// </summary>
        /// <param name="keys">キーバインド</param>
        public void ProcessShortcut(Keys keys)
        {
            if (this.shortCuts.ContainsKey(keys))
            {
                this.shortCuts[keys]();
            }
        }
    }
}
