﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.Extensions
{
    /// <summary>
    /// Class containing extension methods related to layout.
    /// </summary>
    public static class LayoutExtensions
    {
        /// <summary>
        /// Gets the full display size of a layout element.
        /// The margin is taken into account.
        /// </summary>
        /// <param name="element">The layout element to get the display size.</param>
        /// <param name="availableSpace">The available space the element can use.</param>
        /// <returns>Returns the display size used by the layout element.</returns>
        public static Size GetElementDisplaySize(this ILayoutElement element, Size availableSpace)
        {
            return element.GetElementDisplaySize(availableSpace, true);
        }

        /// <summary>
        /// Gets the full display size of a layout element.
        /// </summary>
        /// <param name="element">The layout element to get the display size.</param>
        /// <param name="availableSpace">The available space the element can use.</param>
        /// <param name="applyMargin">Tells whether the margin should be
        /// taken into account or not.</param>
        /// <returns>Returns the display size used by the layout element.</returns>
        public static Size GetElementDisplaySize(
            this ILayoutElement element,
            Size availableSpace,
            bool applyMargin)
        {
            if (element.Visibility == Visibility.Collapsed)
            {
                if (element.Size.Width > 0 && element.Size.Height > 0)
                {
                    element.OriginalSize = element.Size;
                }

                return Size.Empty;
            }
            else if (element is Control &&
                ((Control)element).IsSelfOrParentCollapsed() == true)
            {
                if (element.Size.Width > 0 && element.Size.Height > 0)
                {
                    element.OriginalSize = element.Size;
                }

                return Size.Empty;
            }

            if (element.AutoSize)
            {
                Size size = element.GetPreferredSize(availableSpace);
                return new Size(
                    size.Width + (applyMargin ? element.Margin.Horizontal : 0),
                    size.Height + (applyMargin ? element.Margin.Vertical : 0));
            }
            else
            {
                Size size = element.Size;
                if (size.Width <= 0 || size.Height <= 0)
                {
                    size = element.OriginalSize;
                }

                return new Size(
                    size.Width + (applyMargin ? element.Margin.Horizontal : 0),
                    size.Height + (applyMargin ? element.Margin.Vertical : 0));
            }
        }

        /// <summary>
        /// Set the full display size of a layout element.
        /// </summary>
        /// <param name="element">The layout element to set the display size.</param>
        /// <param name="elementSize">The display size.</param>
        public static void SetElementDisplaySize(this ILayoutElement element, Size elementSize)
        {
            element.SetElementDisplaySize(elementSize, true);
        }

        /// <summary>
        /// Set the full display size of a layout element.
        /// The margin is taken into account.
        /// </summary>
        /// <param name="element">The layout element to set the display size.</param>
        /// <param name="elementSize">The display size.</param>
        /// <param name="applyMargin">Tells whether the margin should be
        /// taken into account or not.</param>
        public static void SetElementDisplaySize(this ILayoutElement element, Size elementSize, bool applyMargin)
        {
            // Sizeプロパティでまとめてセットするように変更
            var width = elementSize.Width - (applyMargin ? element.Margin.Horizontal : 0);
            var height = elementSize.Height - (applyMargin ? element.Margin.Vertical : 0);
            element.Size = new Size(width, height);
        }

        /// <summary>
        /// Arranges a control (location and size) inside a given bounding box,
        /// taking horizontal and vertical alignment into consideration.
        /// </summary>
        /// <param name="layoutElement">The layout element to arrange.</param>
        /// <param name="localBoundingBox">The bounding box in which the
        /// control is to be arranged.</param>
        public static void ArrangeLayoutElement(
            this ILayoutElement layoutElement,
            Rectangle localBoundingBox)
        {
            if (layoutElement == null)
            {
                throw new ArgumentNullException("layoutElement");
            }

            var controlPreferredSize = layoutElement.GetElementDisplaySize(localBoundingBox.Size);

            var left = localBoundingBox.X;
            var top = localBoundingBox.Y;
            var width = controlPreferredSize.Width;
            var height = controlPreferredSize.Height;

            var horizontalReady = false;
            var verticalReady = false;

            if (width > localBoundingBox.Width ||
                layoutElement.HorizontalAlignment == HAlignment.Stretch)
            {
                width = localBoundingBox.Width;
                horizontalReady = true;
            }

            if (height > localBoundingBox.Height ||
                layoutElement.VerticalAlignment == VAlignment.Stretch)
            {
                height = localBoundingBox.Height;
                verticalReady = true;
            }

            if (horizontalReady == false)
            {
                if (layoutElement.HorizontalAlignment == HAlignment.Right)
                {
                    left = localBoundingBox.X + localBoundingBox.Width - width;
                }
                else if (layoutElement.HorizontalAlignment == HAlignment.Center)
                {
                    left = localBoundingBox.X + ((localBoundingBox.Width - width) / 2);
                }
            }

            if (verticalReady == false)
            {
                if (layoutElement.VerticalAlignment == VAlignment.Bottom)
                {
                    top = localBoundingBox.Y + localBoundingBox.Height - height;
                }
                else if (layoutElement.VerticalAlignment == VAlignment.Center)
                {
                    top = localBoundingBox.Y + ((localBoundingBox.Height - height) / 2);
                }
            }

            // Boundsプロパティでまとめてセットするように変更
            var location = new Point(left + layoutElement.Margin.Left, top + layoutElement.Margin.Top);
            var size = new Size(width - layoutElement.Margin.Horizontal, height - layoutElement.Margin.Vertical);
            layoutElement.Bounds = new Rectangle(location, size);
        }
    }
}
