﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace EffectMaker.UIControls.Debug
{
    /// <summary>
    /// An extended TreeNode class that encapsulate Control.
    /// </summary>
    public class ControlTreeNode : TreeNode, IDisposable
    {
        /// <summary>
        /// Initializes the ControlTreeNode given a Control instance.
        /// </summary>
        /// <param name="control">The control that the node represent.</param>
        public ControlTreeNode(object control)
        {
            if (control == null)
            {
                throw new ArgumentNullException("control");
            }

            this.Control = control;

            var ctrl = this.Control as Control;

            if (ctrl != null)
            {
                ctrl.ControlAdded += this.OnControlControlAdded;
                ctrl.ControlRemoved += this.OnControlControlRemoved;
            }

            string name = this.GetControlName(control);

            if (string.IsNullOrWhiteSpace(name))
            {
                this.Text = control.GetType().Name;
            }
            else
            {
                this.Text = string.Format("{0} ({1})", name, control.GetType().Name);
            }

            this.CreateChildren();
        }

        /// <summary>
        /// Gets the related Control.
        /// </summary>
        public object Control { get; private set; }

        /// <summary>
        /// Disposes the ControlTreeNode instance.
        /// </summary>
        public void Dispose()
        {
            var ctrl = this.Control as Control;

            if (ctrl != null)
            {
                ctrl.ControlAdded -= this.OnControlControlAdded;
                ctrl.ControlRemoved -= this.OnControlControlRemoved;
            }
        }

        /// <summary>
        /// Called when a child control is added to the control.
        /// </summary>
        /// <param name="sender">Caller of the event.</param>
        /// <param name="e">Event argument.</param>
        private void OnControlControlAdded(object sender, ControlEventArgs e)
        {
            this.Nodes.Add(new ControlTreeNode(e.Control));
        }

        /// <summary>
        /// Called when a child control is removed from the control.
        /// </summary>
        /// <param name="sender">Caller of the event.</param>
        /// <param name="e">Event argument.</param>
        private void OnControlControlRemoved(object sender, ControlEventArgs e)
        {
            var found = this.Nodes
                .OfType<ControlTreeNode>()
                .FirstOrDefault(ctn => ctn.Control == e.Control);

            if (found != null)
            {
                this.Nodes.Remove(found);
            }
        }

        /// <summary>
        /// Creates the child node controls.
        /// </summary>
        private void CreateChildren()
        {
            var subnodes = this.GetControlChildren(this.Control)
                .Cast<object>()
                .Where(c => c != null);

            foreach (var subnode in subnodes)
            {
                this.Nodes.Add(new ControlTreeNode(subnode));
            }
        }

        /// <summary>
        /// Gets the name of the control.
        /// </summary>
        /// <param name="control">The control to get the name.</param>
        /// <returns>Returns the name of the control.</returns>
        private string GetControlName(object control)
        {
            if (control is Control)
            {
                return ((Control)control).Name;
            }
            else if (control is TreeNode)
            {
                return ((TreeNode)control).Name;
            }
            else
            {
                return control.ToString();
            }
        }

        /// <summary>
        /// Gets the sequence of child elements.
        /// </summary>
        /// <param name="control">The control to retrieve its children.</param>
        /// <returns>Returns the child elements of the control.</returns>
        private IEnumerable GetControlChildren(object control)
        {
            if (control is TreeView)
            {
                return ((TreeView)control).Nodes;
            }
            else if (control is Control)
            {
                return ((Control)control).Controls;
            }
            else if (control is TreeNode)
            {
                return ((TreeNode)control).Nodes;
            }
            else
            {
                return Enumerable.Empty<object>();
            }
        }
    }
}
