﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using EffectMaker.Foundation.Interfaces;

using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls
{
    /// <summary>
    /// Wraps a ControlCollection, and perform specific processing related to IControl instance.
    /// </summary>
    public class ControlCollectionWrapper : IIndexableCollection<ILogicalTreeElement>, IDisposable
    {
        /// <summary>
        /// Stores the control on which the ControlCollection is wrapped.
        /// </summary>
        private Control control;

        /// <summary>
        /// Stores the number of IControl instances.
        /// </summary>
        private int count;

        /// <summary>
        /// Initializes the ControlCollectionWrapper instance.
        /// </summary>
        /// <param name="control">The control to wrap the ControlCollection.</param>
        public ControlCollectionWrapper(Control control)
        {
            if (control == null)
            {
                throw new ArgumentNullException("control");
            }

            this.control = control;
            this.count = control.Controls.OfType<ILogicalTreeElement>().Count();

            this.control.ControlAdded += this.OnControlAdded;
            this.control.ControlRemoved += this.OnControlRemoved;
        }

        /// <summary>
        /// Gets the number of IControl instances in the collection.
        /// </summary>
        public int Count
        {
            get { return this.count; }
        }

        /// <summary>
        /// Gets whether or not the collection is readonly.
        /// </summary>
        public bool IsReadOnly
        {
            get { return this.control.Controls.IsReadOnly; }
        }

        /// <summary>
        /// Gets a control from it's index in the children list.
        /// </summary>
        /// <param name="index">The index of the control to retrieve.</param>
        /// <returns>Returns the control at the given index.</returns>
        public ILogicalTreeElement this[int index]
        {
            get
            {
                return this.control.Controls[index] as ILogicalTreeElement;
            }
        }

        /// <summary>
        /// Disposes the ControlCollectionWrapper instance.
        /// </summary>
        public void Dispose()
        {
            this.control.ControlAdded -= this.OnControlAdded;
            this.control.ControlRemoved -= this.OnControlRemoved;
        }

        /// <summary>
        /// Add a control.
        /// </summary>
        /// <param name="item">Control to add.</param>
        public void Add(ILogicalTreeElement item)
        {
            var ctrl = item as Control;

            if (ctrl == null)
            {
                throw new ArgumentException("Can add only object that inherit from Control.");
            }

            this.control.Controls.Add(ctrl);
        }

        /// <summary>
        /// Clears the control list.
        /// </summary>
        public void Clear()
        {
            LayoutEngineBase.SuspendLayout();
            this.control.Controls.Clear();
            LayoutEngineBase.ResumeLayout();
        }

        /// <summary>
        /// Checks whether the control exists or not.
        /// </summary>
        /// <param name="item">The control to check.</param>
        /// <returns>Returns true if the control exists, false otherwise.</returns>
        public bool Contains(ILogicalTreeElement item)
        {
            var ctrl = item as Control;

            if (ctrl == null)
            {
                return false;
            }

            return this.control.Controls.Contains(ctrl);
        }

        /// <summary>
        /// Not implemented
        /// </summary>
        /// <param name="array">The source array of controls.</param>
        /// <param name="arrayIndex">The copy start index.</param>
        public void CopyTo(ILogicalTreeElement[] array, int arrayIndex)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Removes a control.
        /// </summary>
        /// <param name="item">The control to remove.</param>
        /// <returns>Returns true if the control is removed, false otherwise.</returns>
        public bool Remove(ILogicalTreeElement item)
        {
            var ctrl = item as Control;

            if (ctrl == null)
            {
                return false;
            }

            LayoutEngineBase.SuspendLayout();
            this.control.Controls.Remove(ctrl);
            LayoutEngineBase.ResumeLayout();

            return true;
        }

        /// <summary>
        /// Enumerator of an IControl sequence.
        /// </summary>
        /// <returns>Returns an IControl sequence enumerator.</returns>
        public IEnumerator<ILogicalTreeElement> GetEnumerator()
        {
            return this.control.Controls
                .OfType<ILogicalTreeElement>()
                .Concat(new ILogicalTreeElement[]
                {
                    this.control.ContextMenuStrip as ILogicalTreeElement,
                    ((IControl)this.control).Behaviors,
                })
                .GetEnumerator();
        }

        /// <summary>
        /// Enumerator of an IControl sequence.
        /// </summary>
        /// <returns>Returns an IControl sequence enumerator.</returns>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return (System.Collections.IEnumerator)this.GetEnumerator();
        }

        /// <summary>
        /// Called when a control is added to the underlying ControlCollection of the control.
        /// </summary>
        /// <param name="sender">The event caller.</param>
        /// <param name="e">The event argument.</param>
        private void OnControlAdded(object sender, ControlEventArgs e)
        {
            if (e.Control is ILogicalTreeElement)
            {
                this.count++;
            }
        }

        /// <summary>
        /// Called when a control is removed from the underlying ControlCollection of the control.
        /// </summary>
        /// <param name="sender">The event caller.</param>
        /// <param name="e">The event argument.</param>
        private void OnControlRemoved(object sender, ControlEventArgs e)
        {
            if (e.Control is ILogicalTreeElement)
            {
                this.count--;
            }
        }
    }
}
