﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Windows.Forms;
using System.Windows.Resources;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Input;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended Button class.
    /// </summary>
    public class UIImageButton : UIButton
    {
        /// <summary>
        /// トグル状態
        /// </summary>
        private bool toggleState = false;

        /// <summary>
        /// ボタン用の画像.
        /// </summary>
        private Image normalImage = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public UIImageButton()
        {
            this.AutoSize = false;
            this.BackColor = Color.Transparent;
            this.FlatAppearance.BorderSize = 0;
            this.FlatAppearance.BorderColor = Color.FromArgb(0, 255, 255, 255);
            this.FlatAppearance.CheckedBackColor = Color.Transparent;
            this.FlatAppearance.MouseDownBackColor = Color.Transparent;
            this.FlatAppearance.MouseOverBackColor = Color.Transparent;
            this.FlatStyle = FlatStyle.Flat;

            this.MouseDown += (s, e) =>
            {
                if (e.Button != MouseButtons.Left)
                {
                    return;
                }

                if (this.ToggleType)
                {
                    if (!Checked)
                    {
                        if (this.DownImage != null)
                        {
                            this.Image = this.DownImage;
                        }
                    }
                    else
                    {
                        this.Image = this.NormalImage;
                    }
                }
                else
                {
                    if (this.DownImage != null)
                    {
                        this.Image = this.DownImage;
                    }
                }
            };
            this.MouseUp += (s, e) =>
            {
                if (e.Button != MouseButtons.Left)
                {
                    return;
                }

                if (!this.ToggleType)
                {
                    if (this.DownImage != null)
                    {
                        this.Image = this.NormalImage;
                    }
                }
            };
            this.EnabledChanged += (s, e) =>
            {
                if (this.DisableImage != null)
                {
                    this.Image = this.Enabled ? this.NormalImage : this.DisableImage;
                }
            };
            this.MouseEnter += (s, e) =>
            {
                if (this.Checked)
                {
                    if (this.DownMouseOverImage != null)
                    {
                        this.Image = this.DownMouseOverImage;
                    }
                }
                else
                {
                    if (this.MouseOverImage != null)
                    {
                        this.Image = this.MouseOverImage;
                    }
                }
            };
            this.MouseLeave += (s, e) =>
            {
                if (this.Checked)
                {
                    if (this.DownImage != null)
                    {
                        this.Image = this.DownImage;
                    }
                }
                else
                {
                    if (this.NormalImage != null)
                    {
                        this.Image = this.NormalImage;
                    }
                }
            };
        }

        /// <summary>
        /// ノーマルイメージ.
        /// </summary>
        public Image NormalImage
        {
            get
            {
                return this.normalImage;
            }

            set
            {
                this.normalImage = value;
                if (value != null)
                {
                    var imgSize = value.Size;
                    imgSize.Width += 6;
                    imgSize.Height += 1;
                    this.Size = imgSize;
                    this.MinimumSize = this.Size;
                }

                if (!this.Enabled && this.DisableImage != null)
                {
                    this.Image = this.DisableImage;
                }
                else
                {
                    this.Image = value;
                }
            }
        }

        /// <summary>
        /// 押し下げ時イメージ.
        /// </summary>
        public Image DownImage { get; set; }

        /// <summary>
        /// マウスオーバーイメージ.
        /// </summary>
        public Image MouseOverImage { get; set; }

        /// <summary>
        /// 押し下げ時マウスオーバーイメージ.
        /// </summary>
        public Image DownMouseOverImage { get; set; }

        /// <summary>
        /// 無効化時イメージ(グレーアウト).
        /// </summary>
        public Image DisableImage { get; set; }

        /// <summary>
        /// ボタンタイプ(true:チェックボックス/false:通常ボタン).
        /// </summary>
        public bool ToggleType { get; set; }

        /// <summary>
        /// チェックボックスタイプ時のチェック状態(通常ボタンなら常時false)
        /// </summary>
        public bool Checked
        {
            get
            {
                if (!this.ToggleType)
                {
                    return false;
                }

                return this.toggleState;
            }

            set
            {
                this.toggleState = value;
                this.Image = value ? this.DownImage : this.NormalImage;
            }
        }

        /// <summary>
        /// フォーカス枠の表示を無効化するオーバーライド
        /// </summary>
        protected override bool ShowFocusCues
        {
            get { return false; }
        }

        /// <summary>
        /// 非アクティブなコントロールの上にツールチップを乗せます.
        /// </summary>
        /// <param name="ctl">ツールチップを乗せたいコントロール</param>
        /// <param name="toolTip">ツールチップ</param>
        /// <param name="str">乗せるテキスト</param>
        /// <param name="x">表示位置X</param>
        /// <param name="y">表示位置Y</param>
        /// <param name="d">表示時間</param>
        protected static void PutToolTipForInactive(Control ctl, ToolTip toolTip, string str, int x = 20, int y = -25, int d = 2000)
        {
            // カーソルが親コントロール上で動いているときの処理
            Action<object, MouseEventArgs> mouseMoveToolTipHanlder = (s, e) =>
            {
                if (ctl.Enabled || !ctl.Visible)
                {
                    return;
                }

                var contains = ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position));

                if (toolTip.Tag != ctl && contains)
                {
                    toolTip.Show(str, ctl, x, y, d);
                    toolTip.Tag = ctl;
                }
                else if (!contains)
                {
                    toolTip.Hide(ctl);
                    toolTip.Tag = null;
                }
            };
            ctl.Parent.MouseMove += new MouseEventHandler(mouseMoveToolTipHanlder);

            // カーソルが親コントロールから離れた時の処理
            Action<object, EventArgs> mouseLeaveToolTipHandler = (s, e) =>
            {
                if (ctl.Enabled || !ctl.Visible)
                {
                    return;
                }

                toolTip.Hide(ctl);
                toolTip.Tag = null;
            };
            ctl.Parent.MouseLeave += new EventHandler(mouseLeaveToolTipHandler);
        }

        /// <summary>
        /// Called when the button is clicked.
        /// </summary>
        /// <param name="e">The event argument.</param>
        protected override void OnClick(EventArgs e)
        {
            if (this.ToggleType)
            {
                this.toggleState = !this.toggleState;
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged,
                    "Checked");
            }

            base.OnClick(e);
        }
    }
}
