﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// An extended Control class.
    /// </summary>
    public class UIComboBox : ComboBox, IControl, ILayoutElement
    {
        /// <summary>
        /// Win32APIのGW_CHILD
        /// </summary>
        private const int GwChild = 5;

        /// <summary>
        /// アイテム一覧です。
        /// </summary>
        private IEnumerable<KeyValuePair<string, object>> availableItems;

        /// <summary>
        /// Backing field for the Extender property.
        /// </summary>
        private LogicalTreeElementExtender controlExtender;

        /// <summary>
        /// ILayoutElement extender.
        /// </summary>
        private LayoutElementExtender layoutElementExtender;

        /// <summary>
        /// Backing field for Controls property.
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>
        /// Backing field for the Resources property.
        /// </summary>
        private IDictionary<string, object> resources = new Dictionary<string, object>();

        /// <summary>
        /// サイズを自動調整するフラグです。
        /// </summary>
        private bool autoAdjustWidth;

        /// <summary>
        /// ドロップダウンリストのサイズを自動調整するフラグです。
        /// </summary>
        private bool autoAdjustDropDownWidth;

        /// <summary>
        /// 外部からのプロパティ変更時に設定サイズを保持する変数です。
        /// </summary>
        private int savedWidth;

        /// <summary>
        /// 選択状態が先行して通知された時に保持する変数です。
        /// </summary>
        private object savedSelectedItem = null;

        /// <summary>
        /// The suppress selected index changed event.
        /// </summary>
        private bool suppressSelectedIndexChangedEvent = false;

        /// <summary>
        /// Editウィンドウです。
        /// </summary>
        private EditWindow editWindow;

        /// <summary>
        /// テキストが空のときに表示するコメントです。
        /// </summary>
        private string promptText;

        /// <summary>
        /// ドロップダウンリストの幅を設定可能かどうか。
        /// </summary>
        private bool enableSetDropDownWidth = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UIComboBox()
        {
            this.controlExtender = new LogicalTreeElementExtender(this);
            this.layoutElementExtender = new LayoutElementExtender(this);
            this.autoAdjustWidth = true;
            this.autoAdjustDropDownWidth = false;
            this.savedWidth = 0;
            this.Bindings = new BindingContainer(this);
            this.Behaviors = new BehaviorCollection(this);
            this.DrawMode = DrawMode.OwnerDrawFixed;
            this.DrawItem += this.OnDrawItem;

            this.TabStop = false;

            this.SelectedIndexChanged += (s, e) =>
            {
                ILogicalTreeElement ctl = this;
                while (ctl.Parent != null)
                {
                    ctl = ctl.Parent;
                    if (ctl is UIPanel)
                    {
                        ((UIPanel)ctl).FocusForWheel(true);
                    }
                }
            };

            this.SizeChanged += (s, e) =>
            {
                this.AdjustDropDownWidth();
            };
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// VisibilityがCollapsedになった時に元のサイズを覚えておくためのプロパティ
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Size OriginalSize { get; set; }

        /// <summary>
        /// Gets the resources.
        /// </summary>
        public IDictionary<string, object> Resources
        {
            get { return this.resources; }
        }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return base.Parent as ILogicalTreeElement; }
        }

        /// <summary>
        /// Gets the collection of child controls.
        /// </summary>
        public new IIndexableCollection<ILogicalTreeElement> Controls
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new ControlCollectionWrapper(this);
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the collection of logical tree elements.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                return this.Controls;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.controlExtender; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Gets the behaviors collection.
        /// </summary>
        public BehaviorCollection Behaviors { get; private set; }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See ControlExtender for more information.
        /// <see cref="ControlExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.controlExtender.DataContext; }
            set { this.controlExtender.DataContext = value; }
        }

        #region Layout

        /// <summary>
        /// Gets the layout element extender instance of this control.
        /// </summary>
        public LayoutElementExtender LayoutElementExtender
        {
            get { return this.layoutElementExtender; }
        }

        /// <summary>
        /// Gets or sets the visibility.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public Visibility Visibility
        {
            get { return this.layoutElementExtender.Visibility; }
            set { this.layoutElementExtender.Visibility = value; }
        }

        /// <summary>
        /// Gets or sets the horizontal alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public HAlignment HorizontalAlignment
        {
            get { return this.layoutElementExtender.HorizontalAlignment; }
            set { this.layoutElementExtender.HorizontalAlignment = value; }
        }

        /// <summary>
        /// Gets or sets the vertical alignment.
        /// See ControlExtender for more information.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public VAlignment VerticalAlignment
        {
            get { return this.layoutElementExtender.VerticalAlignment; }
            set { this.layoutElementExtender.VerticalAlignment = value; }
        }

        #endregion // Layout

        /// <summary>
        /// 選択中のアイテム
        /// </summary>
        public new object SelectedItem
        {
            get
            {
                if ((this.SelectedIndex >= 0) &&
                    (this.SelectedIndex < this.Items.Count))
                {
                    var item = this.Items[this.SelectedIndex] as ComboBoxItem;
                    if (item == null)
                    {
                        return null;
                    }

                    return item.Value;
                }
                else
                {
                    return null;
                }
            }

            set
            {
                int index = this.Items
                    .OfType<ComboBoxItem>()
                    .FindIndex(cbi => object.Equals(cbi.Value, value));

                if (index < 0 || index >= this.Items.Count)
                {
                    this.SelectedIndex = -1;
                    this.savedSelectedItem = value;
                    return;
                }

                if (this.SelectedIndex != index)
                {
                    this.SelectedIndex = index;
                    this.savedSelectedItem = value;

                    if (this.suppressSelectedIndexChangedEvent)
                    {
                        return;
                    }

                    this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedIndex");
                    this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedItem");
                }
            }
        }

        /// <summary>
        /// アイテム一覧
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AvailableItems
        {
            get
            {
                return this.availableItems;
            }

            set
            {
                this.suppressSelectedIndexChangedEvent = true;

                IEnumerable oldValue = this.availableItems;

                if (this.controlExtender.SetValue(ref this.availableItems, value))
                {
                    // 古いアイテムにイベントを登録している場合、解除する。
                    if (oldValue != null && oldValue is INotifyCollectionChanged)
                    {
                        ((INotifyCollectionChanged)oldValue).CollectionChanged -=
                            this.OnCollectionChanged;
                    }

                    // INotifyPropertyChangedを継承しているインスタンスの場合、
                    // イベントを登録する。
                    if (value is INotifyCollectionChanged)
                    {
                        ((INotifyCollectionChanged)value).CollectionChanged +=
                            this.OnCollectionChanged;
                    }

                    // アイテムの変更を開始
                    this.BeginUpdate();

                    // アイテムをクリアする。
                    Items.Clear();

                    // ComboBoxItemを登録する。
                    foreach (KeyValuePair<string, object> item in this.availableItems)
                    {
                        Items.Add(new ComboBoxItem(item.Key, item.Value));

                        if (this.savedSelectedItem != null && object.Equals(item.Value, this.savedSelectedItem))
                        {
                            this.SelectedItem = this.savedSelectedItem;
                        }
                    }

                    // アイテムの変更を終了
                    this.EndUpdate();

                    // Items数が正の整数→0に変化した場合、
                    // DropDownMenuの長さが変わらずに伸びたままになってしまうための対処.
                    // このif文以外にも、xamlなどでDropDownHeightプロパティに何かしら値を代入しておく必要がある.
                    if (Items.Count == 0)
                    {
                        // Items数が0の場合は、コントロールの自動高さ調整をOffにする.
                        this.IntegralHeight = false;
                    }
                    else
                    {
                        this.IntegralHeight = true;
                    }

                    // サイズの自動調整を実行する。
                    this.AdjustWidth();
                    this.AdjustDropDownWidth();
                }

                this.suppressSelectedIndexChangedEvent = false;
            }
        }

        /// <summary>
        /// アイテムの文字色一覧を取得または設定します。
        /// </summary>
        public Color[] ItemColors { get; set; }

        /// <summary>
        /// サイズ自動調整のOn/Offを取得または設定します。
        /// </summary>
        public bool AutoAdjustWidth
        {
            get
            {
                return this.autoAdjustWidth;
            }

            set
            {
                this.autoAdjustWidth = value;
                this.AdjustWidth();
            }
        }

        /// <summary>
        /// ドロップダウンリストのサイズ自動調整のOn/Offを取得または設定します。
        /// </summary>
        public bool AutoAdjustDropDownWidth
        {
            get
            {
                return this.autoAdjustDropDownWidth;
            }

            set
            {
                this.autoAdjustDropDownWidth = value;
                this.AdjustDropDownWidth();
            }
        }

        /// <summary>
        /// サイズプロパティ
        /// </summary>
        public new int Width
        {
            get
            {
                return base.Width;
            }

            set
            {
                base.Width = value;
                this.savedWidth = value;
            }
        }

        /// <summary>
        /// テキストが空のときに表示するコメントを取得または設定します。
        /// </summary>
        public string PromptText
        {
            get
            {
                return this.promptText;
            }

            set
            {
                if (this.promptText == null)
                {
                    IntPtr hwnd = GetWindow(this.Handle, GwChild);

                    this.editWindow = new EditWindow(this);
                    this.editWindow.AssignHandle(hwnd);
                }

                this.promptText = value;
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See ControlExtender for more details.
        /// <see cref="ControlExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.controlExtender.ClearDataContext();
        }

        /// <summary>
        /// Find the index of the combo box item with the specified name and value.
        /// </summary>
        /// <param name="item">A key value pair with the name and value to find.</param>
        /// <returns>The index of the combo box item or -1 if the item is not found.</returns>
        public int IndexOfComboBoxItem(KeyValuePair<string, object> item)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var comboBoxItem = this.Items[i] as ComboBoxItem;
                if ((comboBoxItem != null) &&
                    (comboBoxItem.Name == item.Key) &&
                    (comboBoxItem.Value.Equals(item.Value) == true))
                {
                    // The item is found, return the index.
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the combo box item with the specified name and value.
        /// </summary>
        /// <param name="item">A key value pair with the name and value to find.</param>
        /// <returns>The combo box item or null if the item is not found.</returns>
        public ComboBoxItem FindComboBoxItem(KeyValuePair<string, object> item)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var comboBoxItem = this.Items[i] as ComboBoxItem;
                if ((comboBoxItem != null) &&
                    (comboBoxItem.Name == item.Key) &&
                    (comboBoxItem.Value.Equals(item.Value) == true))
                {
                    // The item is found, return the index.
                    return comboBoxItem;
                }
            }

            return null;
        }

        /// <summary>
        /// Find the index of the combo box item with the specified value.
        /// </summary>
        /// <param name="value">The value to find.</param>
        /// <returns>The index of the combo box item or -1 if the item is not found.</returns>
        public int IndexOfComboBoxItemByValue(object value)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var comboBoxItem = this.Items[i] as ComboBoxItem;
                if ((comboBoxItem != null) &&
                    (comboBoxItem.Value.Equals(value) == true))
                {
                    // The item is found, return the index.
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the combo box item with the specified value.
        /// </summary>
        /// <param name="value">The value to find.</param>
        /// <returns>The combo box item or null if the item is not found.</returns>
        public ComboBoxItem FindComboBoxItemByValue(object value)
        {
            for (int i = 0; i < this.Items.Count; ++i)
            {
                var comboBoxItem = this.Items[i] as ComboBoxItem;
                if ((comboBoxItem != null) &&
                    (comboBoxItem.Value.Equals(value) == true))
                {
                    // The item is found, return the index.
                    return comboBoxItem;
                }
            }

            return null;
        }

        /// <summary>
        /// Called when the user changes the selected item from the UI.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnSelectedIndexChanged(EventArgs e)
        {
            base.OnSelectedIndexChanged(e);

            if (this.suppressSelectedIndexChangedEvent)
            {
                return;
            }

            this.savedSelectedItem = this.SelectedItem;

            this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedIndex");
            this.controlExtender.NotifyPropertyChanged(propertyName: "SelectedItem");
        }

        /// <summary>
        /// コンボボックスの幅を自動調整します。
        /// </summary>
        protected void AdjustWidth()
        {
            if (this.Items.Count <= 0)
            {
                return;
            }

            if (this.AutoAdjustWidth == false)
            {
                return;
            }

            // コンボボックスの幅をアイテムのサイズに合わせる
            using (System.Drawing.Graphics g = this.CreateGraphics())
            {
                int maxLength = 0;

                foreach (ComboBoxItem item in this.Items)
                {
                    maxLength = (int)Math.Max(maxLength, g.MeasureString(item.Name, this.Font).Width);
                }

                base.Width = maxLength + 30;
            }
        }

        /// <summary>
        /// コンボボックスのドロップダウンリストの幅を自動調整します。
        /// </summary>
        protected void AdjustDropDownWidth()
        {
            if (this.enableSetDropDownWidth == false)
            {
                return;
            }

            if (this.Items.Count <= 0)
            {
                return;
            }

            if (this.AutoAdjustDropDownWidth == false)
            {
                return;
            }

            // ドロップダウンリストの幅をアイテムのサイズに合わせる
            using (System.Drawing.Graphics g = this.CreateGraphics())
            {
                int maxLength = 0;

                foreach (ComboBoxItem item in this.Items)
                {
                    maxLength = (int)Math.Max(maxLength, g.MeasureString(item.Name, this.Font).Width);
                }

                this.DropDownWidth = Math.Max(this.ClientSize.Width, maxLength + 5);
            }
        }

        /// <summary>
        /// ホイール無効化
        /// </summary>
        /// <param name="m">
        /// The m.
        /// </param>
        protected override void WndProc(ref Message m)
        {
            // ホイールを無効化する
            if (m.Msg == (int)EffectMaker.Foundation.Win32.WM.WM_MOUSEWHEEL)
            {
                return;
            }

            base.WndProc(ref m);

            switch (m.Msg)
            {
                case (int)EffectMaker.Foundation.Win32.WM.WM_PAINT:
                {
                    // コントロール作成後、ここに来るまでにドロップダウンリストの幅を
                    // 設定すると値が正しく反映されなくなる。
                    // プレビューノードの接続モデルのコンボボックスの場合、
                    // LeftMargin="16", RightMargin="0", Width="142" (DropDownWidth は未指定)
                    // と指定されていて、レイアウト処理の結果は Width="126", DropDownWidth="142"
                    // と、Width だけマージンを引かれた値になる。（レイアウト処理の不具合の可能性あり）
                    if (enableSetDropDownWidth == false)
                    {
                        enableSetDropDownWidth = true;
                        AdjustDropDownWidth();
                    }

                    break;
                }
            }
        }

        /// <summary>
        /// Win32APIのGetWindow関数
        /// </summary>
        /// <param name="hwnd">元ウィンドウのハンドル</param>
        /// <param name="ucmd">関係</param>
        /// <returns>成功するとウィンドウのハンドルが返り、失敗するとNULLが返る.</returns>
        [DllImport("user32.dll")]
        static extern IntPtr GetWindow(IntPtr hwnd, uint ucmd);

        /// <summary>
        /// コンボボックスのアイテムを描画する時に呼び出されるイベントです.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnDrawItem(object sender, DrawItemEventArgs e)
        {
            if (e.Index < 0)
            {
                return;
            }

            var originalTextRenderHint = e.Graphics.TextRenderingHint;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;

            // 背景を描画
            if (this.Enabled == false)
            {
                DrawItemEventArgs disableArgs = new DrawItemEventArgs(
                    e.Graphics,
                    e.Font,
                    e.Bounds,
                    e.Index,
                    e.State,
                    e.ForeColor,
                    SystemColors.Control);

                disableArgs.DrawBackground();
            }
            else
            {
                e.DrawBackground();
            }

            // 文字色を取得
            Color strColor = e.ForeColor;
            if (this.ItemColors != null && this.ItemColors.Length > e.Index)
            {
                strColor = this.ItemColors[e.Index];
            }

            // 画像と文字列をペアにして突っ込むといい感じに描画してくれる
            if (this.Items[e.Index] is KeyValuePair<Image, string>)
            {
                var pairItem = (KeyValuePair<Image, string>)this.Items[e.Index];

                // 画像を描画
                var imageItem = (Image)pairItem.Key;
                float imageY = e.Bounds.Y + ((e.Bounds.Height - imageItem.Height) / 2);

                e.Graphics.DrawImage(imageItem, e.Bounds.X + 2, imageY);

                // 文字列を描画
                string text = pairItem.Value;
                using (Brush brush = new SolidBrush(strColor))
                {
                    Font font = this.Font;
                    float textY = e.Bounds.Y + ((e.Bounds.Height -
                                  e.Graphics.MeasureString(text, font).Height) / 2);

                    e.Graphics.DrawString(text, font, brush, e.Bounds.X + imageItem.Width + 6, textY);

                    // フォーカスを示す矩形を描画
                    e.DrawFocusRectangle();
                }
            }
            else
            {
                // 文字列を描画
                string text = this.Items[e.Index].ToString();
                using (Brush brush = new SolidBrush(strColor))
                {
                    Font font = this.Font;
                    float textY = e.Bounds.Y + ((e.Bounds.Height -
                                  e.Graphics.MeasureString(text, font).Height) / 2);

                    e.Graphics.DrawString(text, font, brush, e.Bounds.X, textY);

                    // フォーカスを示す矩形を描画
                    e.DrawFocusRectangle();
                }
            }

            e.Graphics.TextRenderingHint = originalTextRenderHint;
        }

        /// <summary>
        /// コレクションが変更された時に呼び出されるイベントです.
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            this.BeginUpdate();

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    {
                        // The items have to be KeyValuePairs.
                        var newItems = e.NewItems.Cast<KeyValuePair<string, object>>();
                        if (newItems == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Create the combo box items.
                        ComboBoxItem[] itemsToAdd =
                            newItems.Select(item => new ComboBoxItem(item.Key, item.Value)).ToArray();

                        // Add the items.
                        if (e.NewStartingIndex >= 0)
                        {
                            int index = e.NewStartingIndex;
                            foreach (ComboBoxItem item in itemsToAdd)
                            {
                                this.Items.Insert(index, item);
                                ++index;
                            }
                        }
                        else
                        {
                            this.Items.AddRange(itemsToAdd);
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Remove:
                    {
                        // The items have to be KeyValuePairs.
                        var items = e.OldItems.Cast<KeyValuePair<string, object>>();
                        if (items == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Remove the items.
                        foreach (KeyValuePair<string, object> item in items)
                        {
                            // Find the item from the combo box items and remove it.
                            int index = this.IndexOfComboBoxItem(item);
                            if ((index >= 0) && (index < this.Items.Count))
                            {
                                this.Items.RemoveAt(index);
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Replace:
                    {
                        // The items have to be KeyValuePairs.
                        var oldItems = e.OldItems.Cast<KeyValuePair<string, object>>().ToList();
                        var newItems = e.NewItems.Cast<KeyValuePair<string, object>>().ToList();
                        if (oldItems == null || newItems == null)
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Replace the items.
                        for (int i = 0; (i < oldItems.Count) && (i < newItems.Count); ++i)
                        {
                            // Find the item from the combo box items and replace it.
                            int index = this.IndexOfComboBoxItem(oldItems[i]);
                            if ((index >= 0) && (index < this.Items.Count))
                            {
                                // The item is found, replace it.
                                this.Items[index] =
                                    new ComboBoxItem(newItems[i].Key, newItems[i].Value);
                            }
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Move:
                    {
                        if ((e.OldStartingIndex < 0) ||
                            (e.OldStartingIndex >= this.Items.Count))
                        {
                            this.EndUpdate();
                            return;
                        }

                        // Save the item first, it will be removed from the list.
                        var item = this.Items[e.OldStartingIndex];

                        // アイテムを削除する。
                        this.Items.RemoveAt(e.OldStartingIndex);

                        // アイテムを追加する。
                        if ((e.NewStartingIndex < 0) ||
                            (e.NewStartingIndex >= this.Items.Count))
                        {
                            this.Items.Add(item);
                        }
                        else
                        {
                            this.Items.Insert(e.NewStartingIndex, item);
                        }

                        break;
                    }

                case NotifyCollectionChangedAction.Reset:
                    {
                        this.SelectedIndex = -1;
                        Items.Clear();

                        break;
                    }
            }

            this.EndUpdate();

            // サイズの自動調整を実行する。
            this.AdjustWidth();
            this.AdjustDropDownWidth();
        }

        /// <summary>
        /// 新しくセットしようとするアイテムリストが適用できるかどうかをチェックします。
        /// </summary>
        /// <param name="newItems">適用しようとしているアイテムリスト</param>
        /// <returns>完全に今のアイテムリストと等しかったらfalse,それ以外はtrueを返します。</returns>
        private bool IsAcceptableItems(IEnumerable<KeyValuePair<string, object>> newItems)
        {
            // どちらかがnullだったらItemsを受け入れる
            if (this.availableItems == null || newItems == null)
            {
                return true;
            }

            var oldList = this.availableItems.ToList();
            var newList = newItems.ToList();

            // 個数が違ったら受け入れる
            if (oldList.Count != newList.Count)
            {
                return true;
            }

            // KeyとValueを比較して等しくないものがあったら受け入れる
            for (int i = 0; i < oldList.Count; ++i)
            {
                if (oldList[i].Key != newList[i].Key)
                {
                    if (Comparer.Default.Compare(oldList[i].Value, newList[i].Value) != 0)
                    {
                        return true;
                    }
                }
            }

            // 等しいItemsだったら受け入れない
            return false;
        }

        /// <summary>
        /// コンボボックスに実際に登録されるアイテム
        /// コンボボックスに表示される値は、インスタンスのtoString()の
        /// 値となり、意味のない文字列の可能性もあるため、nameに
        /// セットした値を表示する。
        /// </summary>
        public class ComboBoxItem
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="name">コンボボックスに表示する名前</param>
            /// <param name="value">The value of the combo box item.</param>
            public ComboBoxItem(string name, object value)
            {
                this.Name = name;
                this.Value = value;
            }

            /// <summary>
            /// コンボボックスに表示する名前
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            /// The value of the combo box item.
            /// </summary>
            public object Value { get; private set; }

            /// <summary>
            /// コンボボックスに表示する名前を返す
            /// </summary>
            /// <returns>コンボボックスに表示する名前</returns>
            public override string ToString()
            {
                return this.Name;
            }
        }

        /// <summary>
        /// ComboBoxが持っているEditウィンドウを乗っ取るためのクラス
        /// </summary>
        internal class EditWindow : NativeWindow
        {
            /// <summary>
            /// Win32APIのWM_PAINT
            /// </summary>
            private const int WmPaint = 0x000F;

            /// <summary>
            /// Win32APIのWM_SETFOCUS
            /// </summary>
            private const int WmSetFocus = 7;

            /// <summary>
            /// Win32APIのWM_KILLFOCUS
            /// </summary>
            private const int WmKillFocus = 8;

            /// <summary>
            /// Editウィンドウにフォーカスがあるかどうか.
            /// </summary>
            private bool focused = false;

            /// <summary>
            /// 親のComboBox
            /// </summary>
            private UIComboBox comboBox;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="comboBox">親ComboBox</param>
            public EditWindow(UIComboBox comboBox)
            {
                this.comboBox = comboBox;
            }

            /// <summary>
            /// Editウィンドウのウィンドウプロシージャ
            /// </summary>
            /// <param name="m">メッセージ</param>
            protected override void WndProc(ref Message m)
            {
                base.WndProc(ref m);

                switch (m.Msg)
                {
                    case WmPaint:
                        if (this.focused == false && string.IsNullOrEmpty(this.comboBox.Text))
                        {
                            using (Graphics g = Graphics.FromHwnd(this.Handle))
                            {
                                using (var promptFont = new Font(this.comboBox.Font, FontStyle.Regular))
                                {
                                    const TextFormatFlags LeftFlag =
                                        TextFormatFlags.Left | TextFormatFlags.VerticalCenter;

                                    TextRenderer.DrawText(
                                        g,
                                        this.comboBox.PromptText,
                                        promptFont,
                                        new Point(0, 8),
                                        Color.Gray,
                                        Color.White,
                                        LeftFlag);
                                }
                            }
                        }

                        break;

                    case WmSetFocus:
                        this.focused = true;
                        break;

                    case WmKillFocus:
                        this.focused = false;
                        break;
                }
            }
        }
    }
}
