﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Log;
using System.Threading;

namespace EffectMaker.UIControls.BaseControls
{
    /// <summary>
    /// ログビュー.
    /// </summary>
    public class UILogView : UIUserControl, ILogHandler
    {
        /// <summary>
        /// ログビュー描画用キャンバス.
        /// </summary>
        private LogViewCanvas canvas = new LogViewCanvas();

        /// <summary>
        /// Constructor.
        /// </summary>
        public UILogView()
        {
            // 同期コンテキストを取得
            this.SynchronizationContext = SynchronizationContext.Current;

            if (!this.Controls.Contains(this.canvas))
            {
                this.Controls.Add(this.canvas);
            }

            // Adjust canvas size first.
            this.OnSizeChanged(EventArgs.Empty);

            this.BackColor = Color.White;
            this.BorderStyle = BorderStyle.FixedSingle;
            this.AutoScroll = true;

            this.DoubleBuffered = true;
        }

        #region Enumerator for line selecting modes

        /// <summary>
        /// Enumerator for line selecting modes.
        /// </summary>
        public enum SelectModes
        {
            /// <summary>
            /// 一行選択.
            /// </summary>
            SingleSelection,

            /// <summary>
            /// すべて選択.
            /// </summary>
            UnionSelection,

            /// <summary>
            /// 複数行選択.
            /// </summary>
            MultiSelection
        }

        #endregion

        #region Properties

        /// <summary>
        /// ログハンドラの名前を取得します。
        /// </summary>
        public string LogHandlerName
        {
            get { return "LogView"; }
        }

        /// <summary>
        /// ログハンドラの同期コンテキストを取得します。
        /// </summary>
        public SynchronizationContext SynchronizationContext { get; private set; }

        /// <summary>
        /// Get or set the back ground color for the control.
        /// </summary>
        public override Color BackColor
        {
            get
            {
                return base.BackColor;
            }

            set
            {
                base.BackColor = value;
                this.canvas.BackColor = value;
            }
        }

        /// <summary>
        /// Get or set the fore color for the control.
        /// </summary>
        public override Color ForeColor
        {
            get
            {
                return base.ForeColor;
            }

            set
            {
                base.ForeColor = value;
                this.canvas.ForeColor = value;
            }
        }

        /// <summary>
        /// Get or set the image list for the control.
        /// </summary>
        public ImageList ImageList
        {
            get
            {
                return this.canvas.ImageList;
            }

            set
            {
                this.canvas.ImageList = value;
            }
        }

        /// <summary>
        /// Get the number of messages.
        /// </summary>
        public int NumMessages
        {
            get { return this.canvas.NumMessages; }
        }

        /// <summary>
        /// Get the number of selected messages.
        /// </summary>
        public int NumSelectedMessages
        {
            get { return this.canvas.NumSelectedMessages; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to prepend line number before the message.
        /// </summary>
        public bool PrependLineNumber
        {
            get
            {
                return this.canvas.PrependLineNumber;
            }

            set
            {
                this.canvas.PrependLineNumber = value;
            }
        }

        #endregion

        /// <summary>
        /// ロガーからメッセージが送られたときの処理を行います。
        /// </summary>
        /// <param name="destinations">ログ出力先</param>
        /// <param name="level">ログレベル</param>
        /// <param name="message">ログメッセージ</param>
        /// <param name="callStack">コールスタック</param>
        public void Log(IEnumerable<string> destinations, LogLevels level, string message, StackFrame callStack)
        {
            if (this.IsDisposed) return;

            // 出力先をチェック.
            if (destinations == null)
            {
                return;
            }

            if (destinations.Any(s => s == this.LogHandlerName) == false)
            {
                return;
            }

            string[] messages = message.Split(new char[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);

            switch (level)
            {
                case LogLevels.Information:
                    this.WriteLines(messages, Color.Black, 0);
                    break;
                case LogLevels.Debug:
                    this.WriteLines(messages, Color.Blue, 0);
                    break;
                case LogLevels.Warning:
                    this.WriteLines(messages, Color.OrangeRed, 1);
                    break;
                case LogLevels.Error:
                    this.WriteLines(messages, Color.Red, 2);
                    break;
                case LogLevels.Fatal:
                    this.WriteLines(messages, Color.Red, 2);
                    break;
            }
        }

        /// <summary>
        /// Write a new line of log.
        /// </summary>
        /// <param name="messages">The messages to write.</param>
        /// <param name="textColor">The color for the text.</param>
        public void WriteLines(string[] messages, Color textColor)
        {
            foreach (string message in messages)
            {
                this.canvas.WriteLine(message, textColor);
            }

            // Adjust canvas size.
            this.OnSizeChanged(EventArgs.Empty);

            if (this.VerticalScroll.Visible == true)
            {
                this.VerticalScroll.Value = this.canvas.GetLastLinePos();
            }
        }

        /// <summary>
        /// Write a new line of log.
        /// </summary>
        /// <param name="messages">The messages to write.</param>
        /// <param name="textColor">The color for the text.</param>
        /// <param name="imageIndex">The icon index of the message.</param>
        public void WriteLines(string[] messages, Color textColor, int imageIndex)
        {
            foreach (string message in messages)
            {
                this.canvas.WriteLine(message, textColor, imageIndex);
            }

            // Adjust canvas size.
            this.OnSizeChanged(EventArgs.Empty);

            if (this.VerticalScroll.Visible == true)
            {
                this.VerticalScroll.Value = this.canvas.GetLastLinePos();
            }
        }

        /// <summary>
        /// Clear the log view, remove all the logged messages.
        /// </summary>
        public void Clear()
        {
            this.canvas.Clear();
        }

        /// <summary>
        /// Select all the messages.
        /// </summary>
        public void SelectAll()
        {
            this.canvas.SelectAll();
        }

        /// <summary>
        /// Select the specified line with the given selecting mode.
        /// </summary>
        /// <param name="lineNumber">The line number of the line to select.</param>
        /// <param name="mode">The selecting mode.</param>
        public void SelectMessage(int lineNumber, SelectModes mode)
        {
            this.canvas.SelectMessage(lineNumber, mode);
        }

        /// <summary>
        /// Clear the current selection.
        /// </summary>
        public void ClearSelection()
        {
            this.canvas.ClearSelection();
        }

        /// <summary>
        /// Get the text of the selected lines.
        /// </summary>
        /// <returns>The text of the selected lines.</returns>
        public string GetSelectedText()
        {
            return this.canvas.GetSelectedText();
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            this.canvas.UpdateSize();

            if (this.canvas != null &&
                 this.canvas.Width < this.Width)
            {
                this.canvas.Width = this.ClientSize.Width;
            }

            base.OnSizeChanged(e);
        }

        /// <summary>
        /// Class as the rendering canvas for UILogView.
        /// </summary>
        internal class LogViewCanvas : UIUserControl
        {
            #region Member variables

            /// <summary>
            /// 行データ.
            /// </summary>
            private List<LineData> lineData = new List<LineData>();

            /// <summary>
            /// 選択中の行データ.
            /// </summary>
            private LineData lastSelctedLine = null;

            /// <summary>
            /// 表現可能な行の最大数.
            /// </summary>
            private int numMaxLines = 999;

            /// <summary>
            /// NexLineNumber.
            /// </summary>
            private int nexLineNumber = 1;

            /// <summary>
            /// PrependLineNumber.
            /// </summary>
            private bool prependLineNumber = true;

            /// <summary>
            /// 左側スペース.
            /// </summary>
            private int leftSpacing = 5;

            /// <summary>
            /// 行の高さ.
            /// </summary>
            private int lineHeight = 17;

            /// <summary>
            /// シフトボタンが押されているかのフラグ.
            /// </summary>
            private bool shiftPress = false;

            /// <summary>
            /// コントロールボタンが押されているかのフラグ.
            /// </summary>
            private bool ctrlPress = false;

            /// <summary>
            /// イメージリスト.
            /// </summary>
            private ImageList imageList = null;

            /// <summary>
            /// 背景ブラシ.
            /// </summary>
            private SolidBrush backgroundBrush = null;

            /// <summary>
            /// 選択中の背景ブラシ.
            /// </summary>
            private SolidBrush selBgBrush = null;

            /// <summary>
            /// テキストブラシ.
            /// </summary>
            private SolidBrush textBrush = null;

            /// <summary>
            /// 選択中のテキストブラシ.
            /// </summary>
            private SolidBrush selTextBrush = null;

            /// <summary>
            /// フォーカスのある領域を描画するペン.
            /// </summary>
            private Pen focusRectPen = null;

            /// <summary>
            /// 書式.
            /// </summary>
            private StringFormat strFormat = new StringFormat();

            #endregion

            #region Constructor

            /// <summary>
            /// Default constructor.
            /// </summary>
            public LogViewCanvas()
            {
                this.backgroundBrush = new SolidBrush(this.BackColor);
                this.selBgBrush = new SolidBrush(SystemColors.Highlight);
                this.textBrush = new SolidBrush(this.ForeColor);
                this.selTextBrush = new SolidBrush(SystemColors.HighlightText);

                this.focusRectPen = new Pen(Color.Gray, 1);
                this.focusRectPen.DashStyle = DashStyle.Dot;

                this.strFormat.Alignment = StringAlignment.Near;
                this.strFormat.LineAlignment = StringAlignment.Center;
            }

            #endregion

            #region Properties

            /// <summary>
            /// Get or set the back ground color for the control.
            /// </summary>
            public override Color BackColor
            {
                get
                {
                    return base.BackColor;
                }

                set
                {
                    base.BackColor = value;
                    this.backgroundBrush.Color = value;
                }
            }

            /// <summary>
            /// Get or set the fore color for the control.
            /// </summary>
            public override Color ForeColor
            {
                get
                {
                    return base.ForeColor;
                }

                set
                {
                    base.ForeColor = value;
                    this.textBrush.Color = value;
                }
            }

            /// <summary>
            /// Get or set the image list for the control.
            /// </summary>
            public ImageList ImageList
            {
                get
                {
                    return this.imageList;
                }

                set
                {
                    this.imageList = value;
                }
            }

            /// <summary>
            /// Get the number of messages.
            /// </summary>
            public int NumMessages
            {
                get { return this.lineData.Count; }
            }

            /// <summary>
            /// Get the number of selected messages.
            /// </summary>
            public int NumSelectedMessages
            {
                get
                {
                    int numItems = 0;
                    foreach (LineData data in this.lineData)
                    {
                        if (data.Selected == true)
                        {
                            ++numItems;
                        }
                    }

                    return numItems;
                }
            }

            /// <summary>
            /// Get or set the flag indicating whether to prepend line number before the message.
            /// </summary>
            public bool PrependLineNumber
            {
                get
                {
                    return this.prependLineNumber;
                }

                set
                {
                    this.prependLineNumber = value;
                }
            }

            #endregion

            #region Log message

            /// <summary>
            /// Write a new line of log.
            /// </summary>
            /// <param name="message">The message to write.</param>
            /// <param name="textColor">Value for the text.</param>
            public void WriteLine(string message, Color textColor)
            {
                this.lineData.Add(new LineData(
                    this.nexLineNumber, message, textColor, -1));

                ++this.nexLineNumber;

                if (this.lineData.Count > this.numMaxLines)
                {
                    this.lineData.RemoveAt(0);
                }

                if (this.nexLineNumber > this.numMaxLines)
                {
                    this.nexLineNumber = 1;
                }

                this.UpdateSize();
                this.Invalidate();
            }

            /// <summary>
            /// Write a new line of log.
            /// </summary>
            /// <param name="message">The message to write.</param>
            /// <param name="textColor">Value for the text.</param>
            /// <param name="imageIndex">The icon index of the message.</param>
            public void WriteLine(string message, Color textColor, int imageIndex)
            {
                this.lineData.Add(
                    new LineData(this.nexLineNumber, message, textColor, imageIndex));

                ++this.nexLineNumber;

                if (this.lineData.Count > this.numMaxLines)
                {
                    this.lineData.RemoveAt(0);
                }

                if (this.nexLineNumber > this.numMaxLines)
                {
                    this.nexLineNumber = 1;
                }

                this.UpdateSize();
                this.Invalidate();
            }

            /// <summary>
            /// Clear the log view, remove all the logged messages.
            /// </summary>
            public void Clear()
            {
                this.lineData.Clear();
                this.nexLineNumber = 1;

                this.UpdateSize();
                this.Invalidate();
            }

            #endregion

            #region Update

            /// <summary>
            /// Update the size of the lines and this control.
            /// </summary>
            /// <param name="g">The graphics object.</param>
            public void UpdateSize(Graphics g = null)
            {
                if (this.IsHandleCreated == false ||
                     this.Visible == false)
                {
                    return;
                }

                // !ここで作成した場合、Disposeしなくていいのか？
                bool releaseGraphics = false;
                if (g == null)
                {
                    releaseGraphics = true;
                    g = this.CreateGraphics();
                }

                Font font = this.Font;

                int posX = 0;
                int posY = 0;
                int width = 0;
                int height = 0;

                int controlWidth = 0;
                int controlHeight = 0;

                // Render the lines
                foreach (LineData data in this.lineData)
                {
                    posX = this.leftSpacing;

                    // Get the image for the line.
                    Image icon = null;
                    if (this.imageList != null &&
                         data.ImageIndex >= 0 &&
                         data.ImageIndex < this.imageList.Images.Count)
                    {
                        icon = this.imageList.Images[data.ImageIndex];
                    }

                    // Leave some space on the left of the line.
                    width = this.leftSpacing;
                    height = this.lineHeight;

                    // Add the width of the image.
                    if (icon != null)
                    {
                        width += icon.Width;
                        height = Math.Max(height, icon.Height);
                    }

                    // Get the font for the line.
                    if (data.Font == null)
                    {
                        font = this.Font;
                    }
                    else
                    {
                        font = data.Font;
                    }

                    string text = string.Empty;
                    if (this.prependLineNumber == true)
                    {
                        text = data.LineNumberText + " " + data.Text;
                    }
                    else
                    {
                        text = data.Text;
                    }

                    // Measure the text size.
                    if (string.IsNullOrEmpty(text) == false)
                    {
                        SizeF lineSize = g.MeasureString(text, font);

                        // Add the width of the text.
                        width += (int)lineSize.Width + 1;
                        height = Math.Max(height, (int)lineSize.Height + 1);
                    }

                    // Set the extents to the line data.
                    data.SetBounds(0, posY, width, height);

                    controlWidth = Math.Max(controlWidth, width);
                    controlHeight = data.Bounds.Bottom;

                    // Increment the Y position.
                    posY += data.Bounds.Height;
                }

                // Change the size of the log view.
                if (this.Width != controlWidth ||
                     this.Height != controlHeight)
                {
                    this.Size = new Size(controlWidth, controlHeight);
                }

                if (releaseGraphics)
                {
                    g.Dispose();
                }
            }

            #endregion

            #region Selection

            /// <summary>
            /// Select all the messages.
            /// </summary>
            public void SelectAll()
            {
                if (this.lineData.Count <= 0)
                {
                    return;
                }

                foreach (LineData data in this.lineData)
                {
                    data.Selected = true;
                    this.lastSelctedLine = data;
                }

                this.Invalidate();
            }

            /// <summary>
            /// Select the specified line with the given selecting mode.
            /// </summary>
            /// <param name="lineNumber">The line number of the line to select.</param>
            /// <param name="mode">The selecting mode.</param>
            public void SelectMessage(int lineNumber, UILogView.SelectModes mode)
            {
                if (this.lineData.Count <= 0)
                {
                    return;
                }

                if (lineNumber < 0 ||
                     lineNumber >= this.lineData.Count)
                {
                    return;
                }

                if (mode == UILogView.SelectModes.SingleSelection)
                {
                    int index = 0;
                    foreach (LineData data in this.lineData)
                    {
                        if (lineNumber != index)
                        {
                            data.Selected = false;
                        }
                        else
                        {
                            data.Selected = true;
                            this.lastSelctedLine = data;
                        }

                        ++index;
                    }
                }
                else if (mode == UILogView.SelectModes.MultiSelection)
                {
                    int lastSelIndex = 0;
                    if (this.lastSelctedLine != null)
                    {
                        foreach (LineData data in this.lineData)
                        {
                            if (this.lastSelctedLine == data)
                            {
                                break;
                            }

                            ++lastSelIndex;
                        }

                        if (lastSelIndex >= this.lineData.Count)
                        {
                            lastSelIndex = 0;
                        }
                    }

                    int startIndex = Math.Min(lineNumber, lastSelIndex);
                    int endIndex = Math.Max(lineNumber, lastSelIndex);
                    for (int i = 0; i < this.lineData.Count; ++i)
                    {
                        if (i >= startIndex && i <= endIndex)
                        {
                            this.lineData[i].Selected = true;
                        }
                        else
                        {
                            this.lineData[i].Selected = false;
                        }

                        if (i == lineNumber)
                        {
                            this.lastSelctedLine = this.lineData[i];
                        }
                    }
                }
                else if (mode == UILogView.SelectModes.UnionSelection)
                {
                    int index = 0;
                    foreach (LineData data in this.lineData)
                    {
                        if (index == lineNumber)
                        {
                            data.Selected = !data.Selected;
                            this.lastSelctedLine = data;
                            break;
                        }

                        ++index;
                    }
                }

                this.Invalidate();
            }

            /// <summary>
            /// Clear the current selection.
            /// </summary>
            public void ClearSelection()
            {
                foreach (LineData data in this.lineData)
                {
                    data.Selected = false;
                }

                this.lastSelctedLine = null;
                this.Invalidate();
            }

            #endregion

            #region Copy

            /// <summary>
            /// Get the text of the selected lines.
            /// </summary>
            /// <returns>The text of the selected lines.</returns>
            public string GetSelectedText()
            {
                string text = string.Empty;
                foreach (LineData data in this.lineData)
                {
                    if (data.Selected == true)
                    {
                        if (text.Length > 0)
                        {
                            text += "\r\n";
                        }

                        text += data.Text;
                    }
                }

                return text;
            }

            #endregion

            #region Utility public methods

            /// <summary>
            /// Get the Y position of the last line.
            /// </summary>
            /// <returns>The Y position of the last line.</returns>
            public int GetLastLinePos()
            {
                if (this.lineData.Count <= 0)
                {
                    return 0;
                }

                LineData data = this.lineData[this.lineData.Count - 1];

                return data.Bounds.Y;
            }

            #endregion // Utility public methods

            #region Dispose

            /// <summary>
            /// Dispose.
            /// </summary>
            /// <param name="disposing">True on disposing.</param>
            protected override void Dispose(bool disposing)
            {
                base.Dispose(disposing);

                this.backgroundBrush.Dispose();
                this.selBgBrush.Dispose();
                this.textBrush.Dispose();
                this.selTextBrush.Dispose();
                this.focusRectPen.Dispose();
            }

            #endregion

            #region Event handlers

            /// <summary>
            /// Handle Paint event for the control.
            /// </summary>
            /// <param name="e">The event arguments.</param>
            protected override void OnPaint(PaintEventArgs e)
            {
                base.OnPaint(e);

                // Render the background.
                e.Graphics.FillRectangle(this.backgroundBrush, this.ClientRectangle);

                Font font = this.Font;
                int posX = 0;

                RectangleF textRect = new RectangleF();

                // Render the lines
                foreach (LineData data in this.lineData)
                {
                    posX = this.leftSpacing;

                    // Get the image for the line.
                    Image icon = null;
                    if (this.imageList != null &&
                         data.ImageIndex >= 0 &&
                         data.ImageIndex < this.imageList.Images.Count)
                    {
                        icon = this.imageList.Images[data.ImageIndex];
                    }

                    // Render the selected background if the line is selected.
                    if (this.Focused == true &&
                         data.Selected == true)
                    {
                        Rectangle selRect = new Rectangle(
                                                0,
                                                data.Bounds.Y,
                                                this.Width,
                                                data.Bounds.Height - 1);

                        e.Graphics.FillRectangle(this.selBgBrush, selRect);

                        // Render the dotted focus rectangle.
                        if (this.Focused == true)
                        {
                            e.Graphics.DrawRectangle(
                                       this.focusRectPen,
                                       selRect.X,
                                       selRect.Y,
                                       selRect.Width - 1,
                                       selRect.Height - 1);
                        }
                    }

                    // Render the icon image.
                    if (icon != null)
                    {
                        int x = posX;
                        int y = data.Bounds.Y + ((data.Bounds.Height - icon.Height) / 2);
                        e.Graphics.DrawImage(icon, x, y);

                        posX += icon.Width;
                    }

                    string text = string.Empty;
                    if (this.prependLineNumber == true)
                    {
                        text = data.LineNumberText + " " + data.Text;
                    }
                    else
                    {
                        text = data.Text;
                    }

                    if (string.IsNullOrEmpty(text) == false)
                    {
                        // Determine the brush to use for rendering the text.
                        Brush textBrush;
                        if (this.Focused == true &&
                             data.Selected == true)
                        {
                            textBrush = this.selTextBrush;
                        }
                        else
                        {
                            this.textBrush.Color = data.TextColor;
                            textBrush = this.textBrush;
                        }

                        textRect.X = posX;
                        textRect.Y = data.Bounds.Y;
                        textRect.Width = data.Bounds.Width - posX;
                        textRect.Height = data.Bounds.Height;

                        // Render the text.
                        e.Graphics.DrawString(
                                   text,
                                   font,
                                   textBrush,
                                   textRect,
                                   this.strFormat);
                    }
                }
            }

            /// <summary>
            /// Handle MouseDown event for the control.
            /// </summary>
            /// <param name="e">The event arguments.</param>
            protected override void OnMouseDown(MouseEventArgs e)
            {
                base.OnMouseDown(e);

                // 右クリックだった場合は、ログの選択を変更しない
                if (e.Button.Equals(MouseButtons.Right) == true)
                {
                    return;
                }

                int lineNumber = this.GetLineNumberAtPoint(e.X, e.Y);
                if (lineNumber >= 0 &&
                     lineNumber < this.lineData.Count)
                {
                    if (this.shiftPress == true)
                    {
                        this.SelectMessage(lineNumber, UILogView.SelectModes.MultiSelection);
                    }
                    else if (this.ctrlPress == true)
                    {
                        this.SelectMessage(lineNumber, UILogView.SelectModes.UnionSelection);
                    }
                    else
                    {
                        this.SelectMessage(lineNumber, UILogView.SelectModes.SingleSelection);
                    }
                }
                else
                {
                    this.ClearSelection();
                }
            }

            /// <summary>
            /// Handle KeyDown event for the control.
            /// </summary>
            /// <param name="e">The event arguments.</param>
            protected override void OnKeyDown(KeyEventArgs e)
            {
                base.OnKeyDown(e);

                this.shiftPress = e.Shift;
                this.ctrlPress = e.Control;

                if (e.Control == true)
                {
                    if (e.KeyCode == Keys.A)
                    {
                        this.SelectAll();
                    }
                    else if (e.KeyCode == Keys.C &&
                              this.NumSelectedMessages > 0)
                    {
                        Clipboard.SetDataObject(this.GetSelectedText(), true);
                    }
                }
                else if (e.KeyCode == Keys.Delete)
                {
                    this.Clear();
                }
            }

            /// <summary>
            /// Handle KeyUp event for the control.
            /// </summary>
            /// <param name="e">The event arguments.</param>
            protected override void OnKeyUp(KeyEventArgs e)
            {
                base.OnKeyUp(e);

                this.shiftPress = e.Shift;
                this.ctrlPress = e.Control;
            }

            /// <summary>
            /// Handle GotFocus event for the control.
            /// </summary>
            /// <param name="e">The event arguments.</param>
            protected override void OnGotFocus(EventArgs e)
            {
                base.OnGotFocus(e);

                this.Invalidate();
            }

            /// <summary>
            /// Handle LostFocus event for the control.
            /// </summary>
            /// <param name="e">The event arguments.</param>
            protected override void OnLostFocus(EventArgs e)
            {
                base.OnLostFocus(e);

                this.Invalidate();
            }

            #endregion

            #region Utility methods

            /// <summary>
            /// Get the line number at the given point.
            /// </summary>
            /// <param name="x">The X position.</param>
            /// <param name="y">The Y position.</param>
            /// <returns>The line number.</returns>
            private int GetLineNumberAtPoint(int x, int y)
            {
                int index = 0;
                foreach (LineData data in this.lineData)
                {
                    if (data.Bounds.Contains(x, y) == true)
                    {
                        return index;
                    }

                    ++index;
                }

                return -1;
            }

            /// <summary>
            /// Get the line data at the given point.
            /// </summary>
            /// <param name="x">The X position.</param>
            /// <param name="y">The Y position.</param>
            /// <returns>The line data.</returns>
            private LineData GetLineAtPoint(int x, int y)
            {
                foreach (LineData data in this.lineData)
                {
                    if (data.Bounds.Contains(x, y) == true)
                    {
                        return data;
                    }
                }

                return null;
            }

            #endregion // Utility methods

            #region Class for line data

            /// <summary>
            /// Class for storing the line data.
            /// </summary>
            private class LineData
            {
                /// <summary>
                /// 領域.
                /// </summary>
                private Rectangle bounds = new Rectangle();

                /// <summary>
                /// Constructor.
                /// </summary>
                /// <param name="lineNumber">Line number.</param>
                /// <param name="text">The text to show on the line.</param>
                /// <param name="textColor">The color for the text.</param>
                /// <param name="imageIndex">The icon image index for the line.</param>
                public LineData(
                    int lineNumber, string text, Color textColor, int imageIndex)
                {
                    this.LineNumber = lineNumber;
                    this.ImageIndex = imageIndex;
                    this.Text = string.Copy(text);
                    this.TextColor = textColor;
                    this.Font = null;
                    this.Selected = false;

                    this.Text = this.Text.Replace("\r", string.Empty);
                    this.Text = this.Text.Replace("\n", string.Empty);

                    this.LineNumberText = string.Format("[{0:d3}]", this.LineNumber);
                }

                /// <summary>Get or set the line number of the line.</summary>
                public int LineNumber { get; set; }

                /// <summary>Get or set the formatted string of the line number.</summary>
                public string LineNumberText { get; set; }

                /// <summary>Get or set the color for the text.</summary>
                public Color TextColor { get; set; }

                /// <summary>Get or set the text to show on the line.</summary>
                public string Text { get; set; }

                /// <summary>Get or set the font for the text.</summary>
                public Font Font { get; set; }

                /// <summary>Get or set the index of the icon image.</summary>
                public int ImageIndex { get; set; }

                /// <summary>Get or set the flag whether this line is selected.</summary>
                public bool Selected { get; set; }

                /// <summary>Get the bounds of the line.</summary>
                public Rectangle Bounds
                {
                    get { return this.bounds; }
                }

                /// <summary>
                /// Set the bounds for the line.
                /// </summary>
                /// <param name="x">The X position of the line.</param>
                /// <param name="y">The Y position of the line.</param>
                /// <param name="width">The width of the line.</param>
                /// <param name="height">The height of the line.</param>
                public void SetBounds(int x, int y, int width, int height)
                {
                    this.bounds.X = x;
                    this.bounds.Y = y;
                    this.bounds.Width = width;
                    this.bounds.Height = height;
                }
            }

            #endregion
        }
    }
}
