﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.UIControls.BaseControls;
using System;
using System.Drawing;
using System.IO;
using System.Windows.Forms;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// ファイルパス向けTextBox
    /// </summary>
    public partial class UIFilePathTextBox : UITextBox
    {
        /// <summary>
        /// ファイルパスの実態
        /// </summary>
        private string filePath;

        /// <summary>
        /// ツールチップ
        /// </summary>
        private ToolTip toolTip = null;

        /// <summary>
        /// ツールチップ文字列
        /// </summary>
        private string toolTipText;

        /// <summary>
        /// ツールチップを表示させないかどうか.
        /// </summary>
        private bool disableToolTip = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UIFilePathTextBox()
        {
            this.ToolTipPoint = new Point(20, -25);
            this.ToolTipDuration = 2000;
        }

        /// <summary>
        /// FilePathが変更された場合に実行されるイベントハンドラ
        /// </summary>
        public event EventHandler FilePathChanged;

        /// <summary>
        /// ファイルパスのプロパティ
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }

            set
            {
                this.filePath = value;

                if (this.AlwaysShowFullPath == true)
                {
                    // AlwaysShowFullPathがtrueのときは、フルパスを表示する。
                    this.Text = this.FilePath;
                }
                else
                {
                    // テキストボックスで表示する文字列は、ファイル名だけにする.
                    this.Text = Path.GetFileName(this.FilePath);
                }

                // ツールチップのテキストはフルパスにする.
                this.ToolTipText = value;

                // FilePathChangedハンドラに登録されている処理を実行
                if (this.FilePathChanged != null)
                {
                    this.FilePathChanged(this, null);
                }
            }
        }

        /// <summary>
        /// ツールチップテキストの指定
        /// </summary>
        public string ToolTipText
        {
            get
            {
                return this.toolTipText;
            }

            set
            {
                this.toolTipText = value;

                if (this.toolTip == null && this.DisableToolTip == false)
                {
                    this.toolTip = new ToolTip
                    {
                        AutoPopDelay = 1200,
                        InitialDelay = 1,
                        ReshowDelay = 1
                    };

                    this.MouseEnter += (s, e) => this.ShowToolTip();
                    this.MouseLeave += (s, e) => this.toolTip.Hide(this);
                }
            }
        }

        /// <summary>
        /// ツールチップを表示させないかどうか.
        /// </summary>
        public bool DisableToolTip
        {
            get
            {
                return this.disableToolTip;
            }

            set
            {
                this.disableToolTip = value;

                // this.ToolTipTextに文字列が入っていて、this.toolTipがnullでない場合
                if (this.toolTip != null)
                {
                    // DisableToolTipに値が代入される度に、イベントの再登録をやってしまう.
                    // おそらく一番確実で安全な方法.
                    this.MouseEnter -= (s, e) => this.ShowToolTip();
                    this.MouseLeave -= (s, e) => this.toolTip.Hide(this);

                    // falseの場合は、ツールチップを表示するイベントを再度登録する,
                    if (this.disableToolTip == false)
                    {
                        this.MouseEnter += (s, e) => this.ShowToolTip();
                        this.MouseLeave += (s, e) => this.toolTip.Hide(this);
                    }
                }
            }
        }

        /// <summary>
        /// ツールチップテキストを表示する座標
        /// </summary>
        public Point ToolTipPoint { get; set; }

        /// <summary>
        /// ツールチップテキストを表示する時間
        /// -1ならマウスを当てている間ずっと表示する。
        /// </summary>
        public int ToolTipDuration { get; set; }

        /// <summary>
        /// 非選択時でもフルパスを表示するかどうか.
        /// </summary>
        public bool AlwaysShowFullPath { get; set; }

        /// <summary>
        /// クリック時のテキスト全選択
        /// </summary>
        /// <param name="e">event</param>
        protected override void OnClick(EventArgs e)
        {
            base.OnClick(e);
            this.Text = this.FilePath;
            this.SelectAll();
        }

        /// <summary>
        /// The wnd proc.
        /// </summary>
        /// <param name="m">
        /// The m.
        /// </param>
        protected override void WndProc(ref Message m)
        {
            const int WmPaint = 15;
            const int WmSetfocus = 7;
            const int WmKillfocus = 8;

            switch (m.Msg)
            {
                case WmSetfocus:
                    this.IsDrawBackground = false;
                    this.Text = this.FilePath;
                    break;

                case WmKillfocus:
                    this.IsDrawBackground = true;
                    if (this.AlwaysShowFullPath == true)
                    {
                        // AlwaysShowFullPathがtrueのときは、フルパスを表示する。
                        this.Text = this.FilePath;
                    }
                    else
                    {
                        this.Text = Path.GetFileName(this.FilePath);
                    }

                    break;
            }

            base.WndProc(ref m);

            if (m.Msg == WmPaint)
            {
                if (this.IsDrawBackground && string.IsNullOrEmpty(this.Text))
                {
                    using (var g = this.CreateGraphics())
                    {
                        this.DrawPrompt(g);
                    }
                }
            }
        }

        /// <summary>
        /// ツールチップを表示する。
        /// </summary>
        private void ShowToolTip()
        {
            if (this.ToolTipDuration < 0)
            {
                this.toolTip.Show(this.toolTipText, this, this.ToolTipPoint);
            }
            else
            {
                this.toolTip.Show(this.toolTipText, this, this.ToolTipPoint, this.ToolTipDuration);
            }
        }
    }
}
