﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// Class for the bordered label of the components.
    /// </summary>
    public class UIBorderedLabel : UIControl
    {
        #region Member variables

        /// <summary>コントロールが無効なときのテキストカラーです。</summary>
        private static readonly Color TextColorDisabled = Color.FromArgb(255, 150, 150, 150);

        /// <summary>コントロールが無効なときの背景カラーです。</summary>
        private static readonly Color BackColorDisabled = Color.FromArgb(255, 235, 235, 235);

        /// <summary>サイズを自動計算するためのフラグです。</summary>
        private bool myAutoSize = true;

        /// <summary>複数行表示を行うためのフラグです。</summary>
        private bool multiline = false;

        /// <summary>テキストだけを描画するフラグです。</summary>
        private bool drawTextOnly = false;

        /// <summary>ContentAlignment型のテキストアラインメントです。</summary>
        private ContentAlignment textAlign = 0;

        /// <summary>StringFormat型のテキストアラインメントです。</summary>
        private StringFormat stringFormat = new StringFormat();

        /// <summary>ラベルの横幅の最大値です。</summary>
        private int maxWidth = -1;

        /// <summary>テキストの周りのスペースです。</summary>
        private Padding spacing = new Padding(1);

        /// <summary>角の半径です。</summary>
        private int cornerRadius = 4;

        /// <summary>枠線の色です。</summary>
        private Color borderColor = Color.Black;

        /// <summary>背景色です。</summary>
        private Color backColor = Color.FromArgb(255, 255, 255, 153);

        /// <summary>サイズ変更イベントを無視するためのフラグです。</summary>
        private bool ignoreSizeEvent = false;

        /// <summary>アンダーラインの有無です。</summary>
        private bool underline = false;

        /// <summary>通常フォントのバックアップです。</summary>
        private Font regularFont = null;

        /// <summary>アンダーラインフォントのバックアップです。</summary>
        private Font underlineFont = null;

        #endregion

        #region Constructor

        /// <summary>
        /// Default Constructor.
        /// </summary>
        public UIBorderedLabel()
        {
            this.DoubleBuffered = true;
            this.TextAlign = ContentAlignment.MiddleCenter;
            this.TabStop = false;
            this.regularFont = this.Font;

            // ラベルのサイズを計算
            this.ComputeAutoSize();
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the maximum width of the label.
        /// -1 means the width is unlimited.
        /// </summary>
        [DefaultValue(-1)]
        public int MaxWidth
        {
            get
            {
                return this.maxWidth;
            }

            set
            {
                if (value == this.maxWidth)
                {
                    return;
                }

                this.maxWidth = value;

                this.ComputeAutoSize();
            }
        }

        /// <summary>
        /// Get or set the flag to render only text, without the background and border.
        /// </summary>
        [DefaultValue(false)]
        public bool DrawTextOnly
        {
            get
            {
                return this.drawTextOnly;
            }

            set
            {
                if (value == this.drawTextOnly)
                {
                    return;
                }

                this.drawTextOnly = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the radius of the rounded corners.
        /// </summary>
        [DefaultValue(4)]
        public int CornerRadius
        {
            get
            {
                return this.cornerRadius;
            }

            set
            {
                if (value == this.cornerRadius)
                {
                    return;
                }

                this.cornerRadius = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the flag for multi-lined label.
        /// </summary>
        [DefaultValue(false)]
        public bool Multiline
        {
            get
            {
                return this.multiline;
            }

            set
            {
                if (value == this.multiline)
                {
                    return;
                }

                this.multiline = value;
                this.AdjustHeightForMaxWidth();
                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the text alignment.
        /// </summary>
        [DefaultValue(ContentAlignment.MiddleCenter)]
        public ContentAlignment TextAlign
        {
            get
            {
                return this.textAlign;
            }

            set
            {
                if (value == this.textAlign)
                {
                    return;
                }

                this.textAlign = value;

                switch (this.textAlign)
                {
                    case ContentAlignment.TopLeft:
                        this.stringFormat.Alignment = StringAlignment.Near;
                        this.stringFormat.LineAlignment = StringAlignment.Near;
                        break;

                    case ContentAlignment.TopCenter:
                        this.stringFormat.Alignment = StringAlignment.Center;
                        this.stringFormat.LineAlignment = StringAlignment.Near;
                        break;

                    case ContentAlignment.TopRight:
                        this.stringFormat.Alignment = StringAlignment.Far;
                        this.stringFormat.LineAlignment = StringAlignment.Near;
                        break;

                    case ContentAlignment.MiddleLeft:
                        this.stringFormat.Alignment = StringAlignment.Near;
                        this.stringFormat.LineAlignment = StringAlignment.Center;
                        break;

                    case ContentAlignment.MiddleCenter:
                        this.stringFormat.Alignment = StringAlignment.Center;
                        this.stringFormat.LineAlignment = StringAlignment.Center;
                        break;

                    case ContentAlignment.MiddleRight:
                        this.stringFormat.Alignment = StringAlignment.Far;
                        this.stringFormat.LineAlignment = StringAlignment.Center;
                        break;

                    case ContentAlignment.BottomLeft:
                        this.stringFormat.Alignment = StringAlignment.Near;
                        this.stringFormat.LineAlignment = StringAlignment.Far;
                        break;

                    case ContentAlignment.BottomCenter:
                        this.stringFormat.Alignment = StringAlignment.Center;
                        this.stringFormat.LineAlignment = StringAlignment.Far;
                        break;

                    case ContentAlignment.BottomRight:
                        this.stringFormat.Alignment = StringAlignment.Far;
                        this.stringFormat.LineAlignment = StringAlignment.Far;
                        break;
                }

                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the color of the border.
        /// </summary>
        public Color BorderColor
        {
            get
            {
                return this.borderColor;
            }

            set
            {
                if (value == this.borderColor)
                {
                    return;
                }

                this.borderColor = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the background color.
        /// </summary>
        public new Color BackColor
        {
            get
            {
                return this.backColor;
            }

            set
            {
                if (value == this.backColor)
                {
                    return;
                }

                this.backColor = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set auto size property.
        /// </summary>
        [DefaultValue(true)]
        public override bool AutoSize
        {
            get
            {
                return this.myAutoSize;
            }

            set
            {
                if (value == this.myAutoSize)
                {
                    return;
                }

                this.myAutoSize = value;

                if (this.myAutoSize)
                {
                    this.maxWidth = -1;
                }

                this.ComputeAutoSize();
            }
        }

        /// <summary>
        /// テキストを取得または設定します。設定と同時に再描画を行います。
        /// </summary>
        [Editor("System.ComponentModel.Design.MultilineStringEditor, System.Design, Version=2.0.0.0, Culture=neutral, PublicKeyToken=b03f5f7f11d50a3a", typeof(System.Drawing.Design.UITypeEditor))]
        public new string Text
        {
            get
            {
                return base.Text;
            }

            set
            {
                base.Text = value;
                this.Invalidate();
            }
        }

        /// <summary>
        /// アンダーラインを付けるか否かを取得、または設定します。
        /// </summary>
        public bool Underline
        {
            get
            {
                return this.underline;
            }

            set
            {
                this.underline = value;
                if (value && this.underlineFont == null)
                {
                    this.underlineFont = new Font(DefaultFont.FontFamily, DefaultFont.Size, FontStyle.Underline);
                }

                this.Font = value ? underlineFont : regularFont;

                this.Invalidate();
            }
        }

        /// <summary>
        /// Get or set the spacing of the rounded corners.
        /// </summary>
        [DefaultValue(1)]
        protected Padding Spacing
        {
            get
            {
                return this.spacing;
            }

            set
            {
                if (value == this.spacing)
                {
                    return;
                }

                this.spacing = value;
                this.Invalidate();
            }
        }

        #endregion

        #region Event handlers

        /// <summary>
        /// Called when a parent request the desired size.
        /// </summary>
        /// <param name="proposedSize">The available parent size.</param>
        /// <returns>Returns the desired sife of the control.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            int maxWidth = proposedSize.Width;

            if (this.MaxWidth != -1 && this.MaxWidth < maxWidth)
            {
                maxWidth = this.MaxWidth;
            }

            Size size = this.ComputeBestFitSize(maxWidth);

            return size;
        }

        /// <summary>
        /// Handle Pain event.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
        {
            // Get the graphics object for rendering
            Graphics g = e.Graphics;

            // Smooth the lines
            g.SmoothingMode = SmoothingMode.HighQuality;

            base.OnPaint(e);

            // Setup the sizes
            int controlWidth = this.Width - 1;
            int controlHeight = this.Height - 1;
            int textWidth = controlWidth - this.spacing.Horizontal;
            int textHeight = controlHeight - this.spacing.Vertical;

            Color textColor = this.ForeColor;
            Color backColor = this.BackColor;

            if (this.Enabled == false)
            {
                backColor = UIBorderedLabel.BackColorDisabled;
                textColor = UIBorderedLabel.TextColorDisabled;
            }

            // Rectangle for text rendering
            RectangleF textRect = new Rectangle(
                this.spacing.Left,
                this.spacing.Top,
                textWidth + 1,
                textHeight + 1);

            // DrawText
            if (this.DrawTextOnly == true)
            {
                using (SolidBrush brush = new SolidBrush(textColor))
                {
                    g.DrawString(
                        this.Text,
                        this.Font,
                        brush,
                        textRect,
                        this.stringFormat);
                }
            }
            else
            {
                // Setup the rectangles
                Rectangle controlRect = new Rectangle(
                    0,
                    0,
                    controlWidth,
                    controlHeight);

                // Create the path
                GraphicsPath rectPath = new GraphicsPath();

                rectPath.AddArc(
                    controlRect.Left,
                    controlRect.Top,
                    this.CornerRadius,
                    this.CornerRadius,
                    180.0f,
                    90.0f);

                rectPath.AddArc(
                    controlRect.Right - this.CornerRadius,
                    controlRect.Top,
                    this.CornerRadius,
                    this.CornerRadius,
                    270.0f,
                    90.0f);

                rectPath.AddArc(
                    controlRect.Right - this.CornerRadius,
                    controlRect.Bottom - this.CornerRadius,
                    this.CornerRadius,
                    this.CornerRadius,
                    0.0f,
                    90.0f);

                rectPath.AddArc(
                    controlRect.Left,
                    controlRect.Bottom - this.CornerRadius,
                    this.CornerRadius,
                    this.CornerRadius,
                    90.0f,
                    90.0f);

                rectPath.CloseAllFigures();

                // 背景を描画
                using (SolidBrush brush = new SolidBrush(backColor))
                {
                    g.FillPath(brush, rectPath);
                }

                // 文字を描画
                using (SolidBrush brush = new SolidBrush(textColor))
                {
                    g.DrawString(
                        this.Text,
                        this.Font,
                        brush,
                        textRect,
                        this.stringFormat);
                }

                // 枠線を描画
                using (Pen pen = new Pen(this.borderColor))
                {
                    g.DrawPath(pen, rectPath);
                }

                // パスを削除
                if (rectPath != null)
                {
                    rectPath.Dispose();
                    rectPath = null;
                }
            }
        }

        /// <summary>
        /// Handle text changed event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnTextChanged(EventArgs e)
        {
            base.OnTextChanged(e);

            this.ComputeAutoSize();
        }

        /// <summary>
        /// Handle size changed event.
        /// </summary>
        /// <param name="e">Event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (this.ignoreSizeEvent)
            {
                return;
            }

            base.OnSizeChanged(e);

            if (!this.myAutoSize)
            {
                this.AdjustHeightForMaxWidth();
                this.Invalidate();
            }
        }

        #endregion

        #region Utilities

        /// <summary>
        /// Adjust label height for maximum width.
        /// </summary>
        protected void AdjustHeightForMaxWidth()
        {
            if (this.MaxWidth <= 0 ||
                this.Multiline == false ||
                this.AutoSize == true)
            {
                return;
            }

            Graphics g = this.CreateGraphics();
            g.SmoothingMode = SmoothingMode.HighQuality;

            int controlWidth  = this.Width - 1;
            int controlHeight = this.Height - 1;
            int textWidth     = controlWidth - this.spacing.Horizontal;
            int textHeight    = controlHeight - this.spacing.Vertical;

            // Measure text size
            if (this.MaxWidth > 0 && this.AutoSize == false)
            {
                int actualMaxWidth = this.MaxWidth -
                                     1 -
                                     this.spacing.Horizontal;

                SizeF textSize = g.MeasureString(
                    this.Text,
                    this.Font,
                    actualMaxWidth);

                if ((int)textSize.Height > textHeight)
                {
                    controlHeight = (int)textSize.Height + this.spacing.Vertical;
                    textHeight    = (int)textSize.Height;
                }

                this.Height = controlHeight + 1;
            }

            g.Dispose();
        }

        /// <summary>
        /// Compute the best size of the label to fit all the contents.
        /// </summary>
        /// <param name="maxWidth">Maximum width.</param>
        /// <returns>The best size.</returns>
        protected Size ComputeBestFitSize(int maxWidth)
        {
            Graphics g = this.CreateGraphics();
            g.SmoothingMode = SmoothingMode.HighQuality;

            SizeF textSize;

            if (maxWidth > 0)
            {
                int actualMaxWidth = maxWidth -
                                     1 -
                                     this.spacing.Horizontal;

                textSize = g.MeasureString(
                    this.Text,
                    this.Font,
                    actualMaxWidth);
            }
            else
            {
                textSize = g.MeasureString(
                    this.Text,
                    this.Font);
            }

            int textWidth     = (int)textSize.Width;
            int textHeight    = (int)textSize.Height;
            int controlWidth  = textWidth + this.spacing.Horizontal + 1;
            int controlHeight = textHeight + this.spacing.Vertical + 1;

            g.Dispose();

            return new Size(controlWidth, controlHeight);
        }

        /// <summary>
        /// Internal method to compute the best fit label size for the text.
        /// </summary>
        protected void ComputeAutoSize()
        {
            if (!this.myAutoSize)
            {
                this.AdjustHeightForMaxWidth();
                this.Invalidate();

                return;
            }

            Size size = this.ComputeBestFitSize(this.MaxWidth);

            if (size.Width == this.Width && size.Height == this.Height)
            {
                return;
            }

            this.ignoreSizeEvent = true;

            this.Size = size;

            this.ignoreSizeEvent = false;

            this.Invalidate();
        }

        #endregion
    }
}
