﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//// #define DEBUG_RENAME_UI

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Win32;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UILogic.ViewModels;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// 名前変更ボックスです.
    /// </summary>
    public class RenameEditBox : TextBox, IMessageFilter
    {
        /// <summary>
        /// リネームするツリーノードです.
        /// </summary>
        private UITreeNode node = null;

        /// <summary>
        /// リネームするビューモデルです.
        /// </summary>
        private HierarchyViewModel viewModel = null;

        /// <summary>
        /// リネームの開始時間です.
        /// リネーム開始直後のキャンセルイベントを無視するために使います.
        /// </summary>
        private DateTime renameBeginTime = new DateTime();

        /// <summary>
        /// リネーム処理中のフラグです.
        /// EndRenameが重複して呼び出されるのを防ぎます.
        /// </summary>
        private bool comittingRename = false;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public RenameEditBox()
        {
            this.Visible = false;
            this.BorderStyle = BorderStyle.FixedSingle;
            this.AutoSize = false;
            this.MaxLength = 128;

            //// this.BringToFront();
            //// this.SetOnFocusSelectAllEnabledOverride(false);
            this.KeyDown += new KeyEventHandler(this.OnKeyDown);
            this.Leave += new EventHandler(this.OnLeave);

            // メッセージフィルタを登録
            Application.AddMessageFilter(this);
        }

        /// <summary>
        /// リネームを開始します.
        /// </summary>
        /// <param name="node">リネームするノード</param>
        public void BeginRename(UITreeNode node)
        {
            #if DEBUG_RENAME_UI
                Logger.Log(LogLevels.Debug, "RenameEditBox.BeginRename()");
            #endif

            this.CancelRename();

            // ノードを取得
            this.node = node;
            if (this.node == null)
            {
                return;
            }

            // ビューモデルを取得
            this.viewModel = this.node.DataContext as HierarchyViewModel;
            if (this.viewModel == null)
            {
                return;
            }

            // テキストボックスの位置, サイズを調整
            Rectangle rect = this.node.NameRectangle;
            Point screenPos = this.node.TreeView.PointToScreen(rect.Location);
            Point clientPos = this.Parent.PointToClient(screenPos);

            this.Location = clientPos;
            this.Size = rect.Size;

            // テキストボックスのステータスを更新
            this.Text = this.node.Text;
            this.Visible = true;
            this.SelectionStart = 0;
            this.SelectionLength = 0;

            this.BringToFront();
            this.Focus();
            this.SelectAll();

            this.renameBeginTime = DateTime.Now;

            // リネーム中はメニューのショートカットを無効化する
            if (UIUserControl.DisableMenuShortcut != null)
            {
                UIUserControl.DisableMenuShortcut();
            }
        }

        /// <summary>
        /// リネームを終了します.
        /// </summary>
        /// <returns>処理が成功したときtrueを返します.</returns>
        public bool EndRename()
        {
            #if DEBUG_RENAME_UI
                Logger.Log(LogLevels.Debug, "RenameEditBox.EndRename()");
            #endif

            // 関数の同時実行をチェック
            // this.OnKeyDown(), this.OnLostFocus()から同時に呼ばれることがある
            if (this.comittingRename)
            {
                return true;
            }

            // BeginRename()が呼ばれたかチェック
            if (this.node == null)
            {
                return true;
            }

            this.comittingRename = true;

            // IMEを強制的に閉じる
            this.CloseIme();

            // 無効にしていたショートカットを復帰する
            if (UIUserControl.EnableMenuShortcut != null)
            {
                UIUserControl.EnableMenuShortcut();
            }

            // 変更がないとき、そのままリネームを終了
            if (this.Text == this.node.Text)
            {
                this.CancelRename();
                this.comittingRename = false;

                return true;
            }

            // 不正な名前が入力されたとき、テキストボックスの内容をリセットする
            // このときテキストボックスは閉じずに表示させ続ける
            if (this.viewModel.IsNameValid(this.Text) == false)
            {
                this.Reset();
                this.comittingRename = false;

                return false;  // リネームを継続するためfalseを返す
            }

            // リネームコマンドを発行
            Foundation.Command.CommandManager.Execute(new RenameCommand(this.viewModel, this.Text, this.node.Text));

            this.node = null;
            this.viewModel = null;
            this.Visible = false;

            this.comittingRename = false;

            return true;
        }

        /// <summary>
        /// リネームをキャンセルします.
        /// </summary>
        public void CancelRename()
        {
            #if DEBUG_RENAME_UI
                Logger.Log(LogLevels.Debug, "RenameEditBox.CancelRename()");
            #endif

            // リネーム開始直後に発生したキャンセル処理を無視する.
            if (this.Visible == true && (DateTime.Now - this.renameBeginTime).TotalMilliseconds <= 200)
            {
                this.Focus();
                return;
            }

            // IMEを閉じる
            this.CloseIme();

            // 無効にしていたショートカットを復帰する
            if (UIUserControl.EnableMenuShortcut != null)
            {
                UIUserControl.EnableMenuShortcut();
            }

            this.node = null;
            this.viewModel = null;
            this.Visible = false;
        }

        /// <summary>
        /// Applicationへのメッセージをハンドルします.
        /// </summary>
        /// <param name="m">メッセージ</param>
        /// <returns>メッセージをフィルタでカットするときtrueを返します.</returns>
        public bool PreFilterMessage(ref Message m)
        {
            // マウスのボタンが押されたときだけ処理
            if (m.Msg != (int)WM.WM_LBUTTONDOWN &&
                m.Msg != (int)WM.WM_RBUTTONDOWN &&
                m.Msg != (int)WM.WM_NCLBUTTONDOWN)
            {
                return false;
            }

            // 名前を変更しているときだけ処理
            if (this.Visible == false || this.node == null)
            {
                return false;
            }

            #if DEBUG_RENAME_UI
                Logger.Log(LogLevels.Debug, "RenameEditBox.PreFilterMessage()");
            #endif

            Point clientPos = Utility.LParamToSignedPoint(m.LParam);

            // 自身のUIがクリックしたときは何もしない
            if (m.HWnd == this.Handle &&
                this.ClientRectangle.Contains(clientPos))
            {
                return false;
            }

            // リネーム終了処理を実行
            // 特に、メニューボタンをクリックしたときはここからしか呼び出せない
            bool res = this.EndRename();

            // リネームを継続するときはメッセージをカットする
            if (res == false)
            {
                return true;
            }

            return false;
        }

        // IME操作用の関数です.
        [System.Runtime.InteropServices.DllImport("imm32.dll")]
        private static extern IntPtr ImmGetContext(IntPtr wnd);

        [System.Runtime.InteropServices.DllImport("Imm32.dll")]
        private static extern bool ImmReleaseContext(IntPtr wnd, IntPtr imcHandle);

        [System.Runtime.InteropServices.DllImport("Imm32.dll")]
        private static extern int ImmGetCompositionStringW(IntPtr imc, uint index, byte[] buf, int bufLen);

        [System.Runtime.InteropServices.DllImport("Imm32.dll")]
        private static extern bool ImmNotifyIME(IntPtr imc, uint action, uint index, uint value);

        /// <summary>
        /// テキストボックスの内容をリセットします.
        /// </summary>
        private void Reset()
        {
            #if DEBUG_RENAME_UI
                Logger.Log(LogLevels.Debug, "RenameEditBox.Reset()");
            #endif

            // テキストの内容をリセット
            this.Text = this.node.Text;

            // テキストの選択状態をリセット
            this.SelectionStart = 0;
            this.SelectionLength = 0;
            this.SelectAll();
        }

        /// <summary>
        /// IMEに入力中のテキストを取得して、強制的にIMEを閉じます.
        /// </summary>
        private void CloseIme()
        {
            const uint GCS_COMPSTR = 8;
            const int NI_COMPOSITIONSTR = 21;
            const int CPS_CANCEL = 4;

            // IMEコンテキストを取得
            IntPtr imeContext = ImmGetContext(this.Handle);

            try
            {
                // IMEに入力された文字数を取得
                int strLen = ImmGetCompositionStringW(imeContext, GCS_COMPSTR, null, 0);

                if (strLen > 0)
                {
                    byte[] buffer = new byte[strLen];

                    // IMEに入力された文字列を取得
                    ImmGetCompositionStringW(imeContext, GCS_COMPSTR, buffer, strLen);
                    string inputText = Encoding.Unicode.GetString(buffer);

                    // 文字列をカレットの位置に挿入
                    this.Text = this.Text.Substring(0, this.SelectionStart) +
                        inputText +
                        this.Text.Substring(this.SelectionStart + this.SelectionLength);
                }
            }
            finally
            {
                // IMEを閉じる
                ImmNotifyIME(imeContext, NI_COMPOSITIONSTR, CPS_CANCEL, 0);

                // IMEコンテキストを破棄
                ImmReleaseContext(this.Handle, imeContext);
            }
        }

        /// <summary>
        /// キーが押されたときの処理を行います.
        /// </summary>
        /// <param name="sender">Sender.</param>
        /// <param name="e">Event argument.</param>
        private void OnKeyDown(object sender, KeyEventArgs e)
        {
            if (this.Visible == false)
            {
                return;
            }

            if (e.KeyCode == Keys.Enter)
            {
                // Enterでリネームを終了
                #if DEBUG_RENAME_UI
                    Logger.Log(LogLevels.Debug, "RenameEditBox.OnKeyDown() - Enter");
                #endif

                this.EndRename();
            }
            else if (e.KeyCode == Keys.Escape)
            {
                // Escapeでリネームをキャンセル
                #if DEBUG_RENAME_UI
                    Logger.Log(LogLevels.Debug, "RenameEditBox.OnKeyDown() - Escape");
                #endif

                this.CancelRename();
            }
        }

        /// <summary>
        /// フォーカスを失ったときの処理を行います.
        /// </summary>
        /// <param name="sender">Sender</param>
        /// <param name="e">Event argument.</param>
        private void OnLeave(object sender, EventArgs e)
        {
            if (this.Visible == true)
            {
                // リネームを終了
                // 特に、タブキーでフォーカスを失ったときはここからしか呼び出せない
                #if DEBUG_RENAME_UI
                    Logger.Log(LogLevels.Debug, "RenameEditBox.OnLeave()");
                #endif

                this.EndRename();
            }
        }
    }
}
