﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Drawing.Text;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Windows.Forms.Layout;
using EffectMaker.BusinessLogic;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;
using EffectMaker.UILogic.Manager;

using BusinessLogicManagers = EffectMaker.BusinessLogic.Manager;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// 遅延イメージパネル
    /// </summary>
    public class DelayImagePanel : UIPanel
    {
        /// <summary>
        /// ラップタイプ変換テーブル
        /// </summary>
        private static readonly WrapType[] WrapTypeTable = { WrapType.Mirror, WrapType.Repate, WrapType.Clamp };

        /// <summary>
        /// ファイルパス
        /// </summary>
        private string filePath;

        /// <summary>
        /// 現在のイメージ
        /// </summary>
        private Image currentImage;

        /// <summary>
        /// 水平分割数
        /// </summary>
        private int patternDivU;

        /// <summary>
        /// 垂直分割数
        /// </summary>
        private int patternDivV;

        /// <summary>
        /// 水平ラップタイプ
        /// </summary>
        private int wrapTypeU;

        /// <summary>
        /// 垂直ラップタイプ
        /// </summary>
        private int wrapTypeV;

        /// <summary>
        /// 繰り返し数
        /// </summary>
        private int repeatCount;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DelayImagePanel()
        {
            OptionStore.OptionChanged += this.OnGammaCorrectionChanged;
        }

        /// <summary>
        /// コンポーネント種類
        /// </summary>
        public enum ComponentKind
        {
            /// <summary>
            /// RGB
            /// </summary>
            Rgb,

            /// <summary>
            /// アルファ
            /// </summary>
            Alpha
        }

        /// <summary>
        /// ラップタイプ
        /// </summary>
        private enum WrapType
        {
            /// <summary>
            /// ミラー
            /// </summary>
            Mirror,

            /// <summary>
            /// リピート
            /// </summary>
            Repate,

            /// <summary>
            /// クランプ
            /// </summary>
            Clamp
        }

        /// <summary>
        /// ファイルパス
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }

            set
            {
                this.filePath = value;

                this.UpdateImage(this.filePath);
            }
        }

        /// <summary>
        /// 再読み込み要求
        /// </summary>
        public bool RequestReload
        {
            get
            {
                return false;
            }

            set
            {
                if (value)
                {
                    this.UpdateImage(this.FilePath);
                }
            }
        }

        /// <summary>
        /// コンポーネント
        /// </summary>
        public ComponentKind Component { get; set; }

        /// <summary>
        /// 現在のイメージ
        /// </summary>
        public Image CurrentImage
        {
            get
            {
                return this.currentImage;
            }

            set
            {
                this.currentImage = value;
                this.UpdateViewImage();
            }
        }

        /// <summary>
        /// 水平分割数
        /// </summary>
        public int PatternDivU
        {
            get
            {
                return this.patternDivU;
            }

            set
            {
                this.patternDivU = value;
                this.UpdateViewImage();
            }
        }

        /// <summary>
        /// 垂直分割数
        /// </summary>
        public int PatternDivV
        {
            get
            {
                return this.patternDivV;
            }

            set
            {
                this.patternDivV = value;
                this.UpdateViewImage();
            }
        }

        /// <summary>
        /// 水平ラップタイプ
        /// </summary>
        public int WrapTypeU
        {
            get
            {
                return this.wrapTypeU;
            }

            set
            {
                this.wrapTypeU = value;
                this.UpdateViewImage();
            }
        }

        /// <summary>
        /// 垂直ラップタイプ
        /// </summary>
        public int WrapTypeV
        {
            get
            {
                return this.wrapTypeV;
            }

            set
            {
                this.wrapTypeV = value;
                this.UpdateViewImage();
            }
        }

        /// <summary>
        /// 繰り返し数
        /// </summary>
        public int RepeatCount
        {
            get
            {
                return this.repeatCount;
            }

            set
            {
                this.repeatCount = value;
                this.UpdateViewImage();
            }
        }

        /// <summary>
        /// 現在のピクセルフォーマット
        /// </summary>
        private PixelFormats CurrentPixelFormat
        {
            get;
            set;
        }

        /// <summary>
        /// SRGBフォーマットか？
        /// </summary>
        private bool IsSrgbFormat
        {
            get
            {
                return (this.CurrentPixelFormat == PixelFormats.Srgb_8_8_8_8) ||
                       (this.CurrentPixelFormat == PixelFormats.Srgb_bc1) ||
                       (this.CurrentPixelFormat == PixelFormats.Srgb_bc2) ||
                       (this.CurrentPixelFormat == PixelFormats.Srgb_bc3) ||
                       (this.CurrentPixelFormat == PixelFormats.Srgb_bc7);
            }
        }

        /// <summary>
        /// <see cref="T:System.Windows.Forms.Control"/> とその子コントロールが使用しているアンマネージ リソースを解放します。オプションで、マネージ リソースも解放します。
        /// </summary>
        /// <param name="disposing">マネージ リソースとアンマネージ リソースの両方を解放する場合は true。アンマネージ リソースだけを解放する場合は false。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                OptionStore.OptionChanged -= this.OnGammaCorrectionChanged;
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// 指定でビットマップを作る
        /// </summary>
        /// <param name="src">元ビットマップ</param>
        /// <param name="cm">カラー変換行列</param>
        /// <returns>作ったビットマップ</returns>
        private static Bitmap MakeImageBitmap(Bitmap src, ColorMatrix cm)
        {
            var dst = new Bitmap(src.Width, src.Height, PixelFormat.Format32bppArgb);
            {
                using (var ia = new ImageAttributes())
                using (var g = Graphics.FromImage(dst))
                {
                    ia.SetColorMatrix(cm);

                    g.DrawImage(
                        src,
                        new Rectangle(0, 0, src.Width, src.Height),
                        0,
                        0,
                        src.Width,
                        src.Height,
                        GraphicsUnit.Pixel,
                        ia);
                }
            }

            return dst;
        }

        /// <summary>
        /// メッセージ用イメージか？
        /// </summary>
        /// <param name="image">イメージ</param>
        /// <returns>メッセージ用イメージであればtrue</returns>
        private static bool IsMessageImage(Image image)
        {
            return
                (image == Properties.Resources.Image_TexturePreviewLoading) ||
                (image == Properties.Resources.Image_TexturePreviewError);
        }

        /// <summary>
        /// 元イメージをリピートを考慮して描画する
        /// </summary>
        /// <param name="dst">ターゲット</param>
        /// <param name="src">元イメージ</param>
        /// <param name="repeatH">水平繰り返し数</param>
        /// <param name="repeatV">垂直繰り返し数</param>
        /// <param name="wrapTypeH">水平ラップタイプ</param>
        /// <param name="wrapTypeV">垂直ラップタイプ</param>
        private static void DrawRepatedImage(Graphics dst, Image src, int repeatH, int repeatV, WrapType wrapTypeH, WrapType wrapTypeV)
        {
            var cellW = src.Width / Math.Max(repeatH, repeatV);
            var cellH = src.Height / Math.Max(repeatH, repeatV);

            var srcRect = new Rectangle(0, 0, src.Width - 1, src.Height - 1);

            for (var v = 0; v != repeatV; ++v)
            {
                for (var h = 0; h != repeatH; ++h)
                {
                    var p0 = new Point((h + 0) * cellW, (v + 0) * cellH);
                    var p1 = new Point((h + 1) * cellW, (v + 0) * cellH);
                    var p2 = new Point((h + 0) * cellW, (v + 1) * cellH);
                    var p3 = new Point((h + 1) * cellW, (v + 1) * cellH);

                    var isClamped = false;

                    if (wrapTypeH == WrapType.Clamp)
                    {
                        if (h == 1)
                        {
                            isClamped = true;
                        }
                    }
                    else
                    {
                        if ((h & 1) == 1)
                        {
                            if (wrapTypeH == WrapType.Mirror)
                            {
                                Swap(ref p0, ref p1);
                                Swap(ref p2, ref p3);
                            }
                        }
                    }

                    if (wrapTypeV == WrapType.Clamp)
                    {
                        if (v == 1)
                        {
                            isClamped = true;
                        }
                    }
                    else
                    {
                        if ((v & 1) == 1)
                        {
                            if (wrapTypeV == WrapType.Mirror)
                            {
                                Swap(ref p0, ref p2);
                                Swap(ref p1, ref p3);
                            }
                        }
                    }

                    if (isClamped)
                    {
                        dst.FillRectangle(Brushes.LightGray, (h + 0) * cellW, (v + 0) * cellH, cellW, cellH);
                    }
                    else
                    {
                        dst.DrawImage(src, new[] { p0, p1, p2 }, srcRect, GraphicsUnit.Pixel);
                    }
                }
            }
        }

        /// <summary>
        /// 分割数を考慮して区切り線を描画する
        /// </summary>
        /// <param name="g">ターゲット</param>
        /// <param name="divH">水平分割数</param>
        /// <param name="divV">垂直分割数</param>
        private static void DrawDivideLine(Graphics g, int divH, int divV)
        {
            // return;
            var width = g.VisibleClipBounds.Width;
            var height = g.VisibleClipBounds.Height;

            using (var pen = new Pen(Color.FromArgb(255, 71, 91, 128), 2))
            {
                for (var i = 1; i < divH; ++i)
                {
                    var x = i * width / divH;
                    g.DrawLine(pen, x, 0, x, height);
                }

                for (var i = 1; i < divV; ++i)
                {
                    var y = i * height / divV;
                    g.DrawLine(pen, 0, y, width, y);
                }
            }
        }

        /// <summary>
        /// 分割数を考慮してインデックスを描画する
        /// </summary>
        /// <param name="g">ターゲット</param>
        /// <param name="divH">水平分割数</param>
        /// <param name="divV">垂直分割数</param>
        private static void DrawDivideIndex(Graphics g, int divH, int divV)
        {
            var width = g.VisibleClipBounds.Width;
            var height = g.VisibleClipBounds.Height;
            var cellW = (int)(width / divH);
            var cellH = (int)(height / divV);

            if (cellW < 10)
            {
                return;
            }

            using (var font = new Font(FontFamily.GenericSansSerif, 8))
            {
                var index = 0;

                g.TextRenderingHint = TextRenderingHint.AntiAlias;

                for (var y = 0; y != divV; ++y)
                {
                    for (var x = 0; x != divH; ++x, ++index)
                    {
                        var indexStr = index.ToString(CultureInfo.InvariantCulture);

                        if ((cellW < 20) && (index >= 10))
                        {
                            // 幅が狭いので文字単位で改行する
                            indexStr = string.Join("\n", indexStr.Select(c => c));
                        }

                        var x0 = (int)((x + 0) * width / divH);
                        var y0 = (int)((y + 0) * height / divV);
                        var x1 = (int)((x + 1) * width / divH);
                        var y1 = (int)((y + 1) * height / divV);

                        var cellRect = new Rectangle(x0, y0, x1 - x0 + 2, y1 - y0 + 2);

                        TextRenderer.DrawText(
                                g,
                                indexStr,
                                font,
                                cellRect,
                                Color.Black,
                                TextFormatFlags.HorizontalCenter | TextFormatFlags.VerticalCenter);
                        --cellRect.X;
                        --cellRect.Y;
                        TextRenderer.DrawText(
                                g,
                                indexStr,
                                font,
                                cellRect,
                                Color.White,
                                TextFormatFlags.HorizontalCenter | TextFormatFlags.VerticalCenter);
                    }
                }
            }
        }

        /// <summary>
        /// ポイントをスワップ
        /// </summary>
        /// <param name="p0">ポイント０</param>
        /// <param name="p1">ポイント１</param>
        private static void Swap(ref Point p0, ref Point p1)
        {
            var temp = p0;
            p0 = p1;
            p1 = temp;
        }

        /// <summary>
        /// イメージを更新
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        private void UpdateImage(string filePath)
        {
            switch (this.Component)
            {
                case ComponentKind.Rgb:
                    this.UpdateRgbImage(filePath);
                    break;

                case ComponentKind.Alpha:
                    this.UpdateAlphaImage(filePath);
                    break;
            }
        }

        /// <summary>
        /// RGBイメージを更新する
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        private void UpdateRgbImage(string filePath)
        {
            this.UpdateImageInternal(filePath, this.CreateRgbImage);
        }

        /// <summary>
        /// アルファイメージを更新する
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        private void UpdateAlphaImage(string filePath)
        {
            this.UpdateImageInternal(filePath, this.CreateAlphaImage);
        }

        /// <summary>
        /// イメージを更新する
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="createImage">イメージ作成</param>
        private void UpdateImageInternal(string filePath, Func<FtxTextureData, Bitmap> createImage)
        {
            if (string.IsNullOrEmpty(filePath))
            {
                // 未指定
                this.SetImage(null);
            }
            else if (BusinessLogicManagers.TextureManager.Instance.CanLoad(filePath) == false)
            {
                // 読み込めない
                this.SetImage(Properties.Resources.Image_TexturePreviewError);
            }
            else
            {
                if (BusinessLogicManagers.TextureManager.Instance.IsCached(filePath))
                {
                    //// キャッシュ済みなので裏読みしないで直接読みにいく
                    var resultWithData =
                        BusinessLogicManagers.TextureManager.Instance.LoadTexture(filePath, true);

                    var image = resultWithData.ResultCode == LoadTextureResultCode.Success ?
                        createImage((FtxTextureData)resultWithData.TextureData) :
                        Properties.Resources.Image_TexturePreviewError;

                    this.SetImage(image);
                }
                else
                {
                    // 読み込み中にする
                    this.SetImage(Properties.Resources.Image_TexturePreviewLoading);

                    // 裏読み
                    Task.Factory.StartNew(() => BusinessLogicManagers.TextureManager.Instance.LoadTexture(filePath, true))

                    // 裏読み終了時
                    .ContinueWith(task =>
                    {
                        var resultWithData = task.Result;

                        this.BeginInvoke(new MethodInvoker(() =>
                        {
                            var image = resultWithData.ResultCode == LoadTextureResultCode.Success
                                ? createImage((FtxTextureData)resultWithData.TextureData)
                                : Properties.Resources.Image_TexturePreviewError;

                            this.SetImage(image);
                        }));
                    });
                }
            }
        }

        /// <summary>
        /// RGBイメージを作る
        /// </summary>
        /// <param name="textureData">テクスチャデータ</param>
        /// <returns>イメージ</returns>
        private Bitmap CreateRgbImage(FtxTextureData textureData)
        {
            this.CurrentPixelFormat = textureData.PixelFormat;

            var matrix = RenderUtility.CreateRgbColorMatrix(textureData.CompSel, textureData.PixelFormat);

            using (var bitmap = textureData.GeneratePreviewBitmapWithResize(this.Size, 0, matrix))
            {
                return MakeImageBitmap(bitmap, matrix);
            }
        }

        /// <summary>
        /// アルファイメージを作る
        /// </summary>
        /// <param name="textureData">テクスチャデータ</param>
        /// <returns>イメージ</returns>
        private Bitmap CreateAlphaImage(FtxTextureData textureData)
        {
            this.CurrentPixelFormat = textureData.PixelFormat;

            using (var bitmap = textureData.GeneratePreviewBitmapWithResize(this.Size, 0))
            {
                var matrix = RenderUtility.CreateAlphaColorMatrix(textureData.CompSel, textureData.PixelFormat);

                return MakeImageBitmap(bitmap, matrix);
            }
        }

        /// <summary>
        /// イメージを設定する
        /// </summary>
        /// <param name="image">設定するイメージ</param>
        private void SetImage(Image image)
        {
            this.CurrentImage = image;
        }

        /// <summary>
        /// Handle GammaCorrectionChanged event from application.
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void OnGammaCorrectionChanged(object sender, EventArgs e)
        {
            this.UpdateViewImage();
        }

        /// <summary>
        /// ビュー用イメージの更新
        /// </summary>
        private void UpdateViewImage()
        {
            if (this.CurrentImage == null)
            {
                this.BackgroundImage = this.MakeViewImage(null);
                return;
            }

            // ガンマを考慮しない
            {
                if (this.Component != ComponentKind.Rgb)
                {
                    this.BackgroundImage = this.MakeViewImage(this.CurrentImage);
                    return;
                }

                if (IsMessageImage(this.CurrentImage))
                {
                    this.BackgroundImage = this.MakeViewImage(this.CurrentImage);
                    return;
                }

                if (ColorUtility.Gamma == 1.0)
                {
                    this.BackgroundImage = this.MakeViewImage(this.CurrentImage);
                    return;
                }
            }

            // ガンマを考慮したイメージを作る
            {
                var tempImage = new Bitmap(this.CurrentImage.Width, this.CurrentImage.Height);

                using (var tempGr = Graphics.FromImage(tempImage))
                using (var attr = new ImageAttributes())
                {
                    if (this.IsSrgbFormat)
                    {
                        if (ColorUtility.IsGammaCorrectionEnabled == false)
                        {
                            attr.SetGamma((float)ColorUtility.Gamma);
                        }
                    }
                    else
                    {
                        if (ColorUtility.IsGammaCorrectionEnabled)
                        {
                            attr.SetGamma((float)(1.0 / ColorUtility.Gamma));
                        }
                    }

                    var rect = new Rectangle(0, 0, this.CurrentImage.Width, this.CurrentImage.Height);

                    tempGr.DrawImage(
                            this.CurrentImage,
                            rect,
                            0.0f,
                            0.0f,
                            rect.Width,
                            rect.Height,
                            GraphicsUnit.Pixel,
                            attr);
                }

                this.BackgroundImage = this.MakeViewImage(tempImage);
            }
        }

        /// <summary>
        /// 割数、繰り返し、ラップを考慮して表示用イメージを作る。
        /// </summary>
        /// <param name="src">元イメージ</param>
        /// <returns>作成済みイメージ</returns>
        private Image MakeViewImage(Image src)
        {
            if (src == null)
            {
                return null;
            }

            var tempImage = new Bitmap(src.Width, src.Height);

            using (var g = Graphics.FromImage(tempImage))
            {
                // 元イメージをリピートを考慮して描画する
                DrawRepatedImage(g, src, ((this.RepeatCount >> 0) & 1) + 1, ((this.RepeatCount >> 1) & 1) + 1, WrapTypeTable[this.WrapTypeU], WrapTypeTable[this.WrapTypeV]);

                // 分割数を考慮して区切り線を描画する
                DrawDivideLine(g, this.PatternDivU, this.PatternDivV);

                // 分割数を考慮してインデックスを描画する
                DrawDivideIndex(g, this.PatternDivU, this.PatternDivV);
            }

            return tempImage;
        }
    }
}
