﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.Render.Renderable;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// エミッタ時間アニメ用の非アクティブカーブ描画管理クラスです。
    /// </summary>
    public class UnderlayCurveDrawer
    {
        /// <summary>
        /// ビューポート
        /// </summary>
        private readonly LayeredViewports viewports;

        /// <summary>
        /// 時間からX座標
        /// </summary>
        private readonly Func<float, float> calcPosFromTime;

        /// <summary>
        /// 値からY座標
        /// </summary>
        private readonly Func<float, float> calcPosFromValue;

        /// <summary>
        /// 描画処理
        /// </summary>
        private readonly Action drawAction;

        /// <summary>
        /// エミッタ時間アニメモード時の他要素のカーブ
        /// </summary>
        private readonly List<RenderableBase> underlayLines = new List<RenderableBase>();

        /// <summary>
        /// エミッタ時間アニメモード時の他要素のデータソース
        /// </summary>
        private readonly Dictionary<int, UnderlayDataSource> underlaySources = new Dictionary<int, UnderlayDataSource>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        /// <param name="calcPosFromTime">時間からX座標</param>
        /// <param name="calcPosFromValue">値からY座標</param>
        /// <param name="drawAction">描画処理</param>
        internal UnderlayCurveDrawer(
            LayeredViewports viewports,
            Func<float, float> calcPosFromTime,
            Func<float, float> calcPosFromValue,
            Action drawAction)
        {
            this.viewports = viewports;
            this.calcPosFromTime = calcPosFromTime;
            this.calcPosFromValue = calcPosFromValue;
            this.drawAction = drawAction;
        }

        /// <summary>
        /// 薄く表示するカーブを追加します
        /// </summary>
        /// <param name="hashCode">ViewModel由来のハッシュコード</param>
        /// <param name="data">消去するデータソース</param>
        /// <param name="channelNum">チャンネル数</param>
        /// <param name="interpolationMode">補間タイプ</param>
        public void AddUnderlayCurve(int hashCode, AnimationTableData data, int channelNum, int interpolationMode)
        {
            if (!this.underlaySources.ContainsKey(hashCode))
            {
                var source = new UnderlayDataSource
                {
                    HashCode = hashCode,
                    ChannelNum = channelNum,
                    InterpolationMode = interpolationMode,
                    Table = data,
                };
                this.underlaySources.Add(hashCode, source);
            }

            this.UpdateUnderlayCurves();
        }

        /// <summary>
        /// 薄く表示するカーブを選択して消去します
        /// </summary>
        /// <param name="hashCode">消去するデータソースを表すハッシュコード</param>
        public void RemoveUnderlayCurve(int hashCode)
        {
            if (this.underlaySources.ContainsKey(hashCode))
            {
                this.underlaySources.Remove(hashCode);
            }

            this.UpdateUnderlayCurves();
        }

        /// <summary>
        /// 薄く表示するカーブをクリアします
        /// </summary>
        /// <param name="sourceClear">データソースごとクリアするならtrue</param>
        public void ClearUnderlayCurve(bool sourceClear)
        {
            this.viewports.Main.RemoveRenderableRange(this.underlayLines);
            foreach (var line in this.underlayLines)
            {
                line.Dispose();
            }

            this.underlayLines.Clear();
            if (sourceClear)
            {
                this.underlaySources.Clear();
            }
        }

        /// <summary>
        /// 薄く表示するカーブを更新します
        /// </summary>
        private void UpdateUnderlayCurves()
        {
            this.ClearUnderlayCurve(false);
            foreach (var data in this.underlaySources)
            {
                var source = data.Value;
                if (source.InterpolationMode == 0)
                {
                    KeyFrameData prevKey = null;
                    foreach (var keyFrame in source.Table)
                    {
                        if (prevKey == null)
                        {
                            prevKey = keyFrame;
                            continue;
                        }

                        for (int i = 0; i < source.ChannelNum; ++i)
                        {
                            var line = new LineSegment
                            {
                                BorderColor = Color.Gainsboro,
                                BorderThickness = 1,
                                Vertex1 = new PointF(
                                    this.calcPosFromTime(prevKey.Frame),
                                    this.calcPosFromValue(prevKey.Value[i])),
                                Vertex2 = new PointF(
                                    this.calcPosFromTime(keyFrame.Frame),
                                    this.calcPosFromValue(keyFrame.Value[i])),
                            };
                            this.underlayLines.Add(line);
                            this.viewports.Main.AddRenderable(line);
                        }

                        prevKey = keyFrame;
                    }
                }
                else
                {
                    int channelNum = source.ChannelNum;
                    var backCurves = new CurveRenderable[channelNum];
                    for (int i = 0; i < channelNum; ++i)
                    {
                        backCurves[i] = new CurveRenderable
                        {
                            BorderColor = Color.Gainsboro,
                            BorderThickness = 1,
                            Vertices = new List<PointF>()
                        };
                    }

                    foreach (var keyFrame in source.Table)
                    {
                        for (int i = 0; i < channelNum; ++i)
                        {
                            backCurves[i].Vertices.Add(new PointF(
                                this.calcPosFromTime(keyFrame.Frame),
                                this.calcPosFromValue(keyFrame.Value[i])));
                        }
                    }

                    this.underlayLines.AddRange(backCurves);
                    this.viewports.Main.AddRenderableRange(backCurves);
                }
            }

            this.drawAction();
        }
    }
}
