﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// カーブの操作を担当するクラスです。
    /// </summary>
    internal class CurveManupirator
    {
        /// <summary>
        /// カーブリスト
        /// </summary>
        private readonly List<CurveShape> curves = new List<CurveShape>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="editor"></param>
        public CurveManupirator(CurveEditor editor)
        {
            this.Initialize(editor, 4);
        }

        /// <summary>
        /// 指定したチャンネル数でカーブリストを初期化します。
        /// </summary>
        /// <param name="editor"></param>
        /// <param name="channelNum"></param>
        public void Initialize(CurveEditor editor, int channelNum)
        {
            this.curves.Clear();
            for (int i = 0; i < channelNum; ++i)
            {
                this.curves.Add(new CurveShape(
                    editor,
                    Constants.NodeColors[i],
                    Constants.BorderColors[i],
                    Constants.DeactiveColors[i]));
            }
        }

        /// <summary>
        /// アニメーションテーブルにカーブの状態を記録します。
        /// </summary>
        /// <param name="table">書き込むアニメーションテーブル</param>
        /// <param name="targetNode">現在選択しているノード</param>
        public void SetDataToTable(AnimationTableData table, RectangleShape targetNode)
        {
            // 選択キーインデックスの抽出
            int selectedIndex = -1;
            foreach (var curve in this.curves)
            {
                selectedIndex = curve.GetIndex(targetNode);
                if (selectedIndex != -1)
                {
                    break;
                }
            }

            table.Clear();
            var keyNum = this.curves[0].GetNodes().Count;
            var curveValues = this.curves.Select(curve => curve.GetValues()).ToArray();
            for (int i = 0; i < keyNum; ++i)
            {
                int frame = (int)curveValues[0][i].X;
                var value = new Vector4f();
                for (int j = 0; j < curveValues.Length; ++j)
                {
                    value[j] = curveValues[j][i].Y;
                }

                table.AddKeyFrame(frame, value, i == selectedIndex);
            }
        }

        /// <summary>
        /// アニメーションテーブルの情報を元にカーブの状態を形成します。
        /// </summary>
        /// <param name="table">読み込むアニメーションテーブル</param>
        /// <param name="viewports">ビューポート</param>
        /// <param name="checkboxList">編集対象軸チェックボックスリスト</param>
        /// <param name="calcPosFromTime">時間からX座標を得る関数</param>
        /// <param name="calcPosFromValue">値からY座標を得る関数</param>
        public void SetCurvesFromTable(
            AnimationTableData table,
            LayeredViewports viewports,
            List<UICheckBox> checkboxList,
            Func<float, float> calcPosFromTime,
            Func<float, float> calcPosFromValue)
        {
            List<KeyFrameData> keyFrameList = table.GetList();
            for (int j = 0; j < keyFrameList.Count; ++j)
            {
                float frame = calcPosFromTime(keyFrameList[j].Frame);
                for (int i = 0; i < this.curves.Count; ++i)
                {
                    if (j == 0 && this.curves[i].GetNodes().Count > 0)
                    {
                        this.curves[i].MoveKey(
                            viewports,
                            this.curves[i].GetNodes()[0],
                            viewports.Main.InverseTransformPoint(new PointF(frame, calcPosFromValue(keyFrameList[j].Value[i]))),
                            true);
                    }
                    else
                    {
                        this.curves[i].AddKey(
                            viewports,
                            viewports.Main.InverseTransformPoint(new PointF(
                                frame,
                                calcPosFromValue(keyFrameList[j].Value[i]))),
                            checkboxList[i].Checked,
                            true);
                    }
                }
            }
        }

        /// <summary>
        /// カーブ全体で専有している矩形領域を得ます。
        /// </summary>
        /// <param name="minPos">最小のX,Y座標[in, out]</param>
        /// <param name="maxPos">最大のX,Y座標[in, out]</param>
        public void GetUsingRectangleXY(ref PointF minPos, ref PointF maxPos)
        {
            foreach (var curve in this.curves)
            {
                var nodes = curve.GetNodes();
                foreach (var node in nodes)
                {
                    if (maxPos.Y < curve.GetKeyPos(node).Y)
                    {
                        maxPos.Y = curve.GetKeyPos(node).Y;
                    }

                    if (minPos.Y > curve.GetKeyPos(node).Y)
                    {
                        minPos.Y = curve.GetKeyPos(node).Y;
                    }

                    if (maxPos.X < curve.GetKeyPos(node).X)
                    {
                        maxPos.X = curve.GetKeyPos(node).X;
                    }

                    if (minPos.X > curve.GetKeyPos(node).X)
                    {
                        minPos.X = curve.GetKeyPos(node).X;
                    }
                }
            }
        }

        /// <summary>
        /// カーブ全体で専有している矩形領域を得ます。
        /// </summary>
        /// <param name="minPos">最小のX,Y座標[in, out]</param>
        /// <param name="maxPos">最大のX,Y座標[in, out]</param>
        public void GetUsingRectangleY(ref PointF minPos, ref PointF maxPos)
        {
            foreach (var curve in this.curves)
            {
                var nodes = curve.GetNodes();
                foreach (var node in nodes)
                {
                    if (maxPos.Y < curve.GetKeyPos(node).Y)
                    {
                        maxPos.Y = curve.GetKeyPos(node).Y;
                    }

                    if (minPos.Y > curve.GetKeyPos(node).Y)
                    {
                        minPos.Y = curve.GetKeyPos(node).Y;
                    }
                }
            }
        }

        /// <summary>
        /// キーを追加します。
        /// </summary>
        /// <param name="pos">座標値</param>
        /// <param name="viewports">ビューポート</param>
        /// <param name="checkboxList">チェックボックスリスト</param>
        /// <param name="afterAddAction">1チャンネル追加するごとに実行するアクション</param>
        public void AddKey(
            PointF pos,
            LayeredViewports viewports,
            List<UICheckBox> checkboxList,
            Action<RectangleShape, int> afterAddAction)
        {
            for (int i = 0; i < this.curves.Count; ++i)
            {
                var key = this.curves[i].AddKey(viewports, pos, checkboxList[i].Checked);
                if (afterAddAction != null)
                {
                    afterAddAction(key, i);
                }
            }
        }

        /// <summary>
        /// キーを移動します。
        /// </summary>
        /// <param name="targetNode">プライマリー選択ノード</param>
        /// <param name="selectedNodes">選択ノードリスト</param>
        /// <param name="argLoc">移動先座標</param>
        /// <param name="viewports">ビューポート</param>
        /// <returns></returns>
        public PointF MoveKey(RectangleShape targetNode, List<RectangleShape> selectedNodes, PointF argLoc, LayeredViewports viewports)
        {
            // 現在選択されているキーのチャンネルとインデックスを得る
            int channel;
            int index = this.GetIndicesByNode(out channel, targetNode);

            if (index == -1 || channel == -1)
            {
                return default(PointF);
            }

            // プライマリ選択キーを移動
            var nodePos = this.curves[channel].MoveKey(viewports, targetNode, argLoc);

            // 非選択チャンネルは時間軸だけ移動
            foreach (var curve in this.curves)
            {
                if (this.curves.IndexOf(curve) == channel)
                {
                    continue;
                }

                curve.MoveKeyFrameCombined(viewports, index, argLoc);
            }

            // 選択しているキーをまとめて移動
            foreach (var selected in selectedNodes)
            {
                int ch = -1;
                foreach (var curve in this.curves)
                {
                    ch = curve.GetIndex(selected);
                    if (ch != -1)
                    {
                        ch = this.curves.IndexOf(curve);
                        break;
                    }
                }

                if (ch != -1)
                {
                    this.curves[ch].MoveKey(viewports, selected, argLoc);
                }
            }

            return nodePos;
        }

        /// <summary>
        /// キーを削除します。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        /// <param name="targetNode">削除対象ノード</param>
        /// <returns></returns>
        public RectangleShape RemoveKey(LayeredViewports viewports, RectangleShape targetNode)
        {
            int channel;
            int index = this.GetIndicesByNode(out channel, targetNode);

            if (index == -1)
            {
                return null;
            }

            foreach (var curve in this.curves)
            {
                curve.RemoveKey(viewports, index);
            }

            var nodes = this.curves[channel].GetNodes();
            if (index < nodes.Count)
            {
                return nodes[index];
            }
            else if (index - 1 > 0)
            {
                return nodes[index - 1];
            }

            return null;
        }

        /// <summary>
        /// 線分の再接続を行います。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        public void Reconnect(LayeredViewports viewports)
        {
            foreach (var curve in this.curves)
            {
                curve.Reconnect(viewports);
            }
        }

        /// <summary>
        /// カーブに対するスケールの再設定を行います。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        public void Rescale(LayeredViewports viewports)
        {
            foreach (var curve in this.curves)
            {
                curve.Rescale(viewports);
            }
        }

        /// <summary>
        /// カーブを初期状態にリセットします。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        public void Reset(LayeredViewports viewports)
        {
            foreach (var curve in this.curves)
            {
                curve.Reset(viewports);
            }
        }

        /// <summary>
        /// 指定したインデックスと編集対象軸選択状態に応じた、プライマリー選択ノードを取得します。
        /// </summary>
        /// <param name="selectedIndex">キーのインデックス</param>
        /// <param name="checkboxList">チェックボックスリスト</param>
        /// <returns>プライマリー選択ノード</returns>
        public RectangleShape GetNodeAsTarget(int selectedIndex, List<UICheckBox> checkboxList)
        {
            RectangleShape targetNode = null;
            if (selectedIndex != -1)
            {
                for (int i = 0; i < this.curves.Count; ++i)
                {
                    if (checkboxList[i].Checked)
                    {
                        targetNode = this.curves[i].GetNodes()[selectedIndex];
                        break;
                    }
                }
            }

            return targetNode;
        }

        /// <summary>
        /// カーブのチャンネルとキーのインデックスを指定してノードを取得します。
        /// </summary>
        /// <param name="curveIndex">カーブのチャンネル</param>
        /// <param name="keyIndex">キーインデックス</param>
        /// <returns>ノード</returns>
        public RectangleShape GetNodeByIndices(int curveIndex, int keyIndex)
        {
            if (this.curves == null ||
                this.curves.Count <= curveIndex ||
                this.curves[0].GetNodes().Count <= keyIndex)
            {
                return null;
            }

            return this.curves[curveIndex].GetNodes()[keyIndex];
        }

        /// <summary>
        /// ピックによるノードの選択を行います。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        /// <param name="argLoc">マウスによってピックした座標</param>
        /// <param name="checkboxList">チェックボックスリスト</param>
        /// <returns>選択ノードリスト</returns>
        public List<RectangleShape> PickNodes(LayeredViewports viewports, PointF argLoc, List<UICheckBox> checkboxList)
        {
            var selectedNodes = new List<RectangleShape>();
            for (int i = 0; i < this.curves.Count; ++i)
            {
                var node = this.curves[i].Selection(viewports, argLoc);
                if (node != null)
                {
                    if (checkboxList[i].Checked)
                    {
                        selectedNodes.Add(node);
                    }
                }
            }

            return selectedNodes;
        }

        /// <summary>
        /// 指定したインデックス以外のキーで、指定した値に一番近いものを返します。
        /// 値のスナップ用です。
        /// </summary>
        /// <param name="index">操作しているキーのインデックス</param>
        /// <param name="value">操作しているキーの値</param>
        /// <param name="checkboxList">チェックボックスリスト</param>
        /// <returns>スナップの基準とすべき直近の値</returns>
        public float GetNearestValue(int index, float value, List<UICheckBox> checkboxList)
        {
            bool first = true;
            float nearest = 0.0f;
            for (int i = 0; i < this.curves.Count; ++i)
            {
                if (!checkboxList[i].Checked)
                {
                    continue;
                }

                var values = this.curves[i].GetValues().ToArray();
                for (int j = 0; j < values.Length; ++j)
                {
                    if (j != -1 && j == index)
                    {
                        continue;
                    }

                    if (first ||
                        Math.Abs(values[j].Y - value) <
                        Math.Abs(nearest - value))
                    {
                        nearest = values[j].Y;
                        first = false;
                    }
                }
            }

            return nearest;
        }

        /// <summary>
        /// カーブをクリアします。
        /// </summary>
        /// <param name="viewports">ビューポート</param>
        public void Clear(LayeredViewports viewports)
        {
            foreach (var curve in this.curves)
            {
                curve.HideAll(viewports);
            }

            this.curves.Clear();
        }

        /// <summary>
        /// ノードの座標をカーブデータストアから取得します。
        /// </summary>
        /// <param name="argNode">ノード</param>
        /// <returns>座標</returns>
        public PointF GetNodePos(RectangleShape argNode)
        {
            foreach (var curve in this.curves.Where(curve => curve.GetIndex(argNode) != -1))
            {
                return curve.GetKeyPos(argNode);
            }

            return default(PointF);
        }

        /// <summary>
        /// 選択しているノードのインデックスとチャンネルを取得します。
        /// </summary>
        /// <param name="outCh">[OUT]選択したチャンネル</param>
        /// <param name="targetNode">選択ノード</param>
        /// <returns>選択したインデックス</returns>
        public int GetIndicesByNode(out int outCh, RectangleShape targetNode)
        {
            int index = -1, channel = -1;
            if (targetNode != null)
            {
                foreach (var curve in this.curves)
                {
                    index = curve.GetIndex(targetNode);
                    if (index != -1)
                    {
                        channel = this.curves.IndexOf(curve);
                        break;
                    }
                }
            }

            outCh = channel;
            return index;
        }

        /// <summary>
        /// カーブの描画順をソートして調整します。
        /// </summary>
        public void SortCurves(LayeredViewports viewports, int interMode, List<UICheckBox> checkboxList)
        {
            var drawnLines = new List<LineSegment>();

            if (interMode == 0)
            {
                // ラインは同一座標に重なるものをスキップするので、手前から描く
                int count = 0;
                foreach (var t in this.curves)
                {
                    if (checkboxList[count].Checked)
                    {
                        t.ShowLines(viewports, drawnLines, true);
                    }
                    ++count;
                }

                count = 0;
                foreach (var t in this.curves)
                {
                    if (checkboxList[count].Checked == false)
                    {
                        t.ShowLines(viewports, drawnLines, false);
                    }
                    ++count;
                }
            }
            else
            {
                var disableCurves = new List<CurveShape>();
                var enableCurves = new List<CurveShape>();

                for (int i = 0; i < this.curves.Count; ++i)
                {
                    List<CurveShape> curveDrawList = checkboxList[i].Checked ? enableCurves : disableCurves;
                    if (curveDrawList.Count == 0)
                    {
                        curveDrawList.Add(this.curves[i]);
                    }
                    else
                    {
                        bool sameCheck = false;
                        foreach (var curve in curveDrawList)
                        {
                            sameCheck |= this.curves[i].Equals(curve);
                        }

                        if (!sameCheck)
                        {
                            curveDrawList.Add(this.curves[i]);
                        }
                    }
                }

                disableCurves.Reverse();

                foreach (var curve in disableCurves)
                {
                    curve.ShowLines(viewports, drawnLines, false);
                }

                enableCurves.Reverse();

                foreach (var curve in enableCurves)
                {
                    curve.ShowLines(viewports, drawnLines, true);
                }
            }

            // ノードは強制的に上書きするので奥から描く
            for (int i = this.curves.Count - 1; i >= 0; --i)
            {
                if (checkboxList[i].Checked)
                {
                    this.curves[i].ShowNode(viewports);
                }
            }
        }
    }
}
