﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Layout;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// カーブエディタの内部処理です。
    /// </summary>
    public partial class CurveEditor
    {
        #region 内部状態フィールド

        /// <summary>
        /// マウスカーソル表示状態
        /// </summary>
        private bool cursorVisible = true;

        /// <summary>
        /// シフトキーの軸ロック状態
        /// </summary>
        private bool shiftLock = false;

        /// <summary>
        /// 非アクティブ時のヘルプハンドラ設定フラグ
        /// </summary>
        private bool isInactiveHelpReady = false;

        #endregion

        #region 数値計算プロパティ

        /// <summary>
        /// 設置可能なキーの最大数.
        /// </summary>
        internal int MaxNodeNum
        {
            get { return this.EditorMode == EditorModes.ParticleTimeAnim ? 8 : 32; }
        }

        /// <summary>
        /// 最大の時間
        /// </summary>
        internal int MaxCurrentModeTime
        {
            get { return this.EditorMode == EditorModes.ParticleTimeAnim ? Constants.ParticleAnimMax : Constants.EmitterAnimMax; }
        }

        /// <summary>
        /// 最大フレーム数を得る
        /// パーティクル時間アニメでは常時100、エミッタ時間アニメは100か最大フレームのいずれか
        /// </summary>
        private float MaxFrame
        {
            get
            {
                if (this.EditorMode == EditorModes.ParticleTimeAnim || this.values.Count == 0)
                {
                    return Constants.ParticleAnimMax;
                }

                return Math.Max(Constants.ParticleAnimMax, this.values.Select(x => x.Frame).Max());
            }
        }

        /// <summary>
        /// 0～100のタイムスケールを取得します。
        /// </summary>
        private float InitialScaleX
        {
            get
            {
                return this.viewports.Main.Width /
                    (this.CalcPosFromTime(Constants.ParticleAnimMax) - this.CalcPosFromTime(0.0f));
            }
        }

        /// <summary>
        /// 時間軸が全てビューポート内に収まる際の横軸スケールを得る
        /// </summary>
        private float MinimumScaleX
        {
            get
            {
                return this.viewports.Main.Width /
                    (this.CalcPosFromTime(this.MaxCurrentModeTime) - this.CalcPosFromTime(0.0f));
            }
        }

        /// <summary>
        /// 値軸が全てビューポート内に収まる際の縦軸スケールを得る
        /// </summary>
        private float MinimumScaleY
        {
            get
            {
                var index = this.ZeroPinMode;
                float ratio = 1.0f;
                if (index == 0 || index == 2)
                {
                    ratio = 2.0f;
                }

                if (Math.Abs(this.MaxLimit) < 0.00001f && Math.Abs(this.MinLimit) < 0.00001f)
                {
                    this.MaxLimit = 1000000.0f;
                    this.MinLimit = -1000000.0f;
                }

                return ratio * 0.999999f * this.viewports.Main.Height /
                   Math.Abs(this.CalcPosFromValue(this.MaxLimit) - this.CalcPosFromValue(this.MinLimit));
            }
        }

        /// <summary>
        /// 操作されたスケールを有効範囲内にクランプした値を取得します。
        /// </summary>
        private PointF ValidScale
        {
            get
            {
                var validX = Math.Min(30000.0f, Math.Max(this.scale.X, this.MinimumScaleX));
                var validY = Math.Min(30000.0f, Math.Max(this.scale.Y, this.MinimumScaleY));

                if (this.EditorMode == EditorModes.ParticleTimeAnim)
                {
                    validX = this.MinimumScaleX;
                    this.scale.X = validX;
                    this.scale.Y = validY;
                    return new PointF(validX, validY);
                }

                float diffX = Math.Abs(validX - this.scale.X);
                float diffY = Math.Abs(validY - this.scale.Y);

                // どっちも補正域
                if (diffX > 0.001f && diffY > 0.001f)
                {
                    if (diffX < diffY)
                    {
                        // Xの方が超えたてなので、そちらを基準にした比率で補正
                        float adjust = this.scale.Y / this.scale.X;
                        this.scale.X = validX;
                        this.scale.Y = validX * adjust;
                    }
                    else
                    {
                        // Yの方が超えたてなので、そちらを基準にした比率で補正
                        float adjust = this.scale.X / this.scale.Y;
                        this.scale.Y = validY;
                        this.scale.X = validY * adjust;
                    }
                }

                return new PointF(validX, validY);
            }
        }

        #endregion

        #region 数値計算メソッド

        /// <summary>
        /// ノードのX座標から時間を計算します。
        /// </summary>
        /// <param name="argX">ノードのX座標</param>
        /// <param name="digit">有効小数点桁数</param>
        /// <returns>アニメーション時間(% or フレーム)</returns>
        internal float CalcTimeFromPos(float argX, int digit = 0)
        {
            return Constants.CalcTimeFromPos(argX, digit);
        }

        /// <summary>
        /// アニメーション時間からノードのX座標を計算します。
        /// </summary>
        /// <param name="argT">アニメーション時間(% or フレーム)</param>
        /// <returns>ノードのX座標</returns>
        internal float CalcPosFromTime(float argT)
        {
            return Constants.CalcPosFromTime(argT);
        }

        /// <summary>
        /// ノードのY座標からアニメーションのキー値を計算します。
        /// </summary>
        /// <param name="argY">ノードのY座標</param>
        /// <returns>アニメーションのキー値</returns>
        internal float CalcValueFromPos(float argY)
        {
            return Constants.CalcValueFromPos(argY, this.NormalizeAt);
        }

        /// <summary>
        /// アニメーションのキー値からノードのY座標を計算します。
        /// </summary>
        /// <param name="argV">アニメーションのキー値</param>
        /// <returns>ノードのY座標</returns>
        internal float CalcPosFromValue(float argV)
        {
            if (this.NormalizeAt <= 0.0f)
            {
                this.NormalizeAt = 1.0f;
            }

            return Constants.CalcPosFromValue(argV, this.NormalizeAt);
        }

        /// <summary>
        /// 設定されているステップ幅で時間値をスナップします。
        /// </summary>
        /// <param name="argTime">アニメーション時間</param>
        /// <returns>スナップされたアニメーション時間</returns>
        internal float SnapTime(float argTime)
        {
            var time = this.CalcTimeFromPos(argTime);

            if (!this.noSnapping)
            {
                switch (this.combobox_SnapFrame.SelectedIndex)
                {
                    case 0:
                        time = (int)Math.Round(time / 1.0f) * 1.0f;
                        break;
                    case 1:
                        time = (int)Math.Round(time / 2.0f) * 2.0f;
                        break;
                    case 2:
                        time = (int)Math.Round(time / 5.0f) * 5.0f;
                        break;
                    case 3:
                        time = (int)Math.Round(time / 10.0f) * 10.0f;
                        break;
                }
            }

            return this.CalcPosFromTime(time);
        }

        /// <summary>
        /// 設定されているステップでキー値をスナップします。
        /// </summary>
        /// <param name="argValue">キー値</param>
        /// <param name="index">スナップ時に検索から除外するインデックス(-1で全キーを対象)</param>
        /// <returns>スナップしたキー値</returns>
        internal float SnapValue(float argValue, int index = -1)
        {
            var value = this.CalcValueFromPos(argValue);

            if (!this.noSnapping)
            {
                if (this.otherKeySnapping)
                {
                    float nearest = this.manipurator.GetNearestValue(index, value, this.checkboxList);
                    float nearestPos = this.CalcPosFromValue(nearest);
                    float eval = Math.Abs(nearest - value) *
                        this.ValidScale.Y * this.NormalizeAt * Constants.ViewportFullHeight;
                    if (eval / this.NormalizeAt < Constants.SnapDistance * this.NormalizeAt)
                    {
                        return nearestPos;
                    }
                }

                // 最小値がマイナス域で、0固定モードが「中央」か「無し」の時だけ0スナップを効かせる
                if (this.MinLimit < 0.0f && (this.ZeroPinMode == 1 || this.ZeroPinMode == 3))
                {
                    float zeroY = this.CalcPosFromValue(0.0f);
                    float eval = Math.Abs(value) *
                        this.ValidScale.Y * this.NormalizeAt * Constants.ViewportFullHeight;
                    if (eval / this.NormalizeAt < Constants.SnapDistance * this.NormalizeAt)
                    {
                        return zeroY;
                    }
                }

                switch (this.combobox_SnapValue.SelectedIndex)
                {
                    case 1:
                        value = (int)Math.Round(value * 1000.0f) / 1000.0f;
                        break;
                    case 2:
                        value = (int)Math.Round(value * 100.0f) / 100.0f;
                        break;
                    case 3:
                        value = (int)Math.Round(value * 10.0f) / 10.0f;
                        break;
                    case 4:
                        value = (int)Math.Round(value * 1.0f) / 1.0f;
                        break;
                    case 5:
                        value = (int)Math.Round(value * 0.1f) / 0.1f;
                        break;
                }
            }

            return this.CalcPosFromValue(value);
        }

        #endregion

        #region ユーティリティメソッド

        /// <summary>
        /// エディタの状態をViewModel側に同期します。
        /// </summary>
        /// <param name="passNotify">履歴に積む場合はtrue,そうでなければfalse.</param>
        private void SyncData(bool passNotify = false)
        {
            this.manipurator.SetDataToTable(this.values, this.targetNode);

            this.updating = true;

            if (passNotify)
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.PropertyChanged,
                    propertyName: "Values");
            }
            else
            {
                this.LogicalTreeElementExtender.NotifyPropertyChanged(
                    BindingUpdateType.Validation,
                    propertyName: "Values");
                this.OnValueChanged(new ValueChangedExEventArgs(
                    this.oldValue,
                    this.values,
                    false,
                    this.ValuesTargetPropertyName));
                this.oldValue.Set(this.values);
            }

            this.updating = false;
        }

        /// <summary>
        /// エディタの状態をViewModel側のデータにあわせて更新します。
        /// </summary>
        private void SyncView()
        {
            if (this.updating)
            {
                return;
            }

            this.manipurator.Reset(this.viewports);

            this.noSnapping = true;
            this.manipurator.SetCurvesFromTable(
                this.values,
                this.viewports,
                this.checkboxList,
                this.CalcPosFromTime,
                this.CalcPosFromValue);

            this.noSnapping = false;

            this.selectedNodes.Clear();

            this.targetNode = this.manipurator.GetNodeAsTarget(this.values.SelectedKeyFrameIndex, this.checkboxList);

            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            this.UpdateTarget();
            this.UpdateResetButton();

            // カーブの選択状況の変更に合わせてカーブへのフォーカスを行う
            this.AdjustViewport(false);
            this.RequestDraw();
        }

        /// <summary>
        /// デザイナ生成コード外でのUI設定
        /// </summary>
        private void AdditionalInitialize()
        {
            this.combobox_ZeroPin.Items.Add(Properties.Resources.CurveEditorTop);
            this.combobox_ZeroPin.Items.Add(Properties.Resources.CurveEditorCenter);
            this.combobox_ZeroPin.Items.Add(Properties.Resources.CurveEditorBottom);
            this.combobox_ZeroPin.Items.Add(Properties.Resources.CurveEditorNone);
            this.combobox_ZeroPin.SelectedIndex = 3;
            this.combobox_ZeroPin.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
            this.prevZeroPin = 3;

            this.combobox_SnapFrame.Items.Add("1%");
            this.combobox_SnapFrame.Items.Add("2%");
            this.combobox_SnapFrame.Items.Add("5%");
            this.combobox_SnapFrame.Items.Add("10%");
            this.combobox_SnapFrame.SelectedIndex = 0;
            this.combobox_SnapFrame.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;

            this.combobox_SnapValue.Items.Add(Properties.Resources.CurveEditorNone);
            this.combobox_SnapValue.Items.Add("0.001");
            this.combobox_SnapValue.Items.Add("0.01");
            this.combobox_SnapValue.Items.Add("0.1");
            this.combobox_SnapValue.Items.Add("1");
            this.combobox_SnapValue.Items.Add("10");
            this.combobox_SnapValue.SelectedIndex = 2;
            this.combobox_SnapValue.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;

            this.label_InterMode.Text = Properties.Resources.InterMode;
            this.combobox_InterMode.Items.Add(Properties.Resources.InterModeLinear);
            this.combobox_InterMode.Items.Add(Properties.Resources.InterModeSmooth);
            this.combobox_InterMode.SelectedIndex = 0;
            this.combobox_InterMode.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;

            this.loopEnd = 100;
            this.text_LoopEnd.Text = this.loopEnd.ToString("D");

            this.button_Reset.Enabled = false;
            this.button_Reset.BackgroundImage = Properties.Resources.Icon_Clear_Lock;

            this.button_SnapKeyMode.BackgroundImage = Properties.Resources.Icon_Curve__KeySnap_On;

            // イベントハンドラの登録はデザイナコード内では行わない
            this.checkbox_X.CheckedChanged += this.Handle_checkbox_X_CheckedChanged;
            this.checkbox_Y.CheckedChanged += this.Handle_checkbox_Y_CheckedChanged;
            this.checkbox_Z.CheckedChanged += this.Handle_checkbox_Z_CheckedChanged;
            this.checkbox_W.CheckedChanged += this.Handle_checkbox_W_CheckedChanged;
            this.button_Reset.Click += this.Handle_button_Reset_Click;
            this.button_LoopMode.Click += this.Handle_button_LoopMode_Click;
            this.text_LoopEnd.TextChanged += this.Handle_text_LoopEnd_TextChanged;
            this.text_LoopEnd.Enter += this.Handle_text_LoopEnd_Enter;
            this.text_LoopEnd.KeyDown += this.Handle_text_LoopEnd_KeyDown;
            this.text_LoopEnd.KeyPress += this.Handle_text_LoopEnd_KeyPress;
            this.text_LoopEnd.Leave += this.Handle_text_LoopEnd_Leave;
            this.combobox_SnapFrame.SelectedIndexChanged += this.Handle_combobox_SnapFrame_SelectedIndexChanged;
            this.combobox_SnapValue.SelectedIndexChanged += this.Handle_combobox_SnapValue_SelectedIndexChanged;
            this.text_Frame.Enter += this.Handle_text_Frame_Enter;
            this.text_Frame.KeyDown += this.Handle_text_Frame_KeyDown;
            this.text_Frame.KeyPress += this.Handle_text_Frame_KeyPress;
            this.text_Frame.Leave += this.Handle_text_Frame_Leave;
            this.text_Value.Enter += this.Handle_text_Value_Enter;
            this.text_Value.KeyDown += this.Handle_text_Value_KeyDown;
            this.text_Value.KeyPress += this.Handle_text_Value_KeyPress;
            this.text_Value.Leave += this.Handle_text_Value_Leave;
            this.checkbox_Random.CheckedChanged += this.Handle_checkbox_Random_CheckedChanged;
            this.timer_DragScroll.Tick += this.Handle_timer_DragScroll_Tick;
            this.button_SnapKeyMode.Click += this.Handle_button_SnapKeyMode_Click;
            this.combobox_InterMode.SelectedIndexChanged += this.Handle_combobox_InterMode_SelectedIndexChanged;
            this.combobox_ZeroPin.SelectedIndexChanged += this.Handle_combobox_ZeroPin_SelectedIndexChanged;

            this.KeyDown += (this.Handle_CurveEditor_KeyDown);
            this.KeyUp += (this.Handle_CurveEditor_KeyUp);

            this.text_FocusDummy.KeyDown += this.Handle_CurveEditor_KeyDown;
            this.text_FocusDummy.KeyUp += this.Handle_CurveEditor_KeyUp;

            this.text_LoopEnd.KeyDown += this.Handle_CurveEditor_KeyDown;
            this.text_LoopEnd.KeyUp += this.Handle_CurveEditor_KeyUp;
            this.text_Frame.KeyDown += this.Handle_CurveEditor_KeyDown;
            this.text_Frame.KeyUp += this.Handle_CurveEditor_KeyUp;
            this.text_Value.KeyDown += this.Handle_CurveEditor_KeyDown;
            this.text_Value.KeyUp += this.Handle_CurveEditor_KeyUp;

            this.text_FocusDummy.LostFocus +=
                (sender, args) => this.EndDrag(this.PointToClient(Control.MousePosition));

            this.text_FocusSwitcher0.GotFocus += this.Handle_text_FocusSwitcher0_GotFocus;
            this.text_FocusSwitcher1.GotFocus += this.Handle_text_FocusSwitcher1_GotFocus;
            this.text_FocusSwitcher2.GotFocus += this.Handle_text_FocusSwitcher2_GotFocus;

            this.button_Reset.MouseDown += (s, e) =>
                this.button_Reset.BackgroundImage = Properties.Resources.Icon_Clear_On;
            this.button_Reset.MouseUp += (s, e) =>
                this.button_Reset.BackgroundImage = this.button_Reset.Enabled
                    ? Properties.Resources.Icon_Clear_Off
                    : Properties.Resources.Icon_Clear_Lock;

            this.panel_Help.Click += this.Handle_button_PanelHelp_Click;
        }

        /// <summary>
        /// プリセットを適用した後の処理を実行.
        /// </summary>
        private void AfterAdaptPreset()
        {
            this.selectedNodes.Clear();
            this.FocusWithoutScroll();
            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            this.UpdateTarget();
            this.UpdateResetButton();
            this.AdjustViewport(false);
        }

        /// <summary>
        /// マウスカーソルがビューポート内に入った時の処理
        /// </summary>
        private void OnMouseEnterViewport()
        {
            if (this.entering)
            {
                return;
            }

            this.HideCursor();
            this.visualCursor.Show(this.viewports);

            this.entering = true;
        }

        /// <summary>
        /// マウスカーソルがビューポート外に出た時の処理
        /// </summary>
        private void OnMouseLeaveViewport()
        {
            if (!this.entering)
            {
                return;
            }

            this.ShowCursor();
            if (!this.dragging)
            {
                this.visualCursor.Hide(this.viewports);
                this.framePopup.Hide(this.viewports);
            }

            this.entering = false;
        }

        /// <summary>
        /// マウスカーソルを表示します。
        /// </summary>
        private void ShowCursor()
        {
            if (!this.cursorVisible)
            {
                System.Windows.Forms.Cursor.Show();
                this.cursorVisible = true;
            }
        }

        /// <summary>
        /// マウスカーソルを非表示にします。
        /// </summary>
        private void HideCursor()
        {
            if (this.cursorVisible && this.viewports.IsInnerMouseSpace(this.PointToClient(Cursor.Position)))
            {
                System.Windows.Forms.Cursor.Hide();
                this.cursorVisible = false;
            }
        }

        /// <summary>
        /// ドラッグ終了処理
        /// </summary>
        /// <param name="location">ドラッグ終了時の座標</param>
        private void EndDrag(PointF location)
        {
            if (this.ContainsFocus && this.viewports.IsInnerMouseSpace(location))
            {
                this.HideCursor();
            }
            else
            {
                this.ShowCursor();
                this.visualCursor.Hide(this.viewports);
                this.framePopup.Hide(this.viewports);
                this.UpdateTarget();
                this.RequestDraw();
            }

            if (!this.dragging)
            {
                return;
            }

            if (this.shiftLock == false)
            {
                this.EndLock();
            }

            // データを更新して履歴を積む
            this.SyncData();
            this.dragging = false;
        }

        /// <summary>
        /// 軸ロック開始処理
        /// </summary>
        private void BeginLock()
        {
            if (this.shiftLock != false)
            {
                return;
            }

            this.shiftLock = true;
            this.lockDirection = 0;
            if (this.dragging)
            {
                this.MoveNode(this.CalcLockPos(this.PointToClient(Control.MousePosition)));
            }
        }

        /// <summary>
        /// 軸ロック終了処理
        /// </summary>
        private void EndLock()
        {
            this.shiftLock = false;
            this.lockDirection = 0;
            Cursor.Current = Cursors.Default;
            this.HideCursor();
        }

        /// <summary>
        /// 軸ロックを適用した座標値を計算します。
        /// </summary>
        /// <param name="loc">座標値</param>
        /// <returns>軸ロック適用後の座標値</returns>
        private PointF CalcLockPos(PointF loc)
        {
            var transDragPos = this.viewports.DragStartPosition;
            if (this.shiftLock)
            {
                var diffX = Math.Abs(transDragPos.X - loc.X);
                var diffY = Math.Abs(transDragPos.Y - loc.Y);

                if (diffX > Constants.LockBorder && diffX > diffY)
                {
                    this.lockDirection = 1;
                }
                else if (diffY > Constants.LockBorder && diffX < diffY)
                {
                    this.lockDirection = 2;
                }
            }

            var tmpLoc = loc;
            switch (this.lockDirection)
            {
                case 1:
                    System.Windows.Forms.Cursor.Current = System.Windows.Forms.Cursors.NoMoveHoriz;
                    this.ShowCursor();
                    tmpLoc.Y = transDragPos.Y;
                    break;
                case 2:
                    System.Windows.Forms.Cursor.Current = System.Windows.Forms.Cursors.NoMoveVert;
                    this.ShowCursor();
                    tmpLoc.X = transDragPos.X;
                    break;
                default:
                    System.Windows.Forms.Cursor.Current = System.Windows.Forms.Cursors.Default;
                    this.HideCursor();
                    break;
            }

            // 0固定時のドラッグロック
            var viewLoc = this.viewports.Main.TransformPoint(tmpLoc);
            if (this.ZeroPinMode == 0)
            {
                if (this.CalcValueFromPos(viewLoc.Y) > 0.0f)
                {
                    viewLoc.Y = this.CalcPosFromValue(0.0f);
                    tmpLoc = this.viewports.Main.InverseTransformPoint(viewLoc);
                }
            }
            else if (this.ZeroPinMode == 2)
            {
                if (this.CalcValueFromPos(viewLoc.Y) < 0.0f)
                {
                    viewLoc.Y = this.CalcPosFromValue(0.0f);
                    tmpLoc = this.viewports.Main.InverseTransformPoint(viewLoc);
                }
            }

            return tmpLoc;
        }

        /// <summary>
        /// 現在選択しているノードから、次にタブオーダーで選択可能なチャンネルを列挙します。
        /// </summary>
        /// <param name="index">現在の選択ノードインデックス</param>
        /// <returns>編集対象になっていて選択候補になりうるチャンネルリスト</returns>
        private List<int> EnumSelectableChannel(int index)
        {
            var result = new List<int>();
            if (index < 0 || index >= this.values.Count)
            {
                return result;
            }

            for (int i = 0; i < this.Channels.Count; ++i)
            {
                if (this.checkboxList[i].Checked && !result.Contains(i))
                {
                    bool flag = true;
                    foreach (int contains in result)
                    {
                        var posA = this.manipurator.GetNodeByIndices(i, index).Location;
                        var posB = this.manipurator.GetNodeByIndices(contains, index).Location;
                        if (posA.Equals(posB))
                        {
                            flag = false;
                            break;
                        }
                    }

                    if (flag)
                    {
                        result.Add(i);
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// エディタが非アクティブな状態でもヘルプボタンがクリックできるようにします。
        /// </summary>
        private void SetupHelpButtonForInactive()
        {
            if (this.isInactiveHelpReady)
            {
                return;
            }

            var ctl = this.panel_Help;
            Action<object, MouseEventArgs> clickHandler = (s, e) =>
            {
                if ((ctl.Enabled && this.Enabled) || !ctl.Visible)
                {
                    return;
                }

                if (e.Button != MouseButtons.Left)
                {
                    return;
                }

                if (ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position)))
                {
                    this.Handle_button_PanelHelp_Click(null, null);
                }
            };
            Action<object, MouseEventArgs> moveHandler = (s, e) =>
            {
                if ((ctl.Enabled && this.Enabled) || !ctl.Visible)
                {
                    return;
                }

                if (ctl.ClientRectangle.Contains(ctl.PointToClient(Cursor.Position)))
                {
                    ctl.BackColor = SystemColors.GradientActiveCaption;
                    ctl.FlatAppearance.BorderSize = 1;
                    ctl.FlatAppearance.BorderColor = SystemColors.Highlight;
                }
                else
                {
                    ctl.BackColor = SystemColors.Control;
                    ctl.FlatAppearance.BorderSize = 0;
                    ctl.FlatAppearance.BorderColor = Color.FromArgb(0, 0, 0, 0);
                }
            };
            if (this.EditorMode == EditorModes.EmitterTimeAnim)
            {
                this.Parent.MouseDown += new MouseEventHandler(clickHandler);
                this.Parent.MouseMove += new MouseEventHandler(moveHandler);
            }
            else
            {
                this.MouseDown += new MouseEventHandler(clickHandler);
                this.MouseMove += new MouseEventHandler(moveHandler);
            }

            ctl.MouseEnter += (s, e) =>
            {
                ctl.BackColor = SystemColors.GradientActiveCaption;
                ctl.FlatAppearance.BorderSize = 1;
                ctl.FlatAppearance.BorderColor = SystemColors.Highlight;
            };
            ctl.MouseLeave += (s, e) =>
            {
                ctl.BackColor = SystemColors.Control;
                ctl.FlatAppearance.BorderSize = 0;
                ctl.FlatAppearance.BorderColor = Color.FromArgb(0, 0, 0, 0);
            };
            ctl.Click += (s, e) =>
            {
                this.FocusWithoutScroll();
            };

            this.isInactiveHelpReady = true;
        }

        /// <summary>
        /// マウスオーバー時のツールチップ設定
        /// </summary>
        private void SetupTooltips()
        {
            if (this.extendedToolTip == null)
            {
                this.extendedToolTip = new ExtendedToolTip(this, this.EditorMode == EditorModes.EmitterTimeAnim);
            }
            else
            {
                return;
            }

            this.extendedToolTip.PutToolTip(
                this.panel_EditSelection,
                Properties.Resources.CurveEditorAxisSelection);
            this.extendedToolTip.PutToolTip(
                this.button_SnapKeyMode,
                Properties.Resources.SnapToOtherKey);
            this.extendedToolTip.PutToolTip(
                this.panel_Snap,
                Properties.Resources.CurveEditorKeySnap);
            this.extendedToolTip.PutToolTip(
                this.panel_KeyValue,
                Properties.Resources.CurveEditorKeyValue);
            this.extendedToolTip.PutToolTip(
                this.button_Reset,
                Properties.Resources.CurveEditorClear);
            this.extendedToolTip.PutToolTip(
                this.panel_ZeroPin,
                Properties.Resources.CurveEditorZeroPin);

            // ループモードはその都度の状態に応じて表示内容を変更
            this.extendedToolTip.PutToolTipWithLogic(
                this.button_LoopMode,
                () => this.LoopMode == 0
                    ? Properties.Resources.CurveEditorOneTime
                    : Properties.Resources.CurveEditorLoop);

            // ヘルプは長いので、表示位置と時間を変更
            this.extendedToolTip.PutToolTip(
                this.panel_Help,
                Properties.Resources.CurveEditorInstruction,
                20,
                25,
                5000);
        }

        #endregion

        #region キー操作(アンドゥ・リドゥ対象)

        /// <summary>
        /// ノードの追加.
        /// </summary>
        /// <param name="argLoc">追加先の座標値(マウス座標系)</param>
        private void AddNode(PointF argLoc)
        {
            if (this.viewports.IsInnerMouseSpace(argLoc) == false)
            {
                return;
            }

            // 編集対象軸が1本もなかったら追加しない
            int checkedAxis = 0;
            for (int i = 0; i < this.Channels.Count; ++i)
            {
                checkedAxis += this.checkboxList[i].Checked ? 1 : 0;
            }

            if (checkedAxis == 0)
            {
                return;
            }

            if (this.values.Count >= this.MaxNodeNum)
            {
                this.warningToolTip.Show(this, argLoc);
                return;
            }

            this.manipurator.AddKey(argLoc, this.viewports, this.checkboxList, (key, i) =>
            {
                if (key != null && this.checkboxList[i].Checked)
                {
                    this.targetNode = key;
                    this.UpdateTarget();
                }

                // 挿入ではなかった場合、追加と同時に選択状態にする
                if (this.UpdateSelection(argLoc, true))
                {
                    this.dragging = true;
                    this.dragStarted = false;
                }
            });

            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            this.UpdateTarget();

            // データを更新して履歴を積む
            this.SyncData();
        }

        /// <summary>
        /// ノードの削除
        /// </summary>
        private void RemoveNode()
        {
            this.targetNode = this.manipurator.RemoveKey(this.viewports, this.targetNode);
            this.selectedNodes.Clear();
            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            this.UpdateTarget();

            // データを更新して履歴を積む
            this.SyncData();
        }

        /// <summary>
        /// ノードの移動
        /// </summary>
        /// <param name="argLoc">移動先の座標値(マウス座標系)</param>
        private void MoveNode(PointF argLoc)
        {
            var nodePos = this.manipurator.MoveKey(this.targetNode, this.selectedNodes, argLoc, this.viewports);

            this.selectedMarker.Center = nodePos;
            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);
            this.UpdateTarget();
            this.UpdateSelectedFrame();

            // ノードの移動はマウスアップ時とテキストボックスからの入力時に同期する
        }

        /// <summary>
        /// カーブをリセットします。
        /// </summary>
        private void ResetNode()
        {
            if (this.IsUpdatingDataContext)
            {
                return;
            }

            // リセット修正
            this.ResetViewport();

            this.noSnapping = true;

            IExecutable executable = this.ResetExecutable;
            if (executable != null)
            {
                executable.Execute(null);
            }

            this.manipurator.Reset(this.viewports);
            this.noSnapping = false;

            this.selectedNodes.Clear();
            this.targetNode = null;
            this.UpdateTarget();
            this.manipurator.SortCurves(this.viewports, this.InterMode, this.checkboxList);

            this.checkbox_X.Checked = true;
            this.checkbox_Y.Checked = true;
            this.checkbox_Z.Checked = true;
            this.checkbox_W.Checked = true;
            this.combobox_SnapFrame.SelectedIndex = 0;
            this.combobox_SnapValue.SelectedIndex = 2;
            this.otherKeySnapping = true;
            this.button_SnapKeyMode.BackgroundImage = Properties.Resources.Icon_Curve__KeySnap_On;

            this.loopEnd = 100;
            this.text_LoopEnd.Text = this.loopEnd.ToString("D");
            this.text_LoopEnd.Enabled = false;
            this.randomize = 0;
            this.checkbox_Random.Checked = false;
            this.loopMode = 0;
            this.button_LoopMode.BackgroundImage =
                Properties.Resources.CurveEditor_Icon_AnimTable_Loop_Off;
            this.interMode = 0;
            this.combobox_InterMode.SelectedIndex = 0;
            this.ZeroPinMode = this.DefaultZeroPin;

            this.targetNode = null;
            this.UpdateTarget();

            this.RequestDraw();

            this.FocusWithoutScroll();
            this.button_Reset.Enabled = false;
        }

        #endregion
    }
}
