﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.UIControls.Specifics.CurveEditor
{
    /// <summary>
    /// カーブエディタで使用する定数群です。
    /// </summary>
    internal static class Constants
    {
        #region サイズ・座標

        /// <summary>
        /// ノードを表す矩形の1辺の長さ.
        /// </summary>
        public const float NodeLength = 8.0f;

        /// <summary>
        /// ビューポート座標系での原点X.
        /// </summary>
        public const float ViewportOriginX = 10.0f;

        /// <summary>
        /// ビューポート座標系での原点Y.
        /// </summary>
        public const float ViewportOriginY = 300.0f;

        /// <summary>
        /// エディタ上で時間100%を表す幅.
        /// </summary>
        public const float ViewportFullWidth = 540.0f;

        /// <summary>
        /// エディタ上で1.0を表す高さ.
        /// </summary>
        public const float ViewportFullHeight = 280.0f;

        /// <summary>
        /// 他のキーの値にスナップする際の吸着距離.
        /// </summary>
        public const float SnapDistance = 10.0f;

        /// <summary>
        /// Shiftキーでロックする際の縦横判定を行う距離.
        /// </summary>
        public const float LockBorder = 4.0f;

        /// <summary>
        /// ピック時の選択領域幅
        /// </summary>
        public const float PickRange = 10.0f;

        /// <summary>
        /// パーティクル時間アニメの時間最大値
        /// </summary>
        public const int ParticleAnimMax = 100;

        /// <summary>
        /// エミッタ時間アニメの時間最大値
        /// </summary>
        public const int EmitterAnimMax = 65535;

        /// <summary>
        /// ノードの大きさ
        /// </summary>
        public static readonly SizeF NodeSize = new SizeF(NodeLength, NodeLength);

        /// <summary>
        /// ノードの左上角から中心へのオフセット
        /// </summary>
        public static readonly SizeF NodeCenterOffset = new SizeF(NodeLength * 0.5f, NodeLength * 0.5f);

        #endregion

        #region 色

        /// <summary>
        /// ノードカラー
        /// </summary>
        public static readonly Color[] NodeColors = new Color[4]
        {
            Color.FromArgb(255, 0, 0),
            Color.FromArgb(0, 192, 0),
            Color.FromArgb(0, 0, 255),
            Color.FromArgb(128, 128, 128)
        };

        /// <summary>
        /// ボーダーカラー
        /// </summary>
        public static readonly Color[] BorderColors = new Color[4]
        {
            Color.FromArgb(255, 160, 122),
            Color.FromArgb(95, 158, 160),
            Color.FromArgb(135, 135, 235),
            Color.FromArgb(160, 160, 160)
        };

        /// <summary>
        /// 編集非対象時カラー
        /// </summary>
        public static readonly Color[] DeactiveColors = new Color[4]
        {
            Color.FromArgb(255, 185, 185),
            Color.FromArgb(165, 245, 165),
            Color.FromArgb(185, 185, 255),
            Color.FromArgb(185, 185, 185)
        };

        #endregion

        #region 計算ユーティリティ

        /// <summary>
        /// ノードのX座標から時間を計算します。
        /// </summary>
        /// <param name="argX">ノードのX座標</param>
        /// <param name="digit">有効小数点桁数</param>
        /// <returns>アニメーション時間(% or フレーム)</returns>
        public static float CalcTimeFromPos(float argX, int digit = 0)
        {
            return (float)Math.Round((decimal)(argX - ViewportOriginX) / (decimal)ViewportFullWidth * 100.0m, digit);
        }

        /// <summary>
        /// アニメーション時間からノードのX座標を計算します。
        /// </summary>
        /// <param name="argT">アニメーション時間(% or フレーム)</param>
        /// <returns>ノードのX座標</returns>
        public static float CalcPosFromTime(float argT)
        {
            return (float)Math.Round(((decimal)argT / 100.0m * (decimal)ViewportFullWidth) + (decimal)ViewportOriginX, 0);
        }

        /// <summary>
        /// ノードのY座標からアニメーションのキー値を計算します。
        /// </summary>
        /// <param name="argY">ノードのY座標</param>
        /// <param name="normalizeAt">1.0をいくつの値とするか</param>
        /// <returns>アニメーションのキー値</returns>
        public static float CalcValueFromPos(float argY, float normalizeAt)
        {
            return (float)Math.Round((decimal)(ViewportOriginY - argY) / (decimal)ViewportFullHeight * (decimal)normalizeAt, 5);
        }

        /// <summary>
        /// アニメーションのキー値からノードのY座標を計算します。
        /// </summary>
        /// <param name="argV">アニメーションのキー値</param>
        /// <param name="normalizeAt">1.0をいくつの値とするか</param>
        /// <returns>ノードのY座標</returns>
        public static float CalcPosFromValue(float argV, float normalizeAt)
        {
            if (normalizeAt <= 0.0f)
            {
                normalizeAt = 1.0f;
            }

            return (float)((decimal)ViewportOriginY - (decimal)argV * (decimal)ViewportFullHeight / (decimal)normalizeAt);
        }

        #endregion
    }
}
