﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.ComponentModel;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.UIControls.Specifics.ColorPicker
{
    /// <summary>
    /// Control class for the color picker panel.
    /// </summary>
    [DefaultEvent("ValueChanged")]
    public partial class HSVColorPickerPanel : UIUserControl
    {
        /// <summary>
        /// color.
        /// </summary>
        private ColorHsva hsvaColor = new ColorHsva();

        /// <summary>
        /// GammaEnabled プロパティのバッキングフィールドです。
        /// </summary>
        private bool gammaEnabled = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public HSVColorPickerPanel()
        {
            this.InitializeComponent();

            this.TabStop = false;

            this.hsvaColor.H = this.hueSlider.ValueH;
            this.hsvaColor.S = this.saturationBox.ValueS;
            this.hsvaColor.V = this.saturationBox.ValueV;
            this.hsvaColor.A = 1.0f;
        }

        /// <summary>
        /// Triggered when the color value is changed.
        /// </summary>
        public event SequentialValueChangedEventHandler ValueChanged = null;

        /// <summary>
        /// HSVA カラーを取得または設定します。
        /// </summary>
        public ColorHsva ColorHSVA
        {
            get
            {
                return this.hsvaColor.Clone() as ColorHsva;
            }

            set
            {
                this.hsvaColor.Set(value);

                if (this.hueSlider != null)
                {
                    this.hueSlider.ValueH = this.hsvaColor.H;
                }

                if (this.saturationBox != null)
                {
                    this.saturationBox.ValueS = this.hsvaColor.S;
                    this.saturationBox.ValueV = this.hsvaColor.V;
                }
            }
        }

        /// <summary>
        /// RGBA カラーを取得または設定します。
        /// </summary>
        public ColorRgba ColorRGBA
        {
            get
            {
                return ColorUtility.ToColorRgba(this.hsvaColor);
            }

            set
            {
                var oldHsvaColor = new ColorHsva(this.hsvaColor);
                this.hsvaColor = ColorUtility.ToColorHsva(value);

                if (this.hsvaColor.S == 0.0f)
                {
                    this.hsvaColor.H = oldHsvaColor.H;
                }

                if (this.hsvaColor.V == 0.0f)
                {
                    this.hsvaColor.H = oldHsvaColor.H;
                    this.hsvaColor.S = oldHsvaColor.S;
                }

                if ((this.hsvaColor.H == 0.0f) || (this.hsvaColor.H == 1.0f))
                {
                    this.hsvaColor.H = (oldHsvaColor.H > 0.5f) ? 1.0f : 0.0f;
                }

                if (this.hueSlider != null)
                {
                    this.hueSlider.ValueH = this.hsvaColor.H;
                }

                if (this.saturationBox != null)
                {
                    this.saturationBox.SetHue(this.hsvaColor.H);

                    this.saturationBox.ValueS = this.hsvaColor.S;
                    this.saturationBox.ValueV = this.hsvaColor.V;
                }
            }
        }

        /// <summary>
        /// ガンマ補正を有効にするかどうか取得または設定します。
        /// </summary>
        public bool GammaEnabled
        {
            get
            {
                return this.gammaEnabled;
            }

            set
            {
                if (value != this.gammaEnabled)
                {
                    this.gammaEnabled = value;

                    if (this.saturationBox != null)
                    {
                        this.saturationBox.GammaEnabled = this.gammaEnabled;
                    }

                    if (this.hueSlider != null)
                    {
                        this.hueSlider.GammaEnabled = this.gammaEnabled;
                    }

                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// Handle ValueChanged event from the hue slider.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnHueChanged(object sender, SequentialValueChangedEventArgs e)
        {
            if (this.saturationBox != null)
            {
                this.saturationBox.SetHue(this.hueSlider.ValueH);
            }

            var oldH = this.hsvaColor.H;

            this.hsvaColor.H = this.hueSlider.ValueH;

            if ((e.Changing == false) || (oldH != this.hsvaColor.H))
            {
                if (this.ValueChanged != null)
                {
                    this.ValueChanged(this, e);
                }
            }
        }

        /// <summary>
        /// Handle ValueChanged event from the saturation and brightness box.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnSaturationChanged(object sender, SequentialValueChangedEventArgs e)
        {
            var oldS = this.hsvaColor.S;
            var oldV = this.hsvaColor.V;

            this.hsvaColor.S = this.saturationBox.ValueS;
            this.hsvaColor.V = this.saturationBox.ValueV;

            if ((e.Changing == false) || ((oldS != this.hsvaColor.S) || (oldV != this.hsvaColor.V)))
            {
                if (this.ValueChanged != null)
                {
                    this.ValueChanged(this, e);
                }
            }
        }
    }
}
