﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.Layout;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;
using Orientation = EffectMaker.UIControls.Layout.Orientation;
using SysDebug = System.Diagnostics.Debug;

namespace EffectMaker.UIControls.Specifics
{
    /// <summary>
    /// チェックリストです.
    /// </summary>
    public partial class CheckList : UIUserControl
    {
        /// <summary>
        /// スターマークの横幅.
        /// </summary>
        private const int ModifyWidth = 8;

        /// <summary>
        /// スターマークの縦幅.
        /// </summary>
        private const int ModifyHeight = 12;

        /// <summary>
        /// チェックアイテム.
        /// </summary>
        private CheckItem[] items = new CheckItem[0];

        /// <summary>
        /// チェックアイテムのStackPanel
        /// </summary>
        private StackPanel[] stackPanels = new StackPanel[0];

        /// <summary>
        /// アイテムのラベル.
        /// </summary>
        IEnumerable<string> labels;

        /// <summary>
        /// 列数.
        /// </summary>
        private int column = 1;

        /// <summary>
        /// 行間の幅
        /// </summary>
        private int rowPadding = 8;

        /// <summary>
        /// フラグデータ.
        /// </summary>
        private uint flags;

        /// <summary>
        /// 変更フラグの監視ターゲット.
        /// </summary>
        private string modificationFlagTarget;

        /// <summary>
        /// データ更新処理の二重呼び出しを防ぐためのフラグ.
        /// </summary>
        private bool dataUpdating;

        /// <summary>
        /// スターマークアイコンのデータコンテキスト.
        /// </summary>
        private object modifyIconDataContext;

        /// <summary>
        /// SideBySidePanel
        /// </summary>
        private SideBySidePanesPanel sidebySidePanesPanel;

        /// <summary>
        /// 左側のStackPanel
        /// </summary>
        private StackPanel leftStackPanel;

        /// <summary>
        /// 右側のStackPanel
        /// </summary>
        private StackPanel rightStackPanel;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public CheckList()
        {
            this.InitializeComponent();

            this.PropertyChanged += this.OnPropertyChanged;

            this.sidebySidePanesPanel = new SideBySidePanesPanel()
            {
                Dock = DockStyle.Fill,
                LeftPaneWidth = 329,
                RightPaneWidth = 327,
                TopAndBottomDistance = -6
            };

            this.leftStackPanel = new StackPanel()
            {
                Margin = new Padding(7, 6, 7, 6),
                Orientation = Orientation.Vertical
            };

            this.rightStackPanel = new StackPanel()
            {
                Margin = new Padding(7, 6, 7, 6),
                Orientation = Orientation.Vertical
            };

            // レイアウト処理を停止
            this.leftStackPanel.SuspendLayout();
            this.rightStackPanel.SuspendLayout();
            this.sidebySidePanesPanel.SuspendLayout();
            this.SuspendLayout();

            this.sidebySidePanesPanel.Controls.Add(this.leftStackPanel);
            this.sidebySidePanesPanel.Controls.Add(this.rightStackPanel);
            this.Controls.Add(this.sidebySidePanesPanel);

            // レイアウト処理を再開
            this.leftStackPanel.ResumeLayout();
            this.rightStackPanel.ResumeLayout();
            this.sidebySidePanesPanel.ResumeLayout();
            this.ResumeLayout();
        }

        /// <summary>
        /// アイテム数を取得または設定します.
        /// </summary>
        public int NumItems
        {
            get
            {
                return this.items.Length;
            }

            set
            {
                SysDebug.Assert(value >= 0 && value <= 32, "項目数が不正。");

                if (this.items.Length != value)
                {
                    // チェックアイテムを作り直す
                    this.ResetItems(value);
                }
            }
        }

        /// <summary>
        /// フラグデータを取得または設定します.
        /// </summary>
        public uint Flags
        {
            get
            {
                return this.flags;
            }

            set
            {
                if (this.flags != value)
                {
                    this.flags = value;
                    this.UpdateItemChecks();

                    this.LogicalTreeElementExtender.NotifyPropertyChanged();
                }
            }
        }

        /// <summary>
        /// 変更フラグの監視ターゲット名を取得または設定します.
        /// </summary>
        public string ModificationFlagTarget
        {
            get
            {
                return this.modificationFlagTarget;
            }

            set
            {
                if (value != this.modificationFlagTarget)
                {
                    this.modificationFlagTarget = value;

                    this.UpdateModificationFlagTarget();
                }
            }
        }

        /// <summary>
        /// ラベルを取得または設定します.
        /// </summary>
        public IEnumerable<string> Labels
        {
            get
            {
                return this.labels;
            }

            set
            {
                // コレクションの内容だけ変わることもあるので差異チェックはしない
                // if (value == this.labels) ...
                this.labels = value;

                // アイテムのラベルを更新
                this.UpdateItemLabels();
            }
        }

        /// <summary>
        /// 列数を取得または設定します.
        /// </summary>
        public int Column
        {
            get
            {
                return this.column;
            }

            set
            {
                SysDebug.Assert(value >= 1, "列数が不正。");
                int newColumn = value < 1 ? 1 : value;

                if (newColumn != this.column)
                {
                    this.column = newColumn;

                    this.Invalidate();
                }
            }
        }

        /// <summary>
        /// 行間の幅を取得または設定します。
        /// </summary>
        public int RowPadding
        {
            get
            {
                return this.rowPadding;
            }

            set
            {
                if (value != this.rowPadding)
                {
                    this.rowPadding = value;

                    this.PerformLayout();
                }
            }
        }

        /// <summary>
        /// チェック項目のスターマークのデータコンテキストを取得または設定します。
        /// </summary>
        public object ModifyIconDataContext
        {
            get
            {
                return this.modifyIconDataContext;
            }

            set
            {
                if (value != this.modifyIconDataContext)
                {
                    this.modifyIconDataContext = value;

                    foreach (CheckItem item in this.items)
                    {
                        item.ModifyIcon.DataContext = this.ModifyIconDataContext;
                    }
                }
            }
        }

        /// <summary>
        /// 一列表示と分割表示の設定
        /// </summary>
        public bool IsSingleColumn
        {
            get
            {
                return this.sidebySidePanesPanel.IsSingleColumn;
            }

            set
            {
                this.sidebySidePanesPanel.IsSingleColumn = value;
            }
        }

        /// <summary>
        /// UIの推薦サイズ(最小サイズ)を取得します.
        /// </summary>
        /// <param name="proposedSize">有効な領域</param>
        /// <returns>UIの推薦サイズを返します.</returns>
        public override Size GetPreferredSize(Size proposedSize)
        {
            if (this.IsSelfOrParentCollapsed() == true)
            {
                return Size.Empty;
            }

            Size contentpanelSize = this.sidebySidePanesPanel.GetPreferredSize(proposedSize);

            return new Size(
                this.Padding.Horizontal + contentpanelSize.Width,
                this.Padding.Vertical + contentpanelSize.Height);
        }

        /// <summary>
        /// UIのレイアウト処理を行います.
        /// </summary>
        /// <param name="e">イベント情報</param>
        protected override void OnLayout(LayoutEventArgs e)
        {
            base.OnLayout(e);

            this.sidebySidePanesPanel.Left = this.Padding.Left + this.sidebySidePanesPanel.Margin.Left;
            this.sidebySidePanesPanel.Top = this.Padding.Top + this.sidebySidePanesPanel.Margin.Top;

            this.sidebySidePanesPanel.Width = this.ClientSize.Width
                - (this.Padding.Horizontal + this.sidebySidePanesPanel.Margin.Horizontal);

            this.sidebySidePanesPanel.Height = this.ClientSize.Height
                - (this.Padding.Vertical + this.sidebySidePanesPanel.Margin.Vertical);
        }

        /// <summary>
        /// チェック項目を作り直します.
        /// </summary>
        /// <param name="numItems">項目数</param>
        private void ResetItems(int numItems)
        {
            // レイアウト処理を停止
            this.leftStackPanel.SuspendLayout();
            this.rightStackPanel.SuspendLayout();
            this.sidebySidePanesPanel.SuspendLayout();
            this.SuspendLayout();

            // チェック項目を削除
            foreach (CheckItem item in this.items)
            {
                //// TODO:必要であればバインドを削除

                this.Controls.Remove(item.ModifyIcon);
                this.Controls.Remove(item.CheckBox);

                this.leftStackPanel.Controls.Remove(item.ModifyIcon);
                this.leftStackPanel.Controls.Remove(item.CheckBox);
                this.rightStackPanel.Controls.Remove(item.ModifyIcon);
                this.rightStackPanel.Controls.Remove(item.CheckBox);
            }

            // チェック項目を作成
            this.items = new CheckItem[numItems];

            // チェック項目ごとのStackPanelを作成
            this.stackPanels = new StackPanel[numItems];

            for (int i = 0; i < this.stackPanels.Length; ++i)
            {
                this.stackPanels[i] = new StackPanel();
            }

            for (int i = 0; i < this.items.Length; ++i)
            {
                this.items[i] = new CheckItem();
                this.items[i].CheckBox.CheckedChanged += this.OnCheckedChanged;

                this.stackPanels[i] = new StackPanel()
                {
                    AutoSize = true,
                    Margin = new Padding(7, 6, 7, 6),
                    Orientation = Orientation.Horizontal,
                    FocusScrollEnable = true,
                    Size = new Size(206, 16),
                    TabIndex = 0
                };
                this.stackPanels[i].AddChildControls(this.items[i].ModifyIcon);
                this.stackPanels[i].Controls.Add(this.items[i].CheckBox);

                if (i < this.items.Length / 2)
                {
                    this.leftStackPanel.Controls.Add(this.stackPanels[i]);
                }
                else
                {
                    this.rightStackPanel.Controls.Add(this.stackPanels[i]);
                }
            }

            // スターマークのバインドを更新
            this.UpdateModificationDataContext();

            // アイテムのラベルを更新
            this.UpdateItemLabels();

            // アイテムのチェック状態を更新
            this.UpdateItemChecks();

            // レイアウト処理を再開
            this.leftStackPanel.ResumeLayout();
            this.rightStackPanel.ResumeLayout();
            this.sidebySidePanesPanel.ResumeLayout();
            this.ResumeLayout();
        }

        /// <summary>
        /// 変更フラグのDataContextを更新.
        /// </summary>
        private void UpdateModificationDataContext()
        {
            // ModifyIconDataContextのバインドを削除
            {
                DataBinding.Binder[] bindersToRemove = this.Bindings.Where(
                    item => item.ElementPropertyName == "ModifyIconDataContext").ToArray();

                if (bindersToRemove != null && bindersToRemove.Length > 1)
                {
                    Logger.Log(
                        LogLevels.Error,
                        "CheckList.UpdateModificationDataContext(): Data context binds to multiple data sources.");
                }

                foreach (DataBinding.Binder binder in bindersToRemove)
                {
                    // If the bound data source is the same, don't replace the binding with the new one.
                    if (binder.DataSourcePropertyName == "ModificationFlagViewModel")
                    {
                        return;
                    }

                    this.Bindings.Remove(binder);
                }
            }

            this.UpdateModificationFlagTarget();

            // ModifyIconDataContextのバインドを追加
            var newBinder = this.AddBinding("ModifyIconDataContext", "ModificationFlagViewModel");
            newBinder.UpdateElement();
        }

        /// <summary>
        /// 変更フラグの監視ターゲットを更新.
        /// </summary>
        private void UpdateModificationFlagTarget()
        {
            for (int i = 0; i < this.items.Length; ++i)
            {
                UICaptionLabel icon = this.items[i].ModifyIcon;

                // "IsStarMarkVisible" へのバインドを削除
                DataBinding.Binder[] bindersToRemove = icon.Bindings.Where(
                    item => item.ElementPropertyName == "IsStarMarkVisible").ToArray();
                foreach (DataBinding.Binder binder in bindersToRemove)
                {
                    icon.Bindings.Remove(binder);
                }

                // "IsDefaultValue" へのバインドを削除
                bindersToRemove = icon.Bindings.Where(
                    item => item.ElementPropertyName == "IsDefaultValue").ToArray();
                foreach (DataBinding.Binder binder in bindersToRemove)
                {
                    icon.Bindings.Remove(binder);
                }

                uint mask = (uint)1 << i;

                {
                    // "IsStarMarkVisible" へのバインドを作り直す
                    DataBinding.Binder newBinder = icon.AddBinding(
                        "IsStarMarkVisible",
                        this.ModificationFlagTarget,
                        new MaskedValueChangedFlagConverter(),
                        mask);

                    newBinder.Mode = DataBinding.BindingMode.OneWay;
                }

                {
                    DataBinding.Binder newBinder = icon.AddBinding(
                        "IsDefaultValue",
                        this.ModificationFlagTarget,
                        new MaskedDefaultValueFlagConverter(),
                        mask);

                    newBinder.Mode = DataBinding.BindingMode.OneWay;
                }
            }
        }

        /// <summary>
        /// アイテムのラベルを更新します.
        /// </summary>
        private void UpdateItemLabels()
        {
            int index = 0;

            // ラベルを設定
            if (this.labels != null)
            {
                foreach (string label in this.labels)
                {
                    if (index >= this.items.Length)
                    {
                        break;
                    }

                    this.items[index].Label = label;
                    ++index;
                }
            }

            // ラベルが未登録のときはデフォルトの文字列を設定
            for (; index < this.items.Length; ++index)
            {
                this.items[index].Label = string.Format("[{0}]", index.ToString());
            }
        }

        /// <summary>
        /// アイテムのチェック状態をフラグデータに反映します.
        /// </summary>
        private void UpdateFlags()
        {
            if (this.dataUpdating)
            {
                return;
            }

            this.dataUpdating = true;

            // フラグデータを更新
            uint flags = 0;

            for (int i = 0; i < this.items.Length; ++i)
            {
                if (this.items[i].CheckBox.Checked)
                {
                    flags |= (uint)1 << i;
                }
            }

            this.Flags = flags;

            this.dataUpdating = false;
        }

        /// <summary>
        /// フラグデータの状態をアイテムのチェック状態に反映します.
        /// </summary>
        private void UpdateItemChecks()
        {
            if (this.dataUpdating)
            {
                return;
            }

            this.dataUpdating = true;

            // チェック項目を更新
            for (int i = 0; i < this.items.Length; ++i)
            {
                uint mask = (uint)1 << i;

                if ((mask & this.flags) == 0)
                {
                    this.items[i].CheckBox.Checked = false;
                }
                else
                {
                    this.items[i].CheckBox.Checked = true;
                }
            }

            this.dataUpdating = false;
        }

        /// <summary>
        /// PropertyChangedイベントのイベントハンドラです.
        /// </summary>
        /// <param name="sender">イベント発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            // DataContextに変更があったとき、変更フラグのDataContextを更新する
            if (e.PropertyName == "DataContext")
            {
                this.UpdateModificationDataContext();
            }
        }

        /// <summary>
        /// チェックをOn/Offしたときの処理を行います.
        /// </summary>
        /// <param name="sender">イベント発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnCheckedChanged(object sender, EventArgs e)
        {
            this.UpdateFlags();
        }

        /// <summary>
        /// チェック項目
        /// </summary>
        private class CheckItem
        {
            /// <summary>
            /// コンストラクタです.
            /// </summary>
            public CheckItem()
            {
                this.ModifyIcon = new UICaptionLabel()
                {
                    AutoSize = false,
                    Size = new Size(140, 16),
                    Margin = new Padding(0),
                    MaxWidth = 140,
                    TabIndex = 1
                };

                this.CheckBox = new UICheckBox()
                {
                    AutoSize = false,
                    Size = new Size(21, 16),
                    Margin = new Padding(16, 0, 0, 0),
                    BackColor = Color.Transparent
                };
            }

            /// <summary>
            /// スターマークアイコン.
            /// </summary>
            public UICaptionLabel ModifyIcon { get; set; }

            /// <summary>
            /// チェックボックス.
            /// </summary>
            public UICheckBox CheckBox { get; set; }

            /// <summary>
            /// チェック状態を取得または設定します.
            /// </summary>
            public bool Checked
            {
                get
                {
                    return this.CheckBox.Checked;
                }

                set
                {
                    this.CheckBox.Checked = value;
                }
            }

            /// <summary>
            /// ラベルの文字列を取得または設定します.
            /// </summary>
            public string Label
            {
                get
                {
                    return this.ModifyIcon.Text;
                }

                set
                {
                    this.ModifyIcon.Text = value;
                }
            }
        }
    }
}
