﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Behaviors;
using EffectMaker.UILogic.Commands;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UIControls.Specifics.Behaviors
{
    /// <summary>
    /// テクスチャページのランダムプリセットボタン押下時の振る舞い
    /// </summary>
    public class TextureRandomPresetBehavior : Behavior<UIButton>, INotifyPropertyChanged
    {
        /// <summary>
        /// プリセットデータ
        /// </summary>
        private static readonly RandomPresetDataType[] RandomPresetData = new[]
        {
            new RandomPresetDataType
            {
                PatternDivU = 1,
                PatternDivV = 2,
                RepeatCount = 2,
                PatternAnimType = 4
            },
            new RandomPresetDataType
            {
                PatternDivU = 2,
                PatternDivV = 1,
                RepeatCount = 1,
                PatternAnimType = 4
            },
            new RandomPresetDataType
            {
                PatternDivU = 2,
                PatternDivV = 2,
                RepeatCount = 3,
                PatternAnimType = 4
            },
            new RandomPresetDataType
            {
                PatternDivU = 1,
                PatternDivV = 1,
                RepeatCount = 0,
                PatternAnimType = 0
            }
        };

        /// <summary>
        /// Raised when the value of a property changes.
        /// </summary>
        public new event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// ランダムプリセットタイプ
        /// </summary>
        public enum RandomPresetTypes
        {
            /// <summary>
            /// 縦
            /// </summary>
            Vertical = 0,

            /// <summary>
            /// 横
            /// </summary>
            Horizontal = 1,

            /// <summary>
            /// 縦横
            /// </summary>
            VerticalHorizontal = 2,

            /// <summary>
            /// リセット
            /// </summary>
            Reset = 3
        }

        /// <summary>
        /// 水平パターン分割
        /// </summary>
        public int PatternDivU
        {
            get { return RandomPresetData[(int)this.RandomPresetType].PatternDivU; }
            set { }
        }

        /// <summary>
        /// 垂直パターン分割
        /// </summary>
        public int PatternDivV
        {
            get { return RandomPresetData[(int)this.RandomPresetType].PatternDivV; }
            set { }
        }

        /// <summary>
        /// 繰り返し数
        /// </summary>
        public int RepeatCount
        {
            get { return RandomPresetData[(int)this.RandomPresetType].RepeatCount; }
            set { }
        }

        /// <summary>
        /// パターンアニメーションタイプ
        /// </summary>
        public int PatternAnimType
        {
            get { return RandomPresetData[(int)this.RandomPresetType].PatternAnimType; }
            set { }
        }

        /// <summary>
        /// ランダムパターンテーブル数
        /// </summary>
        public int RandomPatternTableCount
        {
            get { return this.PatternDivU * this.PatternDivV; }
            set { }
        }

        /// <summary>
        /// ラップUを常にミラーにする
        /// </summary>
        public int WrapTypeU
        {
            get { return 0; }
            set { }
        }

        /// <summary>
        /// ラップVを常にミラーにする
        /// </summary>
        public int WrapTypeV
        {
            get { return 0; }
            set { }
        }

        /// <summary>
        /// ランダムパターンタイプ
        /// </summary>
        public RandomPresetTypes RandomPresetType { get; set; }

        /// <summary>
        /// UIButtonにアタッチするときの処理を行います。
        /// </summary>
        protected override void OnAttached()
        {
            this.AssociatedObject.Click += this.OnClicked;
        }

        /// <summary>
        /// UIButtonからデタッチするときの処理を行います。
        /// </summary>
        protected override void OnDetaching()
        {
            this.AssociatedObject.Click -= this.OnClicked;
        }

        /// <summary>
        /// ボタンが押されたときの処理を行います。
        /// </summary>
        /// <param name="sender">sender</param>
        /// <param name="e">event</param>
        private void OnClicked(object sender, EventArgs e)
        {
            using (new CommandCombiner())
            {
                this.RaisePropertyChanged("PatternDivU");
                this.RaisePropertyChanged("PatternDivV");
                this.RaisePropertyChanged("RepeatCount");
                this.RaisePropertyChanged("PatternAnimType");
                this.RaisePropertyChanged("RandomPatternTableCount");
                this.RaisePropertyChanged("WrapTypeU");
                this.RaisePropertyChanged("WrapTypeV");
            }
        }

        /// <summary>
        /// PropertyChanged イベントを発生させます。
        /// </summary>
        /// <param name="propertyName">変更されたプロパティの名前。</param>
        private void RaisePropertyChanged(string propertyName)
        {
            if (this.PropertyChanged != null)
            {
                this.PropertyChanged(this, new PropertyChangedEventArgs(propertyName));
            }
        }

        /// <summary>
        /// ランダムプリセット
        /// </summary>
        private class RandomPresetDataType
        {
            /// <summary>
            /// 水平パターン分割
            /// </summary>
            public int PatternDivU { get; set; }

            /// <summary>
            /// 垂直パターン分割
            /// </summary>
            public int PatternDivV { get; set; }

            /// <summary>
            /// 繰り返し数
            /// </summary>
            public int RepeatCount { get; set; }

            /// <summary>
            /// パターンアニメーションタイプ
            /// </summary>
            public int PatternAnimType { get; set; }
        }
    }
}
