﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Extensions;
using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;

namespace EffectMaker.Udd2Converter
{
    /// <summary>
    /// UDD2.0の定義データをパースします。
    /// </summary>
    public class Udd2Parser
    {
        /// <summary>
        /// カスタムシェーダのビルドインデータの名前リスト
        /// </summary>
        internal static readonly string[] CustomShaderBuiltInNames =
        {
            "Flags",
            "Switch0", "Switch1", "Switch2", "Switch3",
            "Switch4", "Switch5", "Switch6", "Switch7",
            "ShaderFlags",
            "ShaderSwitch0", "ShaderSwitch1", "ShaderSwitch2", "ShaderSwitch3",
            "ShaderSwitch4", "ShaderSwitch5", "ShaderSwitch6", "ShaderSwitch7",
        };

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="text">パースする文字列</param>
        public Udd2Parser(string text)
        {
            var xmlDoc = new XmlDocument();
            bool parseResult = true;
            try
            {
                xmlDoc.LoadXml(text);
            }
            catch
            {
                parseResult = false;
            }

            if (!parseResult || xmlDoc.DocumentElement == null)
            {
                try
                {
                    xmlDoc.LoadXml(Preprocess(text));
                }
                catch
                {
                    throw new InvalidDataException("Parse error as XML.");
                }
            }

            var root = xmlDoc.DocumentElement;
            if (root != null && root.Name == "UserDataDefinition")
            {
                this.UsedDataDefinition = new UserDataDefinition();
                this.ExpandUserDataDefinition(root, this.UsedDataDefinition);
            }
            else
            {
                throw new InvalidDataException("<UserDataDefinition> is not found.");
            }
        }

        /// <summary>
        /// 読み込んだUDD2.0定義を取得します。
        /// </summary>
        public UserDataDefinition UsedDataDefinition { get; private set; }

        /// <summary>
        /// 処理中に発生したエラーや警告メッセージを取得します。
        /// </summary>
        public string ReportedMessage { get; private set; }

        /// <summary>
        /// カスタムシェーダ用のビルドインメンバー名からXML保存用の要素名と実インスタンスを指す既定メンバ名を取得します。
        /// </summary>
        /// <param name="memberName">ビルドインメンバー名</param>
        /// <param name="elemName">要素名の出力</param>
        /// <param name="baseName">基底メンバ名の出力</param>
        /// <returns>正しく取得できていればtrue, 失敗したらfalse.</returns>
        internal static bool GetElementAndBaseNames(string memberName, out string elemName, out string baseName)
        {
            elemName = string.Empty;
            baseName = string.Empty;
            if (!CustomShaderBuiltInNames.Contains(memberName))
            {
                return false;
            }

            // Shaderが付いている名前と付いてない名前のオフセット
            int offsetAlias = CustomShaderBuiltInNames.Length / 2;

            if (memberName.Contains("Shader"))
            {
                int index = CustomShaderBuiltInNames.ToList().IndexOf(memberName);
                elemName = memberName;
                baseName = CustomShaderBuiltInNames[index - offsetAlias];
            }
            else
            {
                int index = CustomShaderBuiltInNames.ToList().IndexOf(memberName);
                elemName = CustomShaderBuiltInNames[index + offsetAlias];
                baseName = memberName;
            }

            return true;
        }

        /// <summary>
        /// ルートノードをパースして展開します。
        /// </summary>
        /// <param name="node">ルートノード</param>
        /// <param name="uddInfo">展開先インスタンス</param>
        private void ExpandUserDataDefinition(XmlNode node, UserDataDefinition uddInfo)
        {
            uddInfo.Kind = XmlParseUtil.ReadEnumAttribute(node, "Kind", uddInfo.Kind);
            uddInfo.Name = XmlParseUtil.ReadAttribute(node, "Name", uddInfo.Name);
            uddInfo.Version = XmlParseUtil.ReadAttribute(node, "Version", uddInfo.Version);

            var paramDefNode = node.ChildNodes.OfType<XmlNode>().Where(c => c.Name == "ParameterDefinition").ToArray();
            if (!paramDefNode.Any())
            {
                throw new InvalidDataException("<ParameterDefinition> is not found.");
            }

            foreach (var child in paramDefNode)
            {
                var paramInfo = new ParameterDefinition();
                this.ExpandParameterDefinition(child, paramInfo);
                var options = paramInfo.UISetting as OptionGroup;

                if (uddInfo.Kind == UserDataKind.CustomShader)
                {
                    string builtInMemberName = XmlParseUtil.ReadAttribute(child, "BuiltInMember", string.Empty);
                    if (CustomShaderBuiltInNames.Contains(paramInfo.MemberName) && string.IsNullOrEmpty(builtInMemberName))
                    {
                        throw new InvalidDataException("Can't use built-in member names.");
                    }

                    if (CustomShaderBuiltInNames.Contains(builtInMemberName))
                    {
                        // カスタムシェーダビルドインメンバーは、内部的にはShader～の付かない名前で処理する
                        string elemName, baseName;
                        GetElementAndBaseNames(builtInMemberName, out elemName, out baseName);
                        paramInfo.MemberName = baseName;
                        paramInfo.IsCustomShaderBuiltInMember = true;

                        if (options == null)
                        {
                            throw new InvalidDataException("BuiltInMember must have <OptionGroup>.");
                        }

                        // カスタムシェーダビルドインメンバーの上限チェックは特殊扱い
                        if (paramInfo.Type == PropertyType.CheckButtons)
                        {
                            if (options.Options.Count > 64)
                            {
                                throw new InvalidDataException("Count of Flags must be within 64.");
                            }

                            uddInfo.ParameterDefinitions.Add(paramInfo);
                            continue;
                        }

                        if (paramInfo.Type == PropertyType.RadioButtons)
                        {
                            if (options.Options.Count > 8)
                            {
                                throw new InvalidDataException("Count of Switch's options must be within 8.");
                            }

                            uddInfo.ParameterDefinitions.Add(paramInfo);
                            continue;
                        }

                        if (paramInfo.Type == PropertyType.ComboBox)
                        {
                            if (options.Options.Count > 8)
                            {
                                throw new InvalidDataException("Count of Switch's options must be within 8.");
                            }

                            uddInfo.ParameterDefinitions.Add(paramInfo);
                            continue;
                        }
                    }
                }

                // カスタムシェーダビルドインメンバー以外のチェックボックス・ラジオボタンは32択が上限
                if (options != null)
                {
                    if (paramInfo.Type == PropertyType.CheckButtons)
                    {
                        if (options.Options.Count > 32)
                        {
                            throw new InvalidDataException("Count of Flags must be within 32.");
                        }
                    }
                    else if (paramInfo.Type == PropertyType.RadioButtons)
                    {
                        if (options.Options.Count > 32)
                        {
                            throw new InvalidDataException("Count of Switch's options must be within 32.");
                        }
                    }
                    else if (paramInfo.Type == PropertyType.ComboBox)
                    {
                        if (options.Options.Count > 32)
                        {
                            throw new InvalidDataException("Count of Switch's options must be within 32.");
                        }
                    }
                }

                uddInfo.ParameterDefinitions.Add(paramInfo);
            }

            var names = uddInfo.ParameterDefinitions.Select(p => p.MemberName).ToArray();
            if (names.Count() != names.Distinct().Count())
            {
                throw new InvalidDataException("Duplicated member names.");
            }

            var uiNode = node.ChildNodes.OfType<XmlNode>().FirstOrDefault(c => c.Name == "UILayout");
            if (uiNode != null)
            {
                uddInfo.Layout = new UILayout();
                this.ExpandLayoutDefinition(uiNode, uddInfo);
            }
            else
            {
                throw new InvalidDataException("<UILayout> is not found.");
            }
        }

        /// <summary>
        /// パラメータ定義をパースして展開します。
        /// </summary>
        /// <param name="node">パラメータ定義ノード</param>
        /// <param name="paramInfo">展開先インスタンス</param>
        private void ExpandParameterDefinition(XmlNode node, ParameterDefinition paramInfo)
        {
            paramInfo.Name = XmlParseUtil.ReadAttribute(node, "Name", paramInfo.Name);
            paramInfo.Type = XmlParseUtil.ReadEnumAttribute(node, "Type", paramInfo.Type);
            paramInfo.AddedVersion = XmlParseUtil.ReadAttribute(node, "AddedVersion", paramInfo.AddedVersion);
            paramInfo.DeletedVersion = XmlParseUtil.ReadAttribute(node, "DeletedVersion", paramInfo.DeletedVersion);
            paramInfo.NeedReload = XmlParseUtil.ReadAttribute(node, "NeedReload", paramInfo.NeedReload);
            paramInfo.UsingRows = 0;
            paramInfo.Guid = Guid.NewGuid();

            foreach (var child in node.ChildNodes.OfType<XmlNode>())
            {
                if (child.Name == "ColorSetting")
                {
                    var colorSetting = new ColorSetting();
                    colorSetting.Default = XmlParseUtil.ReadAttribute(child, "Default", colorSetting.Default);
                    switch (paramInfo.Type)
                    {
                        case PropertyType.ColorRgba:
                            if (colorSetting.Default.Length != 4)
                            {
                                throw new InvalidDataException("Unmatched default value dimension: " + paramInfo.Name);
                            }

                            break;
                        case PropertyType.ColorRgb:
                        case PropertyType.ColorAnimeRgb:
                            if (colorSetting.Default.Length != 3)
                            {
                                throw new InvalidDataException("Unmatched default value dimension: " + paramInfo.Name);
                            }

                            break;
                        case PropertyType.ColorA:
                        case PropertyType.ColorAnimeA:
                            if (colorSetting.Default.Length != 1)
                            {
                                throw new InvalidDataException("Unmatched default value dimension: " + paramInfo.Name);
                            }

                            break;
                        default:
                            throw new InvalidDataException("This is not color property: " + paramInfo.Name);
                    }

                    paramInfo.UISetting = colorSetting;
                    paramInfo.UsingRows = 1;
                }
                else if (child.Name == "SliderSetting")
                {
                    // 値の次元数に応じて行数を設定
                    switch (paramInfo.Type)
                    {
                        case PropertyType.Slider1F:
                        case PropertyType.Slider1I:
                            paramInfo.UsingRows = 1;
                            break;
                        case PropertyType.Slider2F:
                        case PropertyType.Slider2I:
                            paramInfo.UsingRows = 3;
                            break;
                        case PropertyType.Slider3F:
                        case PropertyType.Slider3I:
                            paramInfo.UsingRows = 4;
                            break;
                        case PropertyType.Slider4F:
                        case PropertyType.Slider4I:
                            paramInfo.UsingRows = 5;
                            break;
                        default:
                            throw new InvalidDataException("This is not slider property: " + paramInfo.Name);
                    }

                    var sliderSetting = new SliderSetting();
                    sliderSetting.Default = XmlParseUtil.ReadAttribute(child, "Default", sliderSetting.Default);

                    if (sliderSetting.Default != null)
                    {
                        switch (paramInfo.Type)
                        {
                            case PropertyType.Slider1F:
                            case PropertyType.Slider1I:
                                if (sliderSetting.Default.Length != 1)
                                {
                                    throw new InvalidDataException("Unmatched default value dimension: " + paramInfo.Name);
                                }

                                break;
                            case PropertyType.Slider2F:
                            case PropertyType.Slider2I:
                                if (sliderSetting.Default.Length != 2)
                                {
                                    throw new InvalidDataException("Unmatched default value dimension: " + paramInfo.Name);
                                }

                                break;
                            case PropertyType.Slider3F:
                            case PropertyType.Slider3I:
                                if (sliderSetting.Default.Length != 3)
                                {
                                    throw new InvalidDataException("Unmatched default value dimension: " + paramInfo.Name);
                                }

                                break;
                            case PropertyType.Slider4F:
                            case PropertyType.Slider4I:
                                if (sliderSetting.Default.Length != 4)
                                {
                                    throw new InvalidDataException("Unmatched default value dimension: " + paramInfo.Name);
                                }

                                break;
                        }
                    }

                    sliderSetting.Minimum = XmlParseUtil.ReadAttribute(child, "Minimum", sliderSetting.Minimum);
                    sliderSetting.Maximum = XmlParseUtil.ReadAttribute(child, "Maximum", sliderSetting.Maximum);
                    sliderSetting.Step = XmlParseUtil.ReadAttribute(child, "Step", sliderSetting.Step);
                    sliderSetting.IsTextBoxOnly = XmlParseUtil.ReadAttribute(child, "IsTextBoxOnly", sliderSetting.IsTextBoxOnly);

                    if (paramInfo.UsingRows != 1)
                    {
                        sliderSetting.EnableTransform = XmlParseUtil.ReadAttribute(
                            child, "EnableTransform", sliderSetting.EnableTransform);
                        if (!sliderSetting.EnableTransform)
                        {
                            paramInfo.UsingRows--;
                        }
                    }

                    paramInfo.UISetting = sliderSetting;
                }
                else if (child.Name == "OptionGroup")
                {
                    switch (paramInfo.Type)
                    {
                        case PropertyType.CheckButtons:
                        case PropertyType.RadioButtons:
                        case PropertyType.ComboBox:
                        case PropertyType.StringComboBox:
                        case PropertyType.IntComboBox:
                            break;
                        default:
                            throw new InvalidDataException("This is not option type: " + paramInfo.Name);
                    }

                    var optionGroup = new OptionGroup();
                    this.ExpandOptionGroup(child, optionGroup, paramInfo.Type);
                    paramInfo.UISetting = optionGroup;
                    paramInfo.UsingRows = paramInfo.IsComboBox ? 1 : optionGroup.Options.Count;
                }
                else if (child.Name == "TextSetting")
                {
                    switch (paramInfo.Type)
                    {
                        case PropertyType.TextBox:
                            break;
                        default:
                            throw new InvalidDataException("This is not text type: " + paramInfo.Name);
                    }

                    var textSetting = new TextSetting();
                    textSetting.Default = XmlParseUtil.ReadAttribute(child, "Default", textSetting.Default);
                    textSetting.RowCount = XmlParseUtil.ReadAttribute(child, "RowCount", (int)1);
                    paramInfo.UISetting = textSetting;
                    paramInfo.UsingRows = textSetting.RowCount;
                }
                else if (child.Name == "FilePathSetting")
                {
                    switch (paramInfo.Type)
                    {
                        case PropertyType.FilePathTextBox:
                            break;
                        default:
                            throw new InvalidDataException("This is not file path type: " + paramInfo.Name);
                    }

                    var filePathSetting = new FilePathSetting();
                    filePathSetting.Default = XmlParseUtil.ReadAttribute(child, "Default", filePathSetting.Default);
                    filePathSetting.FileType = XmlParseUtil.ReadAttribute(child, "FileType", filePathSetting.FileType);

                    switch (filePathSetting.FileType)
                    {
                        case LastAccessDirectoryTypes.Texture:
                            filePathSetting.FileExtension = ".ftxb";
                            break;
                        case LastAccessDirectoryTypes.Model:
                            filePathSetting.FileExtension = ".fmdb";
                            break;
                        default:
                            throw new InvalidDataException("This is invalid value: " + filePathSetting.FileExtension);
                    }

                    paramInfo.UISetting = filePathSetting;
                    paramInfo.UsingRows = 1;
                }
                else if (child.Name == "UpdateCode")
                {
                    paramInfo.UpdateCode = child.InnerText;
                }
                else if (child.Name.Contains("text"))
                {
                    if (string.IsNullOrEmpty(child.Value))
                    {
                        continue;
                    }

                    string type = "", name = "";
                    int arraySize = 0;
                    var result = child.Value.Split(
                        new[] { '\r', '\n', ' ', '\t', ';', '[', ']' },
                        StringSplitOptions.RemoveEmptyEntries);
                    foreach (var str in result)
                    {
                        if (string.IsNullOrEmpty(type)) type = str;
                        else if (int.TryParse(str, out arraySize))
                        {
                            break;
                        }
                        else
                        {
                            name = str;
                        }
                    }

                    if (string.IsNullOrEmpty(name))
                    {
                        throw new InvalidDataException("Member name is not found: " + paramInfo.Name);
                    }

                    paramInfo.MemberName = name;

                    if (arraySize > 0)
                    {
                        if (paramInfo.Type != PropertyType.StringComboBox && paramInfo.Type != PropertyType.TextBox && paramInfo.Type != PropertyType.FilePathTextBox)
                        {
                            throw new InvalidDataException("Array type is allowed only string: " + paramInfo.Name);
                        }

                        if (arraySize % 4 != 0)
                        {
                            throw new InvalidDataException("String length must be multiple of 4.");
                        }

                        paramInfo.ArraySize = arraySize;
                    }
                    else if (paramInfo.IsAnimation)
                    {
                        if (!type.Contains("ResAnim8KeyParamSet"))
                        {
                            throw new InvalidDataException("The type of animation member must be ResAnim8KeyParamSet.");
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 選択肢グループを展開します。
        /// </summary>
        /// <param name="node">選択肢グループノード</param>
        /// <param name="group">展開先インスタンス</param>
        /// <param name="type">選択肢を利用するデータ種別</param>
        private void ExpandOptionGroup(XmlNode node, OptionGroup group, PropertyType type)
        {
            group.Default = XmlParseUtil.ReadAttribute(node, "Default", group.Default);
            group.Guid = Guid.NewGuid();

            int index = 0;
            foreach (var child in node.ChildNodes.OfType<XmlNode>().Where(n => n.Name == "Option"))
            {
                // BitIndex と Index のどちらでも読み取り対象とする
                int overrideIndex = XmlParseUtil.ReadAttribute(child, "BitIndex", -1);
                overrideIndex = XmlParseUtil.ReadAttribute(child, "Index", overrideIndex);
                index = overrideIndex != -1 ? overrideIndex : index;

                string value = XmlParseUtil.ReadAttribute(child, "Value", "");
                string uniqueKey = XmlParseUtil.ReadAttribute(child, "UniqueKey", "");
                if (string.IsNullOrEmpty(uniqueKey))
                {
                    ReportedMessage += string.Format(
                        "UniqueKey is not specified on {0} (index {1}){2}", value, index, Environment.NewLine);
                }

                group.Options.Add(new OptionGroup.OptionElement(value, index, uniqueKey));
                if (XmlParseUtil.ReadAttribute(child, "Selected", false))
                {
                    if (type == PropertyType.StringComboBox || type == PropertyType.IntComboBox)
                    {
                        // string, intコンボボックスのインデックスは単純なインデックス
                        group.Default = (ulong)index;
                    }
                    else if (type == PropertyType.CheckButtons)
                    {
                        // チェックボタンは複数選択可能な複合ビット
                        group.Default |= (ulong)Math.Pow(2, index);
                    }
                    else if (type == PropertyType.RadioButtons)
                    {
                        // ラジオボタンは択一のビットON値
                        group.Default = (ulong)Math.Pow(2, index);
                    }
                    else if (type == PropertyType.ComboBox)
                    {
                        // ビットフラグのコンボボックスは択一のビットON値
                        group.Default = (ulong)Math.Pow(2, index);
                    }
                }

                ++index;
            }

            var keys = group.Options.Select(o => o.UniqueKey).ToArray();
            if (keys.Count() != keys.Distinct().Count())
            {
                throw new InvalidDataException("Duplicated unique keys.");
            }

            var indices = group.Options.Select(o => o.Index).ToArray();
            if (indices.Count() != indices.Distinct().Count())
            {
                throw new InvalidDataException("Duplicated indices.");
            }
        }

        /// <summary>
        /// UIレイアウト情報をパースして展開します。
        /// </summary>
        /// <param name="node">レイアウトノード</param>
        /// <param name="uddInfo">ルートインスタンス</param>
        private void ExpandLayoutDefinition(XmlNode node, UserDataDefinition uddInfo)
        {
            int index = 0;
            foreach (var child in node.ChildNodes.OfType<XmlNode>().Where(n => n.Name == "Group"))
            {
                var group = new UIGroup();
                group.Name = XmlParseUtil.ReadAttribute(child, "Name", group.Name);
                foreach (var granChild in child.ChildNodes.OfType<XmlNode>().Where(p => p.Name == "Parameter"))
                {
                    var paramName = XmlParseUtil.ReadAttribute(granChild, "Member", "");
                    var paramDef = uddInfo.ParameterDefinitions.FirstOrDefault(p => p.MemberName == paramName);
                    if (paramDef == null && uddInfo.Kind == UserDataKind.CustomShader)
                    {
                        // カスタムシェーダの時はShader~がついていてもいなくてもOKとする
                        string baseName, elemName;
                        if (GetElementAndBaseNames(paramName, out elemName, out baseName))
                        {
                            paramDef = uddInfo.ParameterDefinitions.FirstOrDefault(p => p.MemberName == baseName);
                        }
                    }

                    if (paramDef != null)
                    {
                        var comment = XmlParseUtil.ReadAttribute(granChild, "Comment", "");
                        if (!string.IsNullOrEmpty(comment))
                        {
                            paramDef.Comment = comment;
                            paramDef.UsingRows++;
                        }

                        paramDef.GroupIndex = index;
                        group.Parameters.Add(paramDef);
                    }
                    else
                    {
                        throw new InvalidDataException("Undefined parameter is layouted: " + paramName);
                    }
                }

                ++index;
                uddInfo.Layout.Groups.Add(group);
            }
        }

        /// <summary>
        /// ヘッダファイルに記載されたアノテーションテキストをXMLドキュメントに変換します。
        /// </summary>
        /// <param name="originalText">アノテーションで記載されたUDD2.0ドキュメント</param>
        /// <returns></returns>
        private static string Preprocess(string originalText)
        {
            bool isParamDefOpen = false;
            string line;
            var sr = new StringReader(originalText);
            var sb = new StringBuilder();
            while ((line = sr.ReadLine()) != null)
            {
                if (line.Contains("///"))
                {
                    line = line.Substring(line.IndexOf("///") + 3);
                    if (sb.Length == 0)
                    {
                        while (!string.IsNullOrEmpty(line) && line[0] == ' ')
                        {
                            line = line.Substring(1);
                        }
                    }

                    if (line.Contains("<ParameterDefinition")) isParamDefOpen = true;
                    if (line.Contains("</ParameterDefinition>")) isParamDefOpen = false;
                    sb.Append(line);
                    sb.AppendLine();
                }
                else if (isParamDefOpen)
                {
                    sb.Append(line);
                    sb.AppendLine();
                }
            }

            return sb.ToString();
        }
    }
}
