﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using EffectMaker.Foundation.Core;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EffectMaker.Udd2Converter
{
    /// <summary>
    /// ユーザーデータの種別です。
    /// </summary>
    public enum UserDataKind
    {
        CustomAction,
        CustomShader,
        CustomField,
        EmitterExtParams,
    }

    /// <summary>
    /// データ型に対して使用するUIの種別です。
    /// </summary>
    public enum PropertyType
    {
        ColorRgb,
        ColorA,
        ColorRgba,
        ColorAnimeRgb,
        ColorAnimeA,
        Slider1F,
        Slider1I,
        Slider2F,
        Slider2I,
        Slider3F,
        Slider3I,
        Slider4F,
        Slider4I,
        CheckButtons,
        ComboBox,
        RadioButtons,
        StringComboBox,
        IntComboBox,
        TextBox,
        FilePathTextBox,
    }

    /// <summary>
    /// カラープロパティのUI設定です。
    /// </summary>
    public class ColorSetting
    {
        /// <summary>
        /// デフォルトカラーです。
        /// チャンネル数に応じて要素数が1,3,4のいずれかになります。
        /// </summary>
        public float[] Default { get; set; }
    }

    /// <summary>
    /// スライダープロパティのUI設定です。
    /// </summary>
    public class SliderSetting
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SliderSetting()
        {
            this.EnableTransform = true;
            this.IsTextBoxOnly = false;
        }

        /// <summary>
        /// デフォルト値です。
        /// チャンネル数に応じて要素数が1～4のいずれかになります。
        /// </summary>
        public float[] Default { get; set; }

        /// <summary>
        /// スライダーの下限値です。
        /// </summary>
        public float Minimum { get; set; }

        /// <summary>
        /// スライダーの上限値です。
        /// </summary>
        public float Maximum { get; set; }

        /// <summary>
        /// スライダーの刻み幅初期値です。
        /// 0.0001, 0.001, 0.01, 0.1, 1, 10, 100, 1000, 10000のいずれかを指定します。
        /// </summary>
        public float Step { get; set; }

        /// <summary>
        /// 個別・連動・比率の切り替えを使うか否かを表します。
        /// 使わない場合はスライダーの動作モードが個別で固定になります。
        /// </summary>
        public bool EnableTransform { get; set; }

        /// <summary>
        /// テキストボックスのみの表示にするか否かを表します。
        /// </summary>
        public bool IsTextBoxOnly { get; set; }
    }

    /// <summary>
    /// チェックボックス・ラジオボタン・コンボボックスの選択肢プロパティのUI設定です。
    /// </summary>
    public class OptionGroup
    {
        /// <summary>
        /// 選択肢1つを表す要素です。
        /// </summary>
        public class OptionElement
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="text">UIに表示する文字列</param>
            /// <param name="index">インデックス</param>
            /// <param name="key">シリアライズキー</param>
            public OptionElement(string text, int index, string key)
            {
                this.Text = text;
                this.Index = index;
                this.UniqueKey = string.IsNullOrEmpty(key) ? this.Text : key;
            }

            /// <summary>
            /// UIに表示する文字列を取得します。
            /// </summary>
            public string Text { get; private set; }

            /// <summary>
            /// コレクション中におけるインデックスを取得します。
            /// </summary>
            public int Index { get; private set; }

            /// <summary>
            /// XMLにシリアライズする際のキー文字列を取得します。
            /// </summary>
            public string UniqueKey { get; private set; }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OptionGroup()
        {
            this.Options = new List<OptionElement>();
            this.LeftSideNum = -1;
        }

        /// <summary>
        /// デフォルトの選択状態です。
        /// チェックボックスとラジオボタンは選択状態を表すビットの論理和で、
        /// コンボボックスは単純にアイテムリストのインデックスとして扱います。
        /// </summary>
        public ulong Default { get; set; }

        /// <summary>
        /// 選択肢の文字列リストです。
        /// </summary>
        public List<OptionElement> Options { get; private set; }

        /// <summary>
        /// 左側に配置するチェックボックスのアイテム数です。
        /// ラジオボタンとコンボボックスでは使用しません。
        /// パース処理中に設定される値です。
        /// </summary>
        public int LeftSideNum { get; set; }

        /// <summary>
        /// 選択肢をプロパティ化する際に使うGUIDです。
        /// パース処理中に設定される値です。
        /// </summary>
        public Guid Guid { get; set; }
    }

    /// <summary>
    /// テキストボックスのUI設定です。
    /// </summary>
    public class TextSetting
    {
        /// <summary>
        /// デフォルト値です。
        /// </summary>
        public string Default { get; set; }

        /// <summary>
        /// テキストボックスの行数です。
        /// </summary>
        public int RowCount { get; set; }
    }

    /// <summary>
    /// ファイルパステキストボックスのUI設定です。
    /// </summary>
    public class FilePathSetting
    {
        /// <summary>
        /// デフォルト値です。
        /// </summary>
        public string Default { get; set; }

        /// <summary>
        /// ファイルタイプです。
        /// </summary>
        public LastAccessDirectoryTypes FileType { get; set; }

        /// <summary>
        /// ファイルの拡張子です。
        /// </summary>
        public string FileExtension { get; set; }
    }

    /// <summary>
    /// パラメータ定義です。
    /// </summary>
    public class ParameterDefinition
    {
        /// <summary>
        /// 追加プロパティ用GUID辞書です。
        /// </summary>
        private Dictionary<string, Guid> additionalPropertyGuids = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ParameterDefinition()
        {
            this.AddedVersion = -1;
            this.DeletedVersion = -1;
        }

        /// <summary>
        /// パラメータのUI上での名前です。
        /// キャプションラベルに表示されます。
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// パラメータの型です。
        /// </summary>
        public PropertyType Type { get; set; }

        /// <summary>
        /// パラメータの構造体上での名前です。
        /// データモデルに使用します。
        /// </summary>
        public string MemberName { get; set; }

        /// <summary>
        /// パラメータのUI設定です。
        /// パラメータの型に応じて異なる型が設定されます。
        /// </summary>
        public object UISetting { get; set; }

        /// <summary>
        /// パラメータが文字列型だった場合の配列長です。
        /// </summary>
        public int ArraySize { get; set; }

        /// <summary>
        /// リロードが必要なパラメータか否かを表します。
        /// </summary>
        public bool NeedReload { get; set; }

        /// <summary>
        /// パラメータのUIの下部に表示するコメントです。
        /// </summary>
        public string Comment { get; set; }

        /// <summary>
        /// パラメータが追加されたUDDのバージョンです。
        /// </summary>
        public int AddedVersion { get; set; }

        /// <summary>
        /// パラメータが削除されたUDDのバージョンです。
        /// </summary>
        public int DeletedVersion { get; set; }

        /// <summary>
        /// パラメータのアップデートコードです。
        /// </summary>
        public string UpdateCode { get; set; }

        /// <summary>
        /// カスタムシェーダのビルドインパラメータであることを示します。
        /// パース処理中に設定される値です。
        /// </summary>
        public bool IsCustomShaderBuiltInMember { get; set; }

        /// <summary>
        /// UIレイアウトで決定される配置先グループのインデックスです。
        /// パース処理中に設定される値です。
        /// </summary>
        public int GroupIndex { get; set; }

        /// <summary>
        /// UIレイアウト時に専有する縦幅をおおよその行数で表します。
        /// パース処理中に設定される値です。
        /// </summary>
        public int UsingRows { get; set; }

        /// <summary>
        /// プロパティをコンバートする際に使用するGUIDです。
        /// パース処理中に設定される値です。
        /// </summary>
        public Guid Guid { get; set; }

        /// <summary>
        /// 追加プロパティに対して発行するGUIDの辞書です。
        /// パース処理中に設定される値です。
        /// </summary>
        public Dictionary<string, Guid> AdditionalPropertyGuids
        {
            get
            {
                return this.additionalPropertyGuids ?? (this.additionalPropertyGuids = new Dictionary<string, Guid>());
            }
        }

        /// <summary>
        /// カラープロパティか否かを表します。
        /// </summary>
        public bool IsColor
        {
            get
            {
                switch (this.Type)
                {
                    case PropertyType.ColorRgba:
                    case PropertyType.ColorRgb:
                    case PropertyType.ColorA:
                    case PropertyType.ColorAnimeRgb:
                    case PropertyType.ColorAnimeA:
                        return true;
                    default:
                        return false;
                }
            }
        }

        /// <summary>
        /// プリミティブ型が必要なプロパティか否かを表します。
        /// </summary>
        public bool IsPrimitive
        {
            get
            {
                switch (this.Type)
                {
                    case PropertyType.ColorRgba:
                    case PropertyType.ColorRgb:
                    case PropertyType.ColorA:
                    case PropertyType.ColorAnimeRgb:
                    case PropertyType.ColorAnimeA:
                    case PropertyType.Slider2F:
                    case PropertyType.Slider2I:
                    case PropertyType.Slider3F:
                    case PropertyType.Slider3I:
                    case PropertyType.Slider4F:
                    case PropertyType.Slider4I:
                        return true;
                    default:
                        return false;
                }
            }
        }

        /// <summary>
        /// アニメーション型プロパティか否かを表します。
        /// </summary>
        public bool IsAnimation
        {
            get
            {
                switch (this.Type)
                {
                    case PropertyType.ColorAnimeRgb:
                    case PropertyType.ColorAnimeA:
                        return true;
                    default:
                        return false;
                }
            }
        }

        /// <summary>
        /// コンボボックス型プロパティか否かを表します。
        /// </summary>
        public bool IsComboBox
        {
            get
            {
                switch (this.Type)
                {
                    // PropertyType.ComboBox は別扱い
                    case PropertyType.StringComboBox:
                    case PropertyType.IntComboBox:
                        return true;
                    default:
                        return false;
                }
            }
        }
    }

    /// <summary>
    /// UIレイアウトのグループ定義
    /// </summary>
    public class UIGroup
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UIGroup()
        {
            this.Parameters = new List<ParameterDefinition>();
        }

        /// <summary>
        /// グループ名です。グループボックスのキャプションに表示されます。
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// グループに配置されるパラメータ定義です。
        /// </summary>
        public List<ParameterDefinition> Parameters { get; private set; }

        /// <summary>
        /// カラープロパティを含むか否かを表します。
        /// </summary>
        public bool IsColorGroup
        {
            get { return this.Parameters.Any(p => p.IsColor); }
        }
    }

    /// <summary>
    /// UIレイアウトの定義
    /// </summary>
    public class UILayout
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UILayout()
        {
            this.Groups = new List<UIGroup>();
        }

        /// <summary>
        /// グループリストです。
        /// </summary>
        public List<UIGroup> Groups { get; private set; }
    }

    /// <summary>
    /// UDD2.0の定義
    /// </summary>
    public class UserDataDefinition
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserDataDefinition()
        {
            this.ParameterDefinitions = new List<ParameterDefinition>();
        }

        /// <summary>
        /// UDDの定義対象となるユーザーデータの種類を表します。
        /// </summary>
        public UserDataKind Kind { get; set; }

        /// <summary>
        /// UDDの名前です。カスタムシェーダ・アクションにおけるコールバック名になります。
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// UDDのバージョンです。
        /// </summary>
        public int Version { get; set; }

        /// <summary>
        /// パラメータ定義リストです。
        /// </summary>
        public List<ParameterDefinition> ParameterDefinitions { get; private set; }

        /// <summary>
        /// UIレイアウト定義です。
        /// </summary>
        public UILayout Layout { get; set; }
    }
}
