﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.DataModelMaker.Core.Template;
using EffectMaker.Udd2Converter.Properties;

namespace EffectMaker.Udd2Converter
{
    /// <summary>
    /// UDD2.0定義からバイナリコンバート情報を生成するクラスです。
    /// </summary>
    internal class Udd2BinaryConversionInfoGenerator
    {
        /// <summary>
        /// UDD2.0定義
        /// </summary>
        private readonly UserDataDefinition uddInfo;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="uddInfo">パース済みのUDD2.0定義</param>
        public Udd2BinaryConversionInfoGenerator(UserDataDefinition uddInfo)
        {
            this.uddInfo = uddInfo;
        }

        /// <summary>
        /// バイナリコンバート定義ソースファイルを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデルおよびファイル名</param>
        /// <param name="outputPath">出力先フォルダパス</param>
        /// <returns>成功すればtrue,失敗すればfalse.</returns>
        public bool Generate(string dataModelName, string outputPath)
        {
            string code;
            var result = this.GenerateSourceCode(dataModelName, out code);
            if (result)
            {
                File.WriteAllText(Path.Combine(outputPath, dataModelName + "BinaryConversionInfo.cs"), code);
            }

            return result;
        }

        /// <summary>
        /// バイナリコンバート定義コードを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデル名</param>
        /// <param name="output">出力先インスタンス</param>
        /// <returns>成功すればtrue,失敗したらfalse.</returns>
        public bool GenerateSourceCode(string dataModelName, out string output)
        {
            var templateHelper = new TemplateHelper();
            templateHelper.ParseSimpleTemplate(Resources.Udd2_BCI_Main);

            templateHelper.SetTemplateTagValue("DataModelClassName", dataModelName);

            templateHelper.SetTemplateTagValue("BinaryDefinitionBlock", MakeBinaryDefinitionBlock(dataModelName, this.uddInfo));

            output = templateHelper.Compose();

            return true;
        }

        /// <summary>
        /// バイナリ定義ブロックを生成します。
        /// </summary>
        /// <param name="dataModelName">データモデル名</param>
        /// <param name="uddInfo">UDD2.0定義</param>
        /// <returns>生成コード</returns>
        private static string MakeBinaryDefinitionBlock(string dataModelName, UserDataDefinition uddInfo)
        {
            var builder = new StringBuilder();
            var templateHelper = new TemplateHelper();
            templateHelper.ParseSimpleTemplate(Resources.Udd2_BCI_ConversionInfo);
            foreach (var paramDef in uddInfo.ParameterDefinitions)
            {
                if (uddInfo.Kind == UserDataKind.CustomShader &&
                    Udd2Parser.CustomShaderBuiltInNames.Contains(paramDef.MemberName))
                {
                    // カスタムシェーダのビルドインデータは生成コードに含めない
                    continue;
                }

                // 共通のタグ設定
                Action<TemplateHelper, ParameterDefinition> applyCommonValues = (t, p) =>
                {
                    t.SetTemplateTagValue("DataModelPropertyName", p.MemberName);
                    t.SetTemplateTagValue("DataModelClassName", dataModelName);
                    t.SetTemplateTagValue("ModificationType", p.NeedReload ? "FullBinary" : "ModifiedDataOnly");
                    t.SetTemplateTagValue("DataModelPropertyGuid", p.Guid.ToString());
                };

                if (paramDef.IsAnimation)
                {
                    // アニメーションデータは専用のテンプレートで特殊対応
                    var animTemplateHelper = new TemplateHelper();
                    animTemplateHelper.ParseSimpleTemplate(Resources.Udd2_BCI_AnimationDataConversionInfo);

                    // 共通設定を適用
                    applyCommonValues(animTemplateHelper, paramDef);

                    // ループ・開始位置ランダム・ループフレーム数を追加
                    animTemplateHelper.SetTemplateTagValue(
                        "EnableLoopGuid", paramDef.AdditionalPropertyGuids[paramDef.MemberName + "EnableLoop"].ToString());
                    animTemplateHelper.SetTemplateTagValue(
                        "EnableStartRandomGuid", paramDef.AdditionalPropertyGuids[paramDef.MemberName + "EnableStartRandom"].ToString());
                    animTemplateHelper.SetTemplateTagValue(
                        "LoopFrameGuid", paramDef.AdditionalPropertyGuids[paramDef.MemberName + "LoopFrame"].ToString());

                    // カラーかアルファかでコンバータを切り替え
                    animTemplateHelper.SetTemplateTagValue(
                        "ConverterSetting",
                        paramDef.Type == PropertyType.ColorAnimeRgb ?
                        "new ConverterDefinition(typeof(AnimationTableConverter)," +
                            " new ConverterParam(\"MaxKeyFrameCount\", 8)," +
                            " new ConverterParam(\"FrameConvertMode\", 0))" :
                        "new ConverterDefinition(typeof(AlphaAnimationTableConverter)," +
                            " new ConverterParam(\"MaxKeyFrameCount\", 8))");

                    builder.AppendLine(animTemplateHelper.Compose());
                    continue;
                }

                // 他の型は共通処理
                templateHelper.ResetTemplateTags();

                // 共通設定を適用
                applyCommonValues(templateHelper, paramDef);

                // コンバータを設定
                if (paramDef.Type == PropertyType.StringComboBox)
                {
                    // 今回特製のコンボボックスアイテムリスト＆インデックスからの文字列取得コンバータを使用します。
                    templateHelper.SetTemplateTagValue(
                        "ConverterName",
                        string.Format(
                            "typeof(ComboBoxStringConverter), new ConverterParam(\"CharArrayLength\", {0})",
                            paramDef.ArraySize));
                    var option = (OptionGroup)paramDef.UISetting;
                    templateHelper.SetTemplateTagValue(
                        "AdditionalInputDefinition",
                        string.Format(
                            ", new InputPropertyDefinition(\"{0}\", new Guid(\"{1}\"))",
                            paramDef.MemberName + "Items",
                            option.Guid));
                }
                else if (paramDef.Type == PropertyType.TextBox)
                {
                    templateHelper.SetTemplateTagValue(
                        "ConverterName",
                        string.Format(
                            "typeof(FixSizeStringConverter), new ConverterParam(\"CharArrayLength\", {0})",
                            paramDef.ArraySize));
                }
                else if (paramDef.Type == PropertyType.FilePathTextBox)
                {
                    templateHelper.SetTemplateTagValue(
                        "ConverterName",
                        string.Format(
                            "typeof(FixSizeStringConverter), new ConverterParam(\"CharArrayLength\", {0})",
                            paramDef.ArraySize));
                }
                else if (paramDef.Type == PropertyType.ColorRgb)
                {
                    templateHelper.SetTemplateTagValue("ConverterName", "typeof(ColorToVec3Converter)");
                }
                else if (paramDef.Type == PropertyType.ColorA)
                {
                    templateHelper.SetTemplateTagValue("ConverterName", "typeof(ColorToFloatAlphaConverter)");
                }
                else
                {
                    templateHelper.SetTemplateTagValue("ConverterName", "typeof(DefaultConverter)");
                }

                builder.AppendLine(templateHelper.Compose());
            }

            XmlParseUtil.ChopDoubleNewLine(builder);
            return builder.ToString();
        }
    }
}
