﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Globalization;
using EffectMaker.DataModelLogic;
using EffectMaker.DataModelMaker.Core.Definitions;

namespace EffectMaker.ObsoleteUserDataConverter.ObsoleteUserData
{
    /// <summary>
    /// Generates data model and binary data definitions from user definition data.
    /// </summary>
    public class UserDefinitionDataModelGenerator : IDisposable
    {
        /// <summary>The editor data model definition.</summary>
        private EditorDataModelDefinition editorDataModelDef = null;

        /// <summary>The runtime data model definition.</summary>
        private RuntimeDataModelDefinition runtimeDataModelDef = null;

        /// <summary>The source data model instance definition for the binary data.</summary>
        private SourceDataModelInstanceDefinition sourceDataModelDef = null;

        /// <summary>The binary field group definition for the runtime data model instance.</summary>
        private BinaryFieldGroupDefinition binaryFieldGroupDef = null;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserDefinitionDataModelGenerator()
        {
            this.DataModelBaseClassName = string.Empty;
            this.SkipFirstCheckBoxGroup = false;
            this.SkipFirstRadioButtonGroup = false;
        }

        /// <summary>
        /// Get or set the name of the data model base class.
        /// </summary>
        public string DataModelBaseClassName { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to skip the generation of
        /// the data model property for the first check box group.
        /// </summary>
        public bool SkipFirstCheckBoxGroup { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to skip the generation of
        /// the data model property for the first radio button group.
        /// </summary>
        public bool SkipFirstRadioButtonGroup { get; set; }

        /// <summary>
        /// Get the generated workspace definition.
        /// </summary>
        public WorkspaceDefinition WorkspaceDefinition { get; private set; }

        /// <summary>
        /// Dispose the instance.
        /// </summary>
        public void Dispose()
        {
            this.WorkspaceDefinition.Dispose();
            this.WorkspaceDefinition = null;
        }

        /// <summary>
        /// Generate data model and binary data definitions with the given information.
        /// </summary>
        /// <param name="def">The group definition to generate from.</param>
        /// <param name="dataModelName">The name of the data models.</param>
        /// <returns>True on success.</returns>
        public bool Generate(GroupDefinition def, string dataModelName)
        {
            this.SetupInitialDefinitions(dataModelName);
            this.GenerateFromGroupDefinition(def);

            return true;
        }

        /// <summary>
        /// Generate data model definitions from the group definition and its children.
        /// </summary>
        /// <param name="def">The group definition.</param>
        /// <returns>True on success.</returns>
        private bool GenerateFromGroupDefinition(GroupDefinition def)
        {
            if (def.GroupType == "CheckBox")
            {
                return this.GenerateCheckBoxGroupProperty(def);
            }
            else if (def.GroupType == "RadioButton")
            {
                return this.GenerateRadioButtonGroupProperty(def);
            }
            else
            {
                bool result = true;
                foreach (UserDefinitionBase child in def.Controls)
                {
                    if (child is GroupDefinition)
                    {
                        result &= this.GenerateFromGroupDefinition((GroupDefinition)child);
                    }
                    else if (child is IntSliderDefinition)
                    {
                        result &= this.GenerateIntSliderProperty(child);
                    }
                    else if (child is FloatSliderDefinition)
                    {
                        result &= this.GenerateFloatSliderProperty(child);
                    }
                }

                return result;
            }
        }

        /// <summary>
        /// Generate data model property with the given check box group definition.
        /// </summary>
        /// <param name="def">The check box group definition.</param>
        /// <returns>True on success.</returns>
        private bool GenerateCheckBoxGroupProperty(UserDefinitionBase def)
        {
            var castedUserDef = def as GroupDefinition;
            if (castedUserDef == null || castedUserDef.GroupType != "CheckBox")
            {
                return false;
            }

            // Do not generate the property for the first check box group.
            if (this.SkipFirstCheckBoxGroup == true && def.Index == 0)
            {
                return true;
            }

            // Determine the default value.
            uint defaultValue = 0x00000000;
            uint duplicationTester = 0x00000000;
            foreach (UserDefinitionBase childDef in castedUserDef.Controls)
            {
                var checkBoxDef = childDef as CheckBoxDefinition;
                if (checkBoxDef == null)
                {
                    continue;
                }

                // A check box group binds to a uint (32-bit), so the index has
                // to be between 0 to 31, each of them represents a bit in the uint.
                if (checkBoxDef.Index < 0 || checkBoxDef.Index >= 32)
                {
                    continue;
                }

                uint value = (uint)(0x00000001 << checkBoxDef.Index);
                if ((duplicationTester & value) != 0)
                {
                    // A check box with the same index has already been defined.
                    // Skip this one.
                    continue;
                }

                // Register this bit so we know a check box with this index is already
                // defined.
                duplicationTester |= value;

                // If the check box has default value as true, turn on the bit that
                // corresponds to its index.
                if (checkBoxDef.DefaultValue == true)
                {
                    defaultValue |= value;
                }
            }

            // Create the editor data model property definition.
            var editorPropertyDef = new EditorDataModelPropertyDefinition(0)
            {
                Name = "Flag" + def.Index,
                Namespace = string.Empty,
                Type = "uint",
                Description = "Get or set the flag value " + def.Index + ".",
                DefaultValue = string.Format("0x{0:X8}", defaultValue),
            };

            // Create the runtime data model property definition.
            var runtimePropertyDef = new RuntimeDataModelPropertyDefinition()
            {
                Name = "flag" + def.Index,
                TypeNamespace = string.Empty,
                TypeName = "u32",
                Description = "The flag value " + def.Index + ".",
            };

            this.SetupPropertyDefinitions(editorPropertyDef, runtimePropertyDef);

            return true;
        }

        /// <summary>
        /// Generate data model property with the given radio button group definition.
        /// </summary>
        /// <param name="def">The radio button group definition.</param>
        /// <returns>True on success.</returns>
        private bool GenerateRadioButtonGroupProperty(UserDefinitionBase def)
        {
            var castedUserDef = def as GroupDefinition;
            if (castedUserDef == null || castedUserDef.GroupType != "RadioButton")
            {
                return false;
            }

            // ラジオボタンの最初の8個は基底クラス側で持つので、ここでは生成しない
            if (this.SkipFirstRadioButtonGroup == true && def.Index < 8)
            {
                return true;
            }

            // Determine the default value.
            uint defaultValue = 0x00000000;
            uint duplicationTester = 0x00000000;
            foreach (UserDefinitionBase childDef in castedUserDef.Controls)
            {
                var radioButtonDef = childDef as RadioButtonDefinition;
                if (radioButtonDef == null)
                {
                    continue;
                }

                // A radio button group binds to a uint (32-bit), so the index has
                // to be between 0 to 31, each of them represents a bit in the uint.
                if (radioButtonDef.Index < 0 || radioButtonDef.Index >= 32)
                {
                    continue;
                }

                uint value = (uint)(0x00000001 << radioButtonDef.Index);
                if ((duplicationTester & value) != 0)
                {
                    // A check box with the same index has already been defined.
                    // Skip this one.
                    continue;
                }

                // Register this bit so we know a check box with this index is already
                // defined.
                duplicationTester |= value;

                // Use the index of the first radio button that the default value is true.
                if (radioButtonDef.DefaultValue == true)
                {
                    defaultValue = value;
                    break;
                }
            }

            // Create the editor data model property definition.
            var editorPropertyDef = new EditorDataModelPropertyDefinition(0)
            {
                Name = "Switch" + def.Index,
                Namespace = string.Empty,
                Type = "uint",
                Description = "Get or set the switch value " + def.Index + ".",
                DefaultValue = string.Format("0x{0:X8}", defaultValue),
            };

            // Create the runtime data model property definition.
            var runtimePropertyDef = new RuntimeDataModelPropertyDefinition()
            {
                Name = "switch" + def.Index,
                TypeNamespace = string.Empty,
                TypeName = "u32",
                Description = "The switch value " + def.Index + ".",
            };

            this.SetupPropertyDefinitions(editorPropertyDef, runtimePropertyDef);

            return true;
        }

        /// <summary>
        /// Generate data model property with the given integer slider definition.
        /// </summary>
        /// <param name="def">The integer slider definition.</param>
        /// <returns>True on success.</returns>
        private bool GenerateIntSliderProperty(UserDefinitionBase def)
        {
            var castedUserDef = def as IntSliderDefinition;
            if (castedUserDef == null)
            {
                return false;
            }

            // Create the editor data model property definition.
            var editorPropertyDef = new EditorDataModelPropertyDefinition(0)
            {
                Name = "IntParam" + def.Index,
                Namespace = string.Empty,
                Type = "int",
                Description = "Get or set the integer parameter " + def.Index + ".",
                DefaultValue = castedUserDef.DefaultValue.ToString(CultureInfo.InvariantCulture),
            };

            // Create the runtime data model property definition.
            var runtimePropertyDef = new RuntimeDataModelPropertyDefinition()
            {
                Name = "intParam" + def.Index,
                TypeNamespace = string.Empty,
                TypeName = "s32",
                Description = "The integer parameter " + def.Index + ".",
            };

            this.SetupPropertyDefinitions(editorPropertyDef, runtimePropertyDef);

            return true;
        }

        /// <summary>
        /// Generate data model property with the given floating point slider definition.
        /// </summary>
        /// <param name="def">The floating point slider definition.</param>
        /// <returns>True on success.</returns>
        private bool GenerateFloatSliderProperty(UserDefinitionBase def)
        {
            var castedUserDef = def as FloatSliderDefinition;
            if (castedUserDef == null)
            {
                return false;
            }

            // Create the editor data model property definition.
            var editorPropertyDef = new EditorDataModelPropertyDefinition(0)
            {
                Name = "FloatParam" + def.Index,
                Namespace = string.Empty,
                Type = "float",
                Description = "Get or set the floating point parameter " + def.Index + ".",
                DefaultValue = castedUserDef.DefaultValue.ToString(CultureInfo.InvariantCulture) + "f",
            };

            // Create the runtime data model property definition.
            var runtimePropertyDef = new RuntimeDataModelPropertyDefinition()
            {
                Name = "floatParam" + def.Index,
                TypeNamespace = string.Empty,
                TypeName = "f32",
                Description = "The floating point parameter " + def.Index + ".",
            };

            this.SetupPropertyDefinitions(editorPropertyDef, runtimePropertyDef);

            return true;
        }

        /// <summary>
        /// Set up the initial definitions and make them ready for
        /// adding data model properties.
        /// </summary>
        /// <param name="dataModelName">The data model name.</param>
        private void SetupInitialDefinitions(string dataModelName)
        {
            // Create the editor data model definition.
            this.editorDataModelDef = new EditorDataModelDefinition()
            {
                CreatedVersionIndex = 0,
                Namespace = "EffectMaker.DataModel.Specific.DataModels",
                Name = dataModelName,
                Description = "The user data data model class.",
                IsVersionEnabled = false,
            };

            if (string.IsNullOrEmpty(this.DataModelBaseClassName) == false)
            {
                // Set up the base class for the data model.
                var baseClass = new EditorDataModelDefinition()
                {
                    CreatedVersionIndex = 0,
                    Namespace = "EffectMaker.DataModel.Specific.DataModels",
                    Name = this.DataModelBaseClassName,
                };

                this.editorDataModelDef.SuperClasses.Add(baseClass.Guid);
            }

            // Create the runtime data model definition.
            this.runtimeDataModelDef = new RuntimeDataModelDefinition()
            {
                FileName = dataModelName + ".h",
                Name = dataModelName,
                Description = "The user data class.",
            };

            // Create the binary data definition.
            var binaryDataDef = new BinaryDataDefinition()
            {
                Name = dataModelName + "BinaryData",
                FileName = dataModelName + "BinaryConversionInfo",
                HasBinaryHeader = true,
                IsUserData = true,
            };

            // Create the source data model instance definition for the binary data.
            this.sourceDataModelDef = new SourceDataModelInstanceDefinition()
            {
                DataModelDefinitionGuid = this.editorDataModelDef.Guid,
            };

            binaryDataDef.SourceDataModelInstance = this.sourceDataModelDef;

            // Create the binary field and field group definitions for the runtime data model.
            var binaryFieldDef = new BinaryFieldDefinition()
            {
                OutputFieldGuid = this.runtimeDataModelDef.Guid,
                FieldType = BinaryFieldTypes.Normal,
                SendModificationType = SendModificationTypes.ModifiedDataOnly,
            };

            this.binaryFieldGroupDef = new BinaryFieldGroupDefinition()
            {
                TargetGuid = this.runtimeDataModelDef.Guid,
            };

            binaryFieldDef.GroupDefinition = this.binaryFieldGroupDef;
            binaryDataDef.Fields.Add(binaryFieldDef);

            // Create the root definitions.
            this.WorkspaceDefinition = new WorkspaceDefinition()
            {
                EditorDataModelDefinition = new EditorDataModelRootDefinition() { IsVersionEnabled = false },
                RuntimeDataModelDefinition = new RuntimeDataModelRootDefinition(),
            };

            this.WorkspaceDefinition.EditorDataModelDefinition.EditingVersion =
                this.WorkspaceDefinition.EditorDataModelDefinition.LatestVersion;

            this.WorkspaceDefinition.EditorDataModelDefinition.DataModels.Add(this.editorDataModelDef);
            this.WorkspaceDefinition.RuntimeDataModelDefinition.DataModels.Add(this.runtimeDataModelDef);
            this.WorkspaceDefinition.RuntimeDataModelDefinition.BinaryDatas.Add(binaryDataDef);
        }

        /// <summary>
        /// Helper method for setting up the created property definitions.
        /// </summary>
        /// <param name="editorPropertyDef">The editor data model property definition.</param>
        /// <param name="runtimePropertyDef">The runtime data model property definition.</param>
        private void SetupPropertyDefinitions(
            EditorDataModelPropertyDefinition editorPropertyDef,
            RuntimeDataModelPropertyDefinition runtimePropertyDef)
        {
            // Add the property definitions to the data model definitions.
            this.editorDataModelDef.AllProperties.Add(editorPropertyDef);
            this.runtimeDataModelDef.Properties.Add(runtimePropertyDef);

            // Create the source data model property definition for the editor property.
            var sourcePropertyInstance = new SourcePropertyInstanceDefinition()
            {
                PropertyDefinitionGuid = editorPropertyDef.Guid,
            };

            this.sourceDataModelDef.Properties.Add(sourcePropertyInstance);

            // Create the binary field definition for the runtime property.
            var binaryFieldDef = new BinaryFieldDefinition()
            {
                FieldType = BinaryFieldTypes.Normal,
                OutputFieldGuid = runtimePropertyDef.Guid,
                SendModificationType = SendModificationTypes.ModifiedDataOnly,
            };

            binaryFieldDef.InputPropertyGuidList.Add(sourcePropertyInstance.Guid);

            this.binaryFieldGroupDef.Fields.Add(binaryFieldDef);
        }
    }
}
