﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Reflection;
    using System.Resources;

    /// <summary>
    /// 文字列リソースクラス。
    /// リソースから文字列を取得できます。
    /// </summary>
    public static class Strings
    {
        private static string folderString = ".resources.Strings";
        private static string endString1 = folderString + ".Strings";
        private static string endString2 = folderString + ".Strings.resources";
        private static Dictionary<Assembly, ResourceManager> stringDictionary =
            new Dictionary<Assembly, ResourceManager>();

        /// <summary>
        /// リソースマネージャ作成時の引数の文字列はこの文字列で終わる
        /// </summary>
        public static string EndString1
        {
            get { return Strings.endString1; }
            set { Strings.endString1 = value; }
        }

        /// <summary>
        /// 文字列リソース名の文字列はこの文字列で終わる約束
        /// </summary>
        public static string EndString2
        {
            get { return Strings.endString2; }
            set { Strings.endString2 = value; }
        }

        /// <summary>
        /// リソースから文字列を取得する
        /// </summary>
        /// <param name="type">指定Type型を含むアセンブリを取得するためのTypeオブジェクトです。</param>
        /// <param name="key">リソースから取得する文字列のキーです。</param>
        /// <returns>文字列を返します。</returns>
        public static string Get(Type type, string key)
        {
            Assembly assembly = Assembly.GetAssembly(type);

            string resourceName = type.ToString() + "." + key;

            if (stringDictionary.ContainsKey(assembly))
            {
                // 登録済みの Assembly の場合
                return stringDictionary[assembly].GetString(resourceName);
            }
            else
            {
                // 未登録の Assembly の場合

                // リソースマネージャ作成
                ResourceManager resourceManager = GetResourceManager(assembly);

                // 辞書に追加
                stringDictionary.Add(assembly, resourceManager);

                // リソースマネージャ経由で文字列取得
                return resourceManager.GetString(resourceName);
            }
        }

        /// <summary>
        /// リソースマネージャを取得する
        /// </summary>
        /// <param name="assembly">取得対象のアセンブリです。</param>
        /// <returns>リソースマネージャーを返します。</returns>
        private static ResourceManager GetResourceManager(Assembly assembly)
        {
            string baseName = GetBaseNameSpace(assembly) + Strings.endString1;
            return new ResourceManager(baseName, assembly);
        }

        /// <summary>
        /// アセンブリ内の文字列リソース名から既定の名前空間を取得する
        /// </summary>
        /// <param name="assembly">取得対象のアセンブリです。</param>
        /// <returns>既定の名前空間を文字列として返します。</returns>
        private static string GetBaseNameSpace(Assembly assembly)
        {
            // 以下のまわりくどい処理は
            // 既定の名前空間を直接取得する関数が見当たらないため

            // アセンブリのリソース名を取得する
            string[] resourceNames = assembly.GetManifestResourceNames();

            // リソースが見つからない場合
            Ensure.Argument.True(resourceNames.Length >= 1);

            // [.res.Strings.resources] で終わるリソース名を探す
            string resourceName = string.Empty;
            foreach (string name in resourceNames)
            {
                if (name.EndsWith(Strings.endString2))
                {
                    resourceName = name;
                    break;
                }
            }

            // 合致するリソース名が見つからない場合
            Ensure.Argument.True(resourceName != string.Empty);

            // ドットで文字列を分解する
            string[] separator = { Strings.endString2 };
            string[] baseNameSpace =
                resourceName.Split(separator, StringSplitOptions.RemoveEmptyEntries);

            // 既定の名前空間が取得できない場合
            Ensure.Argument.True(baseNameSpace.Length >= 1);

            return baseNameSpace[0];
        }

        /// <summary>
        /// アセンブリ内のリソースセットを取得する
        /// </summary>
        /// <param name="assembly">取得対象のアセンブリです。</param>
        /// <returns>リソースセットを返します。</returns>
        private static ResourceSet GetResourceSet(Assembly assembly)
        {
            // アセンブリのリソース名を取得する
            string[] resourceNames = assembly.GetManifestResourceNames();

            // リソースのストリームを取得する
            Stream stream =
                assembly.GetManifestResourceStream(resourceNames[0]);

            // リソースセットを取得する
            ResourceSet resourceSet = new ResourceSet(stream);

            return resourceSet;
        }
    }
}
