﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    using System;

    /// <summary>
    /// 拡大縮小、回転、平行移動を表します。
    /// </summary>
    public sealed class Transform : ICloneable, IEquatable<Transform>, ISettable
    {
        //-----------------------------------------------------------------
        private readonly Vector3 scale = new Vector3(1f, 1f, 1f);
        private readonly Vector3 rotate = new Vector3();
        private readonly Vector3 translate = new Vector3();

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Transform()
        {
        }

        /// <summary>
        /// Transform の成分を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="scale">拡大縮小成分です。</param>
        /// <param name="rotate">回転成分です。</param>
        /// <param name="translate">平行移動成分です。</param>
        public Transform(Vector3 scale, Vector3 rotate, Vector3 translate)
        {
            this.Set(scale, rotate, translate);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Transform(Transform source)
        {
            this.Set(source);
        }

        //-----------------------------------------------------------------
        // プロパティの取得または設定
        //-----------------------------------------------------------------

        //-----------------------------------------------------------------

        /// <summary>
        /// 拡大縮小を取得または設定します。
        /// </summary>
        public Vector3 Scale
        {
            get
            {
                return this.scale;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.scale.Set(value);
            }
        }

        /// <summary>
        /// 回転を取得または設定します。
        /// </summary>
        public Vector3 Rotate
        {
            get
            {
                return this.rotate;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.rotate.Set(value);
            }
        }

        /// <summary>
        /// 平行移動を取得または設定します。
        /// </summary>
        public Vector3 Translate
        {
            get
            {
                return this.translate;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.translate.Set(value);
            }
        }

        /// <summary>
        /// 恒等変換かを取得します。
        /// </summary>
        public bool IsIdentity
        {
            get
            {
                return
                    this.translate.Equals(Vector3.Zero) &&
                    this.rotate.Equals(Vector3.Zero) &&
                    this.Scale.Equals(Vector3.One);
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new Transform(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as Transform);
        }

        /// <summary>
        /// Transform を設定します。
        /// </summary>
        /// <param name="source">設定する Transform です。</param>
        public void Set(Transform source)
        {
            Ensure.Argument.NotNull(source);

            this.Scale.Set(source.Scale);
            this.Rotate.Set(source.Rotate);
            this.Translate.Set(source.Translate);
        }

        /// <summary>
        /// 拡大縮小、回転、平行移動を指定の値に設定します。
        /// </summary>
        /// <param name="scale">拡大縮小成分です。</param>
        /// <param name="rotate">回転成分です。</param>
        /// <param name="translate">平行移動成分です。</param>
        public void Set(Vector3 scale, Vector3 rotate, Vector3 translate)
        {
            Ensure.Argument.NotNull(scale);
            Ensure.Argument.NotNull(rotate);
            Ensure.Argument.NotNull(translate);

            this.Scale.Set(scale);
            this.Rotate.Set(rotate);
            this.Translate.Set(translate);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 成分を全て 0 に設定します。
        /// </summary>
        public void SetZero()
        {
            this.Scale.SetZero();
            this.Rotate.SetZero();
            this.Translate.SetZero();
        }

        /// <summary>
        /// 拡大縮小成分は 1 に、それ以外は 0 に設定します。
        /// </summary>
        public void SetIdentity()
        {
            this.Scale.Set(1f, 1f, 1f);
            this.Rotate.SetZero();
            this.Translate.SetZero();
        }

        //-----------------------------------------------------------------
        // 文字列化
        //-----------------------------------------------------------------

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return string.Format(
                "({0} {1} {2})",
                this.Scale.ToString(),
                this.Rotate.ToString(),
                this.Translate.ToString());
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Transform);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Transform other)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                this.Scale.Equals(other.Scale) &&
                this.Rotate.Equals(other.Rotate) &&
                this.Translate.Equals(other.Translate);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <param name="tolerance">各成分毎の差の最大許容値です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Transform other, float tolerance)
        {
            if (other == this)
            {
                return true;
            }

            if (other == null)
            {
                return false;
            }

            return
                this.Scale.Equals(other.Scale, tolerance) &&
                this.Rotate.Equals(other.Rotate, tolerance) &&
                this.Translate.Equals(other.Translate, tolerance);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return
                this.Scale.GetHashCode() ^
                this.Rotate.GetHashCode() ^
                this.Translate.GetHashCode();
        }
    }
}
