﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// 4 × 4 行列を表します。
    /// </summary>
    public sealed class Matrix44 : ObservableObject, IMatrix, IEquatable<Matrix44>
    {
        //-----------------------------------------------------------------
        private const int MatrixRowCount = 4;
        private const int MatrixColumnCount = 4;
        private float[,] m = new float[MatrixRowCount, MatrixColumnCount]
            {
                { 1f, 0f, 0f, 0f },
                { 0f, 1f, 0f, 0f },
                { 0f, 0f, 1f, 0f },
                { 0f, 0f, 0f, 1f },
            };

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Matrix44()
        {
        }

        /// <summary>
        /// 行列の成分を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="m00">[0, 0]成分です。</param>
        /// <param name="m01">[0, 1]成分です。</param>
        /// <param name="m02">[0, 2]成分です。</param>
        /// <param name="m03">[0, 3]成分です。</param>
        /// <param name="m10">[1, 0]成分です。</param>
        /// <param name="m11">[1, 1]成分です。</param>
        /// <param name="m12">[1, 2]成分です。</param>
        /// <param name="m13">[1, 3]成分です。</param>
        /// <param name="m20">[2, 0]成分です。</param>
        /// <param name="m21">[2, 1]成分です。</param>
        /// <param name="m22">[2, 2]成分です。</param>
        /// <param name="m23">[2, 3]成分です。</param>
        /// <param name="m30">[3, 0]成分です。</param>
        /// <param name="m31">[3, 1]成分です。</param>
        /// <param name="m32">[3, 2]成分です。</param>
        /// <param name="m33">[3, 3]成分です。</param>
        public Matrix44(
            float m00,
            float m01,
            float m02,
            float m03,
            float m10,
            float m11,
            float m12,
            float m13,
            float m20,
            float m21,
            float m22,
            float m23,
            float m30,
            float m31,
            float m32,
            float m33)
        {
            this.Set(
                m00, m01, m02, m03, m10, m11, m12, m13, m20, m21, m22, m23, m30, m31, m32, m33);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Matrix44(Matrix44 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Matrix44(Matrix33 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public Matrix44(Matrix34 source)
        {
            this.Set(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="row0">第０行の成分を格納したベクトルです。</param>
        /// <param name="row1">第１行の成分を格納したベクトルです。</param>
        /// <param name="row2">第２行の成分を格納したベクトルです。</param>
        /// <param name="row3">第３行の成分を格納したベクトルです。</param>
        public Matrix44(Vector4 row0, Vector4 row1, Vector4 row2, Vector4 row3)
        {
            this.Set(row0, row1, row2, row3);
        }

        //-----------------------------------------------------------------
        // プロパティの取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// 単位行列を作成します。
        /// </summary>
        /// <returns>単位行列です。</returns>
        public static Matrix44 Identity
        {
            get
            {
                return new Matrix44();
            }
        }

        /// <summary>
        /// 成分が全て 0 の行列を作成します。
        /// </summary>
        /// <returns>成分が全て 0 の行列です。</returns>
        public static Matrix44 Zero
        {
            get
            {
                Matrix44 result = new Matrix44();
                result.SetZero();
                return result;
            }
        }

        /// <summary>
        /// 列数を取得します。
        /// </summary>
        public int ColumnCount
        {
            get
            {
                return MatrixColumnCount;
            }
        }

        /// <summary>
        /// 行数を取得します。
        /// </summary>
        public int RowCount
        {
            get
            {
                return MatrixRowCount;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 行列式を取得します。
        /// </summary>
        /// <returns>行列式の値です。</returns>
        public float Determinant
        {
            get
            {
                return
                    (this.m[0, 0] * this.m[1, 1] * this.m[2, 2] * this.m[3, 3]) +
                    (this.m[0, 0] * this.m[1, 2] * this.m[2, 3] * this.m[3, 1]) +
                    (this.m[0, 0] * this.m[1, 3] * this.m[2, 1] * this.m[3, 2]) +
                    (this.m[0, 1] * this.m[1, 0] * this.m[2, 3] * this.m[3, 2]) +
                    (this.m[0, 1] * this.m[1, 2] * this.m[2, 0] * this.m[3, 3]) +
                    (this.m[0, 1] * this.m[1, 3] * this.m[2, 2] * this.m[3, 0]) +
                    (this.m[0, 2] * this.m[1, 0] * this.m[2, 1] * this.m[3, 3]) +
                    (this.m[0, 2] * this.m[1, 1] * this.m[2, 3] * this.m[3, 0]) +
                    (this.m[0, 2] * this.m[1, 3] * this.m[2, 0] * this.m[3, 1]) +
                    (this.m[0, 3] * this.m[1, 0] * this.m[2, 2] * this.m[3, 1]) +
                    (this.m[0, 3] * this.m[1, 1] * this.m[2, 0] * this.m[3, 2]) +
                    (this.m[0, 3] * this.m[1, 2] * this.m[2, 1] * this.m[3, 0]) -
                    (this.m[0, 0] * this.m[1, 1] * this.m[2, 3] * this.m[3, 2]) -
                    (this.m[0, 0] * this.m[1, 2] * this.m[2, 1] * this.m[3, 3]) -
                    (this.m[0, 0] * this.m[1, 3] * this.m[2, 2] * this.m[3, 1]) -
                    (this.m[0, 1] * this.m[1, 0] * this.m[2, 2] * this.m[3, 3]) -
                    (this.m[0, 1] * this.m[1, 2] * this.m[2, 3] * this.m[3, 0]) -
                    (this.m[0, 1] * this.m[1, 3] * this.m[2, 0] * this.m[3, 2]) -
                    (this.m[0, 2] * this.m[1, 0] * this.m[2, 3] * this.m[3, 1]) -
                    (this.m[0, 2] * this.m[1, 1] * this.m[2, 0] * this.m[3, 3]) -
                    (this.m[0, 2] * this.m[1, 3] * this.m[2, 1] * this.m[3, 0]) -
                    (this.m[0, 3] * this.m[1, 0] * this.m[2, 1] * this.m[3, 2]) -
                    (this.m[0, 3] * this.m[1, 1] * this.m[2, 2] * this.m[3, 0]) -
                    (this.m[0, 3] * this.m[1, 2] * this.m[2, 0] * this.m[3, 1]);
            }
        }

        //-----------------------------------------------------------------
        // 行列の成分の取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M00
        {
            get
            {
                return this.m[0, 0];
            }

            set
            {
                this.m[0, 0] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M01
        {
            get
            {
                return this.m[0, 1];
            }

            set
            {
                this.m[0, 1] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M02
        {
            get
            {
                return this.m[0, 2];
            }

            set
            {
                this.m[0, 2] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M03
        {
            get
            {
                return this.m[0, 3];
            }

            set
            {
                this.m[0, 3] = value;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M10
        {
            get
            {
                return this.m[1, 0];
            }

            set
            {
                this.m[1, 0] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M11
        {
            get
            {
                return this.m[1, 1];
            }

            set
            {
                this.m[1, 1] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M12
        {
            get
            {
                return this.m[1, 2];
            }

            set
            {
                this.m[1, 2] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M13
        {
            get
            {
                return this.m[1, 3];
            }

            set
            {
                this.m[1, 3] = value;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M20
        {
            get
            {
                return this.m[2, 0];
            }

            set
            {
                this.m[2, 0] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M21
        {
            get
            {
                return this.m[2, 1];
            }

            set
            {
                this.m[2, 1] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M22
        {
            get
            {
                return this.m[2, 2];
            }

            set
            {
                this.m[2, 2] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M23
        {
            get
            {
                return this.m[2, 3];
            }

            set
            {
                this.m[2, 3] = value;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M30
        {
            get
            {
                return this.m[3, 0];
            }

            set
            {
                this.m[3, 0] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M31
        {
            get
            {
                return this.m[3, 1];
            }

            set
            {
                this.m[3, 1] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M32
        {
            get
            {
                return this.m[3, 2];
            }

            set
            {
                this.m[3, 2] = value;
            }
        }

        /// <summary>
        /// 行列の成分を取得または設定します。
        /// </summary>
        public float M33
        {
            get
            {
                return this.m[3, 3];
            }

            set
            {
                this.m[3, 3] = value;
            }
        }

        /// <summary>
        /// 行成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Rows
        {
            get
            {
                IVector[] data =
                {
                     new Vector4(this[0, 0], this[0, 1], this[0, 2], this[0, 3]),
                     new Vector4(this[1, 0], this[1, 1], this[1, 2], this[1, 3]),
                     new Vector4(this[2, 0], this[2, 1], this[2, 2], this[2, 3]),
                     new Vector4(this[3, 0], this[3, 1], this[3, 2], this[3, 3])
                };

                return data;
            }
        }

        /// <summary>
        /// 列成分のベクトル列を取得します。
        /// </summary>
        public IVector[] Columns
        {
            get
            {
                IVector[] data =
                {
                     new Vector4(this[0, 0], this[1, 0], this[2, 0], this[3, 0]),
                     new Vector4(this[0, 1], this[1, 1], this[2, 1], this[3, 1]),
                     new Vector4(this[0, 2], this[1, 2], this[2, 2], this[3, 2]),
                     new Vector4(this[0, 3], this[1, 3], this[2, 3], this[3, 3])
                };

                return data;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したインデックス位置の成分へのアクセスを行うインデクサです。
        /// </summary>
        /// <param name="row">アクセスする成分の行の 0 から始まるインデックスです。</param>
        /// <param name="column">アクセスする成分の列の 0 から始まるインデックスです。</param>
        /// <returns>指定したインデックス位置の成分です。</returns>
        public float this[int row, int column]
        {
            get
            {
                return this.m[row, column];
            }

            set
            {
                this.m[row, column] = value;
            }
        }

        //-----------------------------------------------------------------
        // オペレータ
        //-----------------------------------------------------------------

        /// <summary>
        /// 2 つの行列を加算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix44 operator +(Matrix44 lhs, Matrix44 rhs)
        {
            return (Matrix44)MatrixFunction<Matrix44>.Add(lhs, rhs);
        }

        /// <summary>
        /// 2 つの行列を減算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix44 operator -(Matrix44 lhs, Matrix44 rhs)
        {
            return (Matrix44)MatrixFunction<Matrix44>.Subtract(lhs, rhs);
        }

        /// <summary>
        /// 行列とスカラーを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix44 operator *(Matrix44 mtx, float scalar)
        {
            return (Matrix44)MatrixFunction<Matrix44>.Multiply(mtx, scalar);
        }

        /// <summary>
        /// 行列とスカラーを除算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix44 operator /(Matrix44 mtx, float scalar)
        {
            return (Matrix44)MatrixFunction<Matrix44>.Divide(mtx, scalar);
        }

        /// <summary>
        /// 2 つの行列を乗算します。
        /// </summary>
        /// <param name="lhs">演算の左の行列です。</param>
        /// <param name="rhs">演算の右の行列です。</param>
        /// <returns>演算結果です。</returns>
        public static Matrix44 operator *(Matrix44 lhs, Matrix44 rhs)
        {
            return (Matrix44)MatrixSquareFunction<Matrix44>.Multiply(lhs, rhs);
        }

        /// <summary>
        /// 行列にベクトルを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="vec">演算の右のベクトルです。</param>
        /// <returns>演算結果です。</returns>
        public static Vector3 operator *(Matrix44 mtx, Vector3 vec)
        {
            Ensure.Argument.NotNull(mtx);

            Vector3 result = new Vector3();
            result.X = (mtx[0, 0] * vec.X) + (mtx[0, 1] * vec.Y) + (mtx[0, 2] * vec.Z) + mtx[0, 3];
            result.Y = (mtx[1, 0] * vec.X) + (mtx[1, 1] * vec.Y) + (mtx[1, 2] * vec.Z) + mtx[1, 3];
            result.Z = (mtx[2, 0] * vec.X) + (mtx[2, 1] * vec.Y) + (mtx[2, 2] * vec.Z) + mtx[2, 3];
            return result;
        }

        /// <summary>
        /// 行列にベクトルを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="vec">演算の右のベクトルです。</param>
        /// <returns>演算結果です。</returns>
        public static Vector4 operator *(Matrix44 mtx, Vector4 vec)
        {
            Ensure.Argument.NotNull(mtx);

            Vector4 result = new Vector4();
            result.X = (mtx[0, 0] * vec.X) + (mtx[0, 1] * vec.Y) + (mtx[0, 2] * vec.Z) + (mtx[0, 3] * vec.W);
            result.Y = (mtx[1, 0] * vec.X) + (mtx[1, 1] * vec.Y) + (mtx[1, 2] * vec.Z) + (mtx[1, 3] * vec.W);
            result.Z = (mtx[2, 0] * vec.X) + (mtx[2, 1] * vec.Y) + (mtx[2, 2] * vec.Z) + (mtx[2, 3] * vec.W);
            result.W = (mtx[3, 0] * vec.X) + (mtx[3, 1] * vec.Y) + (mtx[3, 2] * vec.Z) + (mtx[3, 3] * vec.W);
            return result;
        }

        /// <summary>
        /// 指定した行列の逆行列を作成します。
        /// 逆行列に設定できなかった場合は例外を返します。
        /// </summary>
        /// <param name="mat">指定の行列です。</param>
        public static void Invert(Matrix44 mat)
        {
            Matrix44 result = InverseSolver(mat);

            Ensure.Operation.ObjectNotNull(result);

            mat.Set(result);
        }

        /// <summary>
        /// 逆行列に設定します。
        /// 逆行列が存在しない場合は指定した行列に設定します。
        /// </summary>
        /// <param name="mtx">元の行列です。</param>
        /// <param name="substitute">逆行列が存在しない場合に
        /// 代わりに設定する行列です。</param>
        /// <returns>逆行列が存在した場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public static bool SafeInvert(Matrix44 mtx, Matrix44 substitute)
        {
            Ensure.Argument.NotNull(substitute);

            Matrix44 result = InverseSolver(mtx);
            if (result == null)
            {
                if (mtx == substitute)
                {
                    return false;
                }

                mtx.Set(substitute);
                return false;
            }

            mtx.Set(result);
            return true;
        }

        /// <summary>
        /// 指定した行列の転置行列を取得します。
        /// </summary>
        /// <param name="mtx">指定の行列です。</param>
        /// <returns>転置行列です。</returns>
        public static Matrix44 Transpose(Matrix44 mtx)
        {
            Matrix44 result = new Matrix44(mtx);
            MatrixSquareFunction<Matrix44>.Transpose(result);
            return result;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new Matrix44(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as Matrix44);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix44 source)
        {
            MatrixFunction<Matrix44>.Set(source, this);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix33 source)
        {
            MatrixFunction<Matrix44>.Set<Matrix33>(source, this);
        }

        /// <summary>
        /// 行列を設定します。
        /// </summary>
        /// <param name="source">コピー元の行列です。</param>
        public void Set(Matrix34 source)
        {
            MatrixFunction<Matrix44>.Set<Matrix34>(source, this);
        }

        /// <summary>
        /// 行列の成分を指定の値に設定します。
        /// </summary>
        /// <param name="m00">[0, 0]成分です。</param>
        /// <param name="m01">[0, 1]成分です。</param>
        /// <param name="m02">[0, 2]成分です。</param>
        /// <param name="m03">[0, 3]成分です。</param>
        /// <param name="m10">[1, 0]成分です。</param>
        /// <param name="m11">[1, 1]成分です。</param>
        /// <param name="m12">[1, 2]成分です。</param>
        /// <param name="m13">[1, 3]成分です。</param>
        /// <param name="m20">[2, 0]成分です。</param>
        /// <param name="m21">[2, 1]成分です。</param>
        /// <param name="m22">[2, 2]成分です。</param>
        /// <param name="m23">[2, 3]成分です。</param>
        /// <param name="m30">[3, 0]成分です。</param>
        /// <param name="m31">[3, 1]成分です。</param>
        /// <param name="m32">[3, 2]成分です。</param>
        /// <param name="m33">[3, 3]成分です。</param>
        public void Set(
            float m00,
            float m01,
            float m02,
            float m03,
            float m10,
            float m11,
            float m12,
            float m13,
            float m20,
            float m21,
            float m22,
            float m23,
            float m30,
            float m31,
            float m32,
            float m33)
        {
            this[0, 0] = m00;
            this[0, 1] = m01;
            this[0, 2] = m02;
            this[0, 3] = m03;

            this[1, 0] = m10;
            this[1, 1] = m11;
            this[1, 2] = m12;
            this[1, 3] = m13;

            this[2, 0] = m20;
            this[2, 1] = m21;
            this[2, 2] = m22;
            this[2, 3] = m23;

            this[3, 0] = m30;
            this[3, 1] = m31;
            this[3, 2] = m32;
            this[3, 3] = m33;
        }

        /// <summary>
        /// 行列の成分を指定の値に設定します。
        /// </summary>
        /// <param name="row0">第０行の成分を格納したベクトルです。</param>
        /// <param name="row1">第１行の成分を格納したベクトルです。</param>
        /// <param name="row2">第２行の成分を格納したベクトルです。</param>
        /// <param name="row3">第３行の成分を格納したベクトルです。</param>
        public void Set(Vector4 row0, Vector4 row1, Vector4 row2, Vector4 row3)
        {
            this.M00 = row0.X;
            this.M01 = row0.Y;
            this.M02 = row0.Z;
            this.M03 = row0.W;
            this.M10 = row1.X;
            this.M11 = row1.Y;
            this.M12 = row1.Z;
            this.M13 = row1.W;
            this.M20 = row2.X;
            this.M21 = row2.Y;
            this.M22 = row2.Z;
            this.M23 = row2.W;
            this.M30 = row3.X;
            this.M31 = row3.Y;
            this.M32 = row3.Z;
            this.M33 = row3.W;
        }

        /// <summary>
        /// 行列の成分を全て 0 に設定します。
        /// </summary>
        public void SetZero()
        {
            MatrixFunction<Matrix44>.SetZero(this);
        }

        /// <summary>
        /// 単位行列に設定します。
        /// </summary>
        public void SetIdentity()
        {
            MatrixFunction<Matrix44>.SetIdentity(this);
        }

        //-----------------------------------------------------------------
        // 数学演算
        //-----------------------------------------------------------------

        /// <summary>
        /// 逆行列に設定します。
        /// 逆行列に設定できなかった場合は行列の成分を変更しません。
        /// </summary>
        public void Invert()
        {
            Invert(this);
        }

        /// <summary>
        /// 逆行列に設定します。
        /// 逆行列が存在しない場合は指定した行列に設定します。
        /// </summary>
        /// <param name="substitute">逆行列が存在しない場合に
        /// 代わりに設定する行列です。</param>
        /// <returns>逆行列が存在した場合は true を返します。
        /// それ以外の場合は false を返します。</returns>
        public bool SafeInvert(Matrix44 substitute)
        {
            return SafeInvert(this, substitute);
        }

        /// <summary>
        /// 行列のスケール・回転成分でベクトルを変換します。
        /// </summary>
        /// <param name="vec">ベクトルです。</param>
        /// <returns>演算結果です。</returns>
        public Vector3 MultiplyVectorScaleRotate(Vector3 vec)
        {
            Ensure.Argument.NotNull(vec);

            return new Vector3(
                (this[0, 0] * vec.X) + (this[0, 1] * vec.Y) + (this[0, 2] * vec.Z),
                (this[1, 0] * vec.X) + (this[1, 1] * vec.Y) + (this[1, 2] * vec.Z),
                (this[2, 0] * vec.X) + (this[2, 1] * vec.Y) + (this[2, 2] * vec.Z));
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 転置行列に設定します。
        /// </summary>
        public void Transpose()
        {
            MatrixSquareFunction<Matrix44>.Transpose(this);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// x 軸、y 軸、z 軸に沿ってスケーリングする行列に設定します。
        /// </summary>
        /// <param name="scale">スケールです。</param>
        public void SetScale(Vector3 scale)
        {
            this.SetScale(scale.X, scale.Y, scale.Z);
        }

        /// <summary>
        /// x 軸、y 軸、z 軸に沿ってスケーリングする行列に設定します。
        /// </summary>
        /// <param name="x">x 軸に沿ったスケールです。</param>
        /// <param name="y">y 軸に沿ったスケールです。</param>
        /// <param name="z">z 軸に沿ったスケールです。</param>
        public void SetScale(float x, float y, float z)
        {
            Matrix3DScaleRotateFunction<Matrix44>.SetScale(this, x, y, z);
        }

        /// <summary>
        /// 行列を x 軸、y 軸、z 軸に沿ってスケーリングします。
        /// </summary>
        /// <param name="scale">スケールです。</param>
        public void Scale(Vector3 scale)
        {
            Ensure.Argument.NotNull(scale);

            Matrix44 mtx = new Matrix44();
            mtx.SetScale(scale);
            this.Set(mtx * this);
        }

        /// <summary>
        /// 行列を x 軸、y 軸、z 軸に沿ってスケーリングします。
        /// </summary>
        /// <param name="x">x 軸に沿ったスケールです。</param>
        /// <param name="y">y 軸に沿ったスケールです。</param>
        /// <param name="z">z 軸に沿ったスケールです。</param>
        public void Scale(float x, float y, float z)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetScale(x, y, z);
            this.Set(mtx * this);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 任意の軸を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="axis">回転軸です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void SetRotate(Vector3 axis, float radian)
        {
            this.SetIdentity();
            Matrix3DScaleRotateFunction<Matrix44>.SetRotateAxis(this, axis, radian);
        }

        /// <summary>
        /// クォータニオンが表す回転行列に設定します。
        /// </summary>
        /// <param name="quat">クォータニオンです。</param>
        public void SetRotate(Quaternion quat)
        {
            Matrix3DScaleRotateFunction<Matrix44>.SetRotate(this, quat);
        }

        /// <summary>
        /// x 軸を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void SetRotateX(float radian)
        {
            Matrix3DScaleRotateFunction<Matrix44>.SetRotateX(this, radian);
        }

        /// <summary>
        /// y 軸を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void SetRotateY(float radian)
        {
            Matrix3DScaleRotateFunction<Matrix44>.SetRotateY(this, radian);
        }

        /// <summary>
        /// z 軸を中心に回転する行列に設定します。
        /// </summary>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void SetRotateZ(float radian)
        {
            Matrix3DScaleRotateFunction<Matrix44>.SetRotateZ(this, radian);
        }

        /// <summary>
        /// 行列を任意の軸を中心に回転します。
        /// </summary>
        /// <param name="axis">回転軸です。</param>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void Rotate(Vector3 axis, float radian)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetRotate(axis, radian);
            this.Set(mtx * this);
        }

        /// <summary>
        /// 行列をクォータニオンが表す回転で回転します。
        /// </summary>
        /// <param name="quat">クォータニオンです。</param>
        public void Rotate(Quaternion quat)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetRotate(quat);
            this.Set(mtx * this);
        }

        /// <summary>
        /// 行列を x 軸を中心に回転します。
        /// </summary>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void RotateX(float radian)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetRotateX(radian);
            this.Set(mtx * this);
        }

        /// <summary>
        /// 行列を y 軸を中心に回転します。
        /// </summary>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void RotateY(float radian)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetRotateY(radian);
            this.Set(mtx * this);
        }

        /// <summary>
        /// 行列を z 軸を中心に回転します。
        /// </summary>
        /// <param name="radian">回転角度 (ラジアン単位)です。</param>
        public void RotateZ(float radian)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetRotateZ(radian);
            this.Set(mtx * this);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// x 軸、y 軸、z 軸に沿って平行移動する行列に設定します。
        /// </summary>
        /// <param name="translate">移動量です。</param>
        public void SetTranslate(Vector3 translate)
        {
            Ensure.Argument.NotNull(translate);

            this.SetTranslate(translate.X, translate.Y, translate.Z);
        }

        /// <summary>
        /// x 軸、y 軸、z 軸に沿って平行移動する行列に設定します。
        /// </summary>
        /// <param name="x">x 軸に沿った移動量です。</param>
        /// <param name="y">y 軸に沿った移動量です。</param>
        /// <param name="z">z 軸に沿った移動量です。</param>
        public void SetTranslate(float x, float y, float z)
        {
            Matrix3DTranslateFunction<Matrix44>.SetTranslate(this, x, y, z);
        }

        /// <summary>
        /// 行列を x 軸、y 軸、z 軸に沿って平行移動します。
        /// </summary>
        /// <param name="translate">移動量です。</param>
        public void Translate(Vector3 translate)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetTranslate(translate);
            this.Set(mtx * this);
        }

        /// <summary>
        /// 行列を x 軸、y 軸、z 軸に沿って平行移動します。
        /// </summary>
        /// <param name="x">x 軸に沿った移動量です。</param>
        /// <param name="y">y 軸に沿った移動量です。</param>
        /// <param name="z">z 軸に沿った移動量です。</param>
        public void Translate(float x, float y, float z)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetTranslate(x, y, z);
            this.Set(mtx * this);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// Transform の表現する行列を設定します。
        /// </summary>
        /// <param name="trfm">拡大縮小、回転、平行移動です。</param>
        public void SetTransform(Transform trfm)
        {
            Matrix3DTransformFunction<Matrix44>.SetTransform(this, trfm);
        }

        /// <summary>
        /// Transform による変換を施します。
        /// </summary>
        /// <param name="trfm">拡大縮小、回転、平行移動です。</param>
        public void Transform(Transform trfm)
        {
            Matrix44 mtx = new Matrix44();
            mtx.SetTransform(trfm);
            this.Set(mtx * this);
        }

        /// <summary>
        /// 行列の表現するSRTを取得します。
        /// 行列が SRT で表現できるときのみ有効です。
        /// </summary>
        /// <param name="trfm">拡大縮小、回転、平行移動です。</param>
        public void GetTransform(Transform trfm)
        {
            Matrix3DTransformFunction<Matrix44>.GetTransform(this, trfm);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 要素を格納した配列を返します。
        /// </summary>
        /// <returns>要素の配列です。</returns>
        public float[] ToArray()
        {
            return new float[]
            {
                this.m[0, 0],
                this.m[0, 1],
                this.m[0, 2],
                this.m[0, 3],
                this.m[1, 0],
                this.m[1, 1],
                this.m[1, 2],
                this.m[1, 3],
                this.m[2, 0],
                this.m[2, 1],
                this.m[2, 2],
                this.m[2, 3],
                this.m[3, 0],
                this.m[3, 1],
                this.m[3, 2],
                this.m[3, 3]
            };
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Matrix44);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Matrix44 other)
        {
            return MatrixFunction<Matrix44>.Equals(this, other);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <param name="tolerance">各成分毎の差の最大許容値です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public bool Equals(Matrix44 other, float tolerance)
        {
            return MatrixFunction<Matrix44>.Equals(this, other, tolerance);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return MatrixFunction<Matrix44>.GetHashCode(this);
        }

        //-----------------------------------------------------------------
        // 文字列化
        //-----------------------------------------------------------------

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return MatrixFunction<Matrix44>.ToString(this);
        }

        private static Matrix44 InverseSolver(Matrix44 mtx)
        {
            // RevolutionSDK 準拠 mtx44.c C_MTX44Inverse
            Matrix44 source = new Matrix44(mtx);
            Matrix44 inverse = new Matrix44();

            for (int i = 0; i < 4; i++)
            {
                float max = 0.0f;
                int pivot = i;

                // ピボット選択
                for (int k = i; k < 4; k++)
                {
                    float ftmp = Math.Abs(source[k, i]);
                    if (ftmp > max)
                    {
                        max = ftmp;
                        pivot = k;
                    }
                }

                // 逆行列は存在しない
                if (max == 0f)
                {
                    return null;
                }

                // 行の交換
                if (pivot != i)
                {
                    SwapRow(source, i, pivot);
                    SwapRow(inverse, i, pivot);
                }

                float w = 1.0f / source[i, i];

                for (int j = 0; j < 4; j++)
                {
                    source[i, j] *= w;
                    inverse[i, j] *= w;
                }

                for (int k = 0; k < 4; k++)
                {
                    if (k == i)
                    {
                        continue;
                    }

                    w = source[k, i];
                    for (int j = 0; j < 4; j++)
                    {
                        source[k, j] -= source[i, j] * w;
                        inverse[k, j] -= inverse[i, j] * w;
                    }
                }
            }

            return inverse;
        }

        // 行の交換。
        private static void SwapRow(Matrix44 mtx, int r1, int r2)
        {
            if (r1 == r2)
            {
                return;
            }

            for (int i = 0; i < MatrixColumnCount; i++)
            {
                float temp = mtx[r1, i];
                mtx[r1, i] = mtx[r2, i];
                mtx[r2, i] = temp;
            }
        }
    }
}
