﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.ToolDevelopmentKit.Xml
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Xml;
    using System.Xml.Serialization;

    /// <summary>
    /// XML のユーティリティです。
    /// </summary>
    public static class XmlUtility
    {
        /// <summary>
        /// UTF8 の XML 宣言です。
        /// </summary>
        public const string Utf8Declaration = "<?xml version=\"1.0\" encoding=\"utf-8\"?>";

        /// <summary>
        /// スキーマインスタンスの名前です。
        /// </summary>
        public const string XmlSchemaInstanceName = "xsi";

        /// <summary>
        /// スキーマインスタンスの名前空間です。
        /// </summary>
        public const string XmlSchemaInstanceNamespace =
            "http://www.w3.org/2001/XMLSchema-instance";

        // HACK : 高速化のために初期化時の状態をキャシュします。アセンブリの遅延ロードには対応していません。
        private static XmlAttributeOverrides xmlAttributeOverrides = null;
        private static Type[] xmlTypes = null;
        private static List<XmlSerializer> serializers;
        private static List<Type> serializerTypes;


        /// <summary>
        /// XML をインデントします。
        /// </summary>
        /// <param name="source">インデント元となる XML 文字列です。</param>
        /// <returns>インデントした XML 文字列です。</returns>
        public static string Indent(string source)
        {
            XmlDocument xmlDocument = new XmlDocument();
            xmlDocument.LoadXml(source);

            XmlWriterSettings setting = new XmlWriterSettings();
            setting.OmitXmlDeclaration = true;
            setting.Indent = true;
            setting.IndentChars = "\t";

            StringBuilder builder = new StringBuilder();
            using (XmlWriter writer = XmlWriter.Create(builder, setting))
            {
                xmlDocument.Save(writer);
            }

            return builder.ToString();
        }

        //----------------------------------------------------------
        // ファイルシリアライズ補助
        //----------------------------------------------------------

        /// <summary>
        /// ファイルにシリアライズ出力します。
        /// </summary>
        /// <typeparam name="TType">シリアライズする型です。</typeparam>
        /// <param name="target">シリアライズする</param>
        /// <param name="filePath">書き出しファイルパスです。</param>
        public static void SerializeToFile<TType>(TType target, string filePath)
            where TType : class
        {
            XmlSerializer xmlSerializer = CreateCustomXmlSerializer<TType>();
            using (StreamWriter writer = new StreamWriter(filePath))
            {
                XmlSerializerNamespaces ns = new XmlSerializerNamespaces();
                ns.Add(string.Empty, string.Empty);

                xmlSerializer.Serialize(writer, target, ns);
            }
        }

        /// <summary>
        /// ファイルからデシリアライズします。
        /// </summary>
        /// <typeparam name="TType">シリアライズする型です。</typeparam>
        /// <param name="filePath">読み込みファイルパスです。</param>
        /// <returns>デシリアライズ結果です。</returns>
        public static TType DeserializeFromFile<TType>(string filePath)
            where TType : class
        {
            XmlSerializer xmlSerializer = CreateCustomXmlSerializer<TType>();
            using (StreamReader reader = new StreamReader(filePath))
            {
                return xmlSerializer.Deserialize(reader) as TType;
            }
        }

        /// <summary>
        /// ファイルにシリアライズ出力します。
        /// </summary>
        /// <typeparam name="TType">シリアライズする型です。</typeparam>
        /// <param name="target">シリアライズする</param>
        /// <param name="output">書き出しファイルパスです。</param>
        public static void SerializeToString<TType>(TType target, out string output)
            where TType : class
        {
            XmlSerializer xmlSerializer = CreateCustomXmlSerializer<TType>();

            StringBuilder sb = new StringBuilder();
            using (StringWriter writer = new StringWriter(sb))
            {
                XmlSerializerNamespaces ns = new XmlSerializerNamespaces();
                ns.Add(string.Empty, string.Empty);

                xmlSerializer.Serialize(writer, target, ns);
            }

            output = sb.ToString();
        }

        /// <summary>
        /// ファイルからデシリアライズします。
        /// </summary>
        /// <typeparam name="TType">シリアライズする型です。</typeparam>
        /// <param name="input">読み込みファイルパスです。</param>
        /// <returns>デシリアライズ結果です。</returns>
        public static TType DeserializeFromString<TType>(string input)
            where TType : class
        {
            XmlSerializer xmlSerializer = CreateCustomXmlSerializer<TType>();
            using (StringReader reader = new StringReader(input))
            {
                return xmlSerializer.Deserialize(reader) as TType;
            }
        }

        /// <summary>
        /// ストリームにシリアライズ出力します。
        /// </summary>
        /// <typeparam name="TType">シリアライズする型です。</typeparam>
        /// <param name="target">シリアライズする目標です。</param>
        /// <param name="output">書き出しストリームです。</param>
        public static void SerializeToStream<TType>( TType target,
                                                     ref Stream output ) where TType : class
        {
            XmlSerializer xmlSerializer = CreateCustomXmlSerializer<TType>();

            XmlSerializerNamespaces ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            xmlSerializer.Serialize( output, target, ns );
        }

        /// <summary>
        /// ストリームからデシリアライズします。
        /// </summary>
        /// <typeparam name="TType">シリアライズする型です。</typeparam>
        /// <param name="input">読み込み元です。</param>
        /// <returns>デシリアライズ結果です。</returns>
        public static TType DeserializeFromStream<TType>( Stream input ) where TType : class
        {
            XmlSerializer xmlSerializer = CreateCustomXmlSerializer<TType>();
            return xmlSerializer.Deserialize(input) as TType;
        }

        /// <summary>
        /// シリアライズ挙動を上書きするアトリビュートを適用したXmlシリアライザを生成します。
        /// </summary>
        /// <typeparam name="TValue">シリアライズする型です。</typeparam>
        /// <returns>Xmlシリアライザです。</returns>
        public static XmlSerializer CreateCustomXmlSerializer<TValue>() where TValue : class
        {
            if (xmlAttributeOverrides == null)
            {
                xmlAttributeOverrides =
                    XmlAttributeOverridesGenerator.GenerateXmlAttributeOverrides(
                    EnumerateXmlTypes());

                // Init pools
                serializers = new List<XmlSerializer>(10);
                serializerTypes = new List<Type>(10);
            }

            // Console.WriteLine("XML_SERIALIZER " + typeof(TValue).ToString());// + " : " + xmlAttributeOverrides.ToString());
            int i = 0;
            foreach (Type serializerType in serializerTypes)
            {
                if (serializerType == typeof(TValue))
                {
                    return serializers[i];
                }
                ++i;
            }

            Type          valueType  = typeof( TValue );
            XmlSerializer serializer = new XmlSerializer( valueType, xmlAttributeOverrides );

            serializerTypes.Add( valueType );
            serializers.Add( serializer );

            return serializer;
        }

        /// <summary>
        /// XMLシリアライズクラスの型を列挙します。
        /// </summary>
        /// <returns>XMLシリアライズクラスの型の列挙子を返します。</returns>
        public static IEnumerable<Type> EnumerateXmlTypes()
        {
            if (xmlTypes == null)
            {
                xmlTypes = TypeUtility.EnumrateTypes(type => IsXmlSerializeClass(type)).ToArray();
            }

            return xmlTypes;
        }

        /// <summary>
        /// XMLシリアライズ具象クラスの型を列挙します。
        /// </summary>
        /// <returns>XMLシリアライズ具象クラスの型の列挙子を返します。</returns>
        public static IEnumerable<Type> EnumerateXmlConcreteTypes()
        {
            return EnumerateXmlTypes().Where(type => !type.IsAbstract);
        }

        /// <summary>
        /// XML要素のタグ名を変更します。
        /// </summary>
        /// <param name="document">XmlDocumentです。</param>
        /// <param name="element">名前を変更する要素です。この要素への参照は無効になるので注意してください。</param>
        /// <param name="newName">新しいタグ名です。</param>
        /// <returns>新しいXML要素です。</returns>
        public static XmlElement Rename(XmlDocument document, XmlElement element, string newName)
        {
            var newElement = document.CreateElement(newName);

            // 属性のコピー
            foreach (XmlAttribute attribute in element.Attributes)
            {
                newElement.SetAttribute(attribute.Name, attribute.Value);
            }

            // 子供のコピー
            foreach (XmlElement child in element.ChildNodes)
            {
                newElement.AppendChild(child);
            }

            var parent = element.ParentNode;
            parent.RemoveChild(element);
            parent.AppendChild(newElement);

            return newElement;
        }

        /// <summary>
        /// 指定したXML要素の属性の名前を変更します。
        /// </summary>
        /// <param name="document">XmlDocumentです。</param>
        /// <param name="element">対象の属性を持つ要素です。</param>
        /// <param name="oldAttrName">変更する属性の名前です。この属性への参照は無効になるので注意してください。</param>
        /// <param name="newAttrName">変更後の名前です。</param>
        /// <returns>変更後の属性です。</returns>
        public static XmlAttribute RenameAttribute(
            XmlDocument document, XmlElement element, string oldAttrName, string newAttrName)
        {
            var oldAttr = element.Attributes[oldAttrName];
            Ensure.Operation.ObjectNotNull(oldAttr);

            var newAttr = document.CreateAttribute(newAttrName);
            element.Attributes.InsertAfter(newAttr, oldAttr);
            element.Attributes.Remove(oldAttr);

            return newAttr;
        }

        /// <summary>
        /// XMLシリアライズクラスかどうかを調べます。
        /// </summary>
        /// <param name="type">型を指定します。</param>
        /// <returns>XMLシリアライズクラスの場合は true、それ以外の場合は false を返します。</returns>
        private static bool IsXmlSerializeClass(Type type)
        {
            if (type == null)
            {
                return false;
            }

            // 高速化のため、比較的処理の軽い以下のプロパティをチェックします。
            if (type.IsInterface ||
                type.IsNotPublic)
            {
                return false;
            }

            return type.Name.EndsWith("Xml");
        }
    }
}
