﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.ToolDevelopmentKit.Xml.Complex
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Reflection;
    using System.Text;
    using System.Xml;
    using System.Xml.Serialization;

    /// <summary>
    /// XMLシリアライザでシリアライズ処理される、データを
    /// 表現するクラスです。
    /// </summary>
    /// <typeparam name="TSerializedType">シリアライズ処理されるデータ型です。</typeparam>
    public class XmlSerializedContent<TSerializedType>
        : XmlContent, IXmlSerializedContent<TSerializedType> where TSerializedType : class
    {
        /// <summary>
        /// 必要に応じて指定する、シリアライズに必要な追加型情報です。
        /// </summary>
        private readonly Type[] extraTypes;

        /// <summary>
        /// シリアライズ結果インスタンスです。
        /// </summary>
        private TSerializedType instance;

        /// <summary>
        /// シリアライズに使用するXmlSerializerです。
        /// </summary>
        private XmlSerializer xmlSerializer = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public XmlSerializedContent()
        {
            List<Type> types = GetExtraTypes(typeof(TSerializedType));
            this.extraTypes = types.ToArray();
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="xmlSerializer">シリアライズに使用するXmlSerializerです。</param>
        public XmlSerializedContent(XmlSerializer xmlSerializer)
        {
            this.xmlSerializer = xmlSerializer;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元のインスタンスです。</param>
        public XmlSerializedContent(XmlSerializedContent<TSerializedType> source)
            : this()
        {
            Ensure.Argument.NotNull(source);

            this.Set(source);
        }

        /// <summary>
        /// シリアライズ結果インスタンスを取得または設定します。
        /// </summary>
        object IXmlSerializedContent.SerializeInstance
        {
            get
            {
                return this.instance;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.instance = value as TSerializedType;
                OnPropertyChanged("SerializeInstance");
            }
        }

        /// <summary>
        /// シリアライズ結果インスタンスを取得または設定します。
        /// </summary>
        public TSerializedType SerializeInstance
        {
            get
            {
                return this.instance;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.instance = value;
                OnPropertyChanged("SerializeInstance");
            }
        }

        /// <summary>
        /// XML文章から実体を生成します。
        /// </summary>
        /// <param name="xmlString">XML文章です。</param>
        /// <param name="deserializer">デシリアライズに使用するXmlSerializerです。</param>
        /// <returns>実体です。</returns>
        public static IXmlContent FromXml(string xmlString, XmlSerializer deserializer)
        {
            Ensure.Argument.StringNotEmpty(xmlString);

            XmlSerializedContent<TSerializedType> result =
                new XmlSerializedContent<TSerializedType>();

            result.Xml = xmlString;
            result.DoDeserializeInstanceFromXml(deserializer);

            return result;
        }

        /// <summary>
        /// シリアライズ実体から実体を生成します。
        /// </summary>
        /// <param name="instance">シリアライズされる実体です。</param>
        /// <returns>実体です。</returns>
        public static IXmlContent FromInstance(TSerializedType instance)
        {
            Ensure.Argument.NotNull(instance);

            XmlSerializedContent<TSerializedType> result =
                new XmlSerializedContent<TSerializedType>();

            result.SerializeInstance = instance;
            result.DoSerializeInstanceFromInstance();

            return result;
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(XmlSerializedContent<TSerializedType> source)
        {
            Ensure.Argument.NotNull(source);

            base.Set(source as IXmlContent);
            this.SerializeInstance = source.SerializeInstance;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override object Clone()
        {
            return new XmlSerializedContent<TSerializedType>(this);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return CheckClassEqualityUtil<XmlSerializedContent<TSerializedType>>.Equals(
                this, other, this.Equals);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            int result = base.GetHashCode();
            foreach (Type extraType in this.extraTypes)
            {
                result ^= extraType.GetHashCode();
            }

            if (this.SerializeInstance != null)
            {
                result ^= this.SerializeInstance.GetHashCode();
            }

            return result;
        }

        /// <summary>
        /// XmlWriterに内容を書き出します。
        /// </summary>
        /// <param name="writer">XmlWriterです。</param>
        public override void Export(XmlWriter writer)
        {
            Ensure.Argument.NotNull(writer);

            XmlSerializerNamespaces ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            if (this.xmlSerializer == null)
            {
                XmlSerializer serializer = new XmlSerializer(typeof(TSerializedType), this.extraTypes);
                serializer.Serialize(writer, this.instance, ns);
            }
            else
            {
                this.xmlSerializer.Serialize(writer, this.instance, ns);
            }
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        protected bool Equals(XmlSerializedContent<TSerializedType> other)
        {
            if (!base.Equals(other))
            {
                return false;
            }

            // ExtraTypesについて比較します。
            if (this.extraTypes.Length != other.extraTypes.Length)
            {
                return false;
            }

            for (int i = 0; i < this.extraTypes.Length; i++)
            {
                if (this.extraTypes[i] != other.extraTypes[i])
                {
                    return false;
                }
            }

            // SerializeInstanceを比較します。
            if (this.SerializeInstance == null)
            {
                return other.SerializeInstance != null;
            }

            if (!this.SerializeInstance.Equals(other.SerializeInstance))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 型が含む型セットを取得します。
        /// </summary>
        /// <param name="targetType">調査する型です。</param>
        /// <returns>結果を格納する型セットです。</returns>
        private static List<Type> GetExtraTypes(Type targetType)
        {
            Ensure.Argument.NotNull(targetType);

            // Type型なら、処理せずに空を返します。
            if (targetType == typeof(Type))
            {
                return new List<Type>();
            }

            List<Type> resultSet = new List<Type>();

            // シリアライズ対象の型フィールドすべてについて...
            BindingFlags bindingFlags = BindingFlags.Public | BindingFlags.Instance;
            foreach (PropertyInfo srcPropInfo in targetType.GetProperties(bindingFlags))
            {
                Type propertyType = srcPropInfo.PropertyType;

                if (propertyType == targetType)
                {
                    continue;
                }

                if (!propertyType.IsSerializable)
                {
                    continue;
                }

                if (propertyType.IsPrimitive || (propertyType == typeof(string)))
                {
                    continue;
                }

                if (propertyType.IsArray)
                {
                    continue;
                }

                if (propertyType.IsGenericType &&
                    (propertyType.GetGenericTypeDefinition() == typeof(List<>)))
                {
                    continue;
                }

                if (resultSet.Contains(propertyType))
                {
                    continue;
                }

                resultSet.Add(srcPropInfo.PropertyType);

                // 再帰的に調査します。
                resultSet.AddRange(GetExtraTypes(propertyType));
            }

            return resultSet;
        }

        /// <summary>
        /// デシリアライズして実体を生成します。
        /// </summary>
        /// <param name="deserializer">デシリアライズに使用するXmlSerializerです。</param>
        private void DoDeserializeInstanceFromXml(XmlSerializer deserializer)
        {
            Ensure.Operation.True(!string.IsNullOrEmpty(this.Xml));

            // デシリアライズして実体を生成します。
            // 既定の動作では、List<XXX>のルート名が、ArrayOfXXXとなります。
            // 扱いにくいことが多い場合には、XmlSerializerにXmlRootAttribute("ルート名")を指定して、
            // ルート要素名を明示的に指定するべきかもしれません。
            if (deserializer == null)
            {
                deserializer = new XmlSerializer(typeof(TSerializedType), this.extraTypes);
            }

            using (XmlTextReader reader = new XmlTextReader(new StringReader(this.Xml)))
            {
                // シリアライズするにあたって空白を無視するように設定します。
                reader.WhitespaceHandling = WhitespaceHandling.None;
                this.SerializeInstance = (TSerializedType)deserializer.Deserialize(reader);
            }
        }

        /// <summary>
        /// シリアライズしてXMLを生成します。
        /// </summary>
        private void DoSerializeInstanceFromInstance()
        {
            Ensure.Operation.ObjectNotNull(this.instance);

            StringBuilder sb = new System.Text.StringBuilder();
            XmlWriterSettings setting = new XmlWriterSettings();
            setting.OmitXmlDeclaration = true;
            XmlWriter writer = XmlTextWriter.Create(sb, setting);

            this.Export(writer);

            this.Xml = sb.ToString();
        }
    }
}
