﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit.Logs
{
    using System;

    /// <summary>
    /// ロガーです。
    /// </summary>
    public class Logger : ILogger
    {
        //-----------------------------------------------------------------
        private ILogInput[] inputs;
        private LogListener[] listeners;

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public Logger()
        {
            int count = Enum.GetValues(typeof(LogLevel)).Length;
            this.inputs = new ILogInput[count];

            this.listeners = new LogListener[count];
            for (int i = 0; i < count; i++)
            {
                LogLevel logLevel = (LogLevel)i;
                this.listeners[i] = delegate(ILog log)
                {
                    if (this.AllLogListener != null)
                    {
                        this.AllLogListener(log, logLevel);
                    }
                };
            }

            this.Setup();

            for (int i = 0; i < count; i++)
            {
                Ensure.Operation.True(this[(LogLevel)i] != null);
            }
        }

        //-----------------------------------------------------------------
        // ログのリスナ
        //-----------------------------------------------------------------

        /// <summary>
        /// 全てのログのリスナイベントです。
        /// </summary>
        public event AllLogListener AllLogListener;

        //-----------------------------------------------------------------
        // プロパティの取得または設定
        //-----------------------------------------------------------------

        //-----------------------------------------------------------------

        /// <summary>
        /// 致命的な障害ログのログ入力です。
        /// </summary>
        public ILogInput Fatal
        {
            get { return this[LogLevel.Fatal]; }
            protected set { this.SetLogInput(value, LogLevel.Fatal); }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// エラーログのログ入力です。
        /// </summary>
        public ILogInput Error
        {
            get { return this[LogLevel.Error]; }
            protected set { this.SetLogInput(value, LogLevel.Error); }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 警告ログのログ入力です。
        /// </summary>
        public ILogInput Warn
        {
            get { return this[LogLevel.Warn]; }
            protected set { this.SetLogInput(value, LogLevel.Warn); }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 情報ログのログ入力です。
        /// </summary>
        public ILogInput Info
        {
            get { return this[LogLevel.Info]; }
            protected set { this.SetLogInput(value, LogLevel.Info); }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// デバッグ用ログのログ入力です。
        /// </summary>
        public ILogInput Debug
        {
            get { return this[LogLevel.Debug]; }
            protected set { this.SetLogInput(value, LogLevel.Debug); }
        }

        /// <summary>
        /// インデクサによりログ入力を取得します。
        /// </summary>
        /// <param name="logLevel">取得するログ入力のログレベルです。</param>
        /// <returns>指定したログレベルのログ入力です。</returns>
        public ILogInput this[LogLevel logLevel]
        {
            get { return this.inputs[(int)logLevel]; }
        }

        /// <summary>
        /// ログ入力をセットアップします。
        /// </summary>
        protected virtual void Setup()
        {
            this.Fatal = new LogInput();
            this.Error = new LogInput();
            this.Warn = new LogInput();
            this.Info = new LogInput();
#if DEBUG
            this.Debug = new LogInput();
#else
            this.Debug = new NullLogInput();
#endif
        }

        // ログ入力の設定
        private void SetLogInput(ILogInput input, LogLevel logLevel)
        {
            Ensure.Argument.NotNull(input);

            int index = (int)logLevel;
            ILogInput swap = this.inputs[index];
            if (swap != null)
            {
                swap.LogListener -= this.listeners[index];
            }

            this.inputs[index] = input;
            input.LogListener += this.listeners[index];
        }
    }
}
