﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

// Comment out this line if you want to disable the ping messages.
#define ENABLE_VIEWER_PING_MSG

// Uncomment the line if you want to dump the receive network message to log.
//#define DUMP_RECEIVED_MESSAGE

// Uncomment the line if you want to dump the outgoing network message to log.
//#define DUMP_OUTGOING_MESSAGE

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Windows.Forms;

using NintendoWare.ToolDevelopmentKit.Logs;

using NWCore.Protocols;

namespace NWCore.Viewer
{
    /// <summary>
    /// メッセージマネージャ
    /// </summary>
    public sealed class MessageManager : IDisposable
    {
        #region Debug variables

        #if DUMP_RECEIVED_MESSAGE_TO_FILE

        private byte[] m_debugMsgBuffer    = null;
        private int    m_iDebugBufferIndex = 0;
        private string m_msgDumpFilePath   = "msg_dump_file.dat";

        #endif

        #endregion

        #region Constructors

        /// <summary>
        /// Internal constructor.
        /// </summary>
        internal MessageManager()
        {
            // Create default connector
            m_connecters.Add(new TcpViewerConnecter(this));
            ////m_connecters.Add( new HostIOConnecter( this ) );
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize
        /// </summary>
        public void Initialize( string[] winViewerNames )
        {
            m_syncContext = SynchronizationContext.Current;

            // Initialize connectors.
            foreach ( IConnecter connecter in m_connecters )
            {
                if ( connecter==null )
                    continue;

                connecter.Initialize();
            }

            // Prepare the thread for communication
            m_thread = new Thread( new ThreadStart( ThreadProcess ) );

            m_thread.IsBackground     = true;
            m_thread.Priority         = ThreadPriority.AboveNormal;
            m_thread.CurrentUICulture = Thread.CurrentThread.CurrentUICulture;
            m_thread.Name             = "通信スレッド";

            m_thread.Start();
        }

        #endregion

        #region Dispose

        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            m_bFinalizing = true;

            // Terminate the communication thread
            if ( m_thread!=null )
            {
                m_thread.Join( TIMEOUT_MS );
                if ( m_thread.IsAlive==true )
                    m_thread.Abort();

                m_thread = null;
            }

            // Destroy connectors.
            foreach ( IConnecter connecter in m_connecters )
            {
                if ( connecter==null )
                    continue;

                connecter.Close();
                connecter.Dispose();
            }

            m_connecters.Clear();
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the string encoding.
        /// </summary>
        public Encoding EncodingType
        {
            get
            {
                // Return default encoding of the active connection context.
                ConnectionContext context = GetActiveConnectionContext();
                if ( context!=null )
                    return context.Encoding;

                return Encoding.UTF8;
            }

            set
            {
                ConnectionContext context = GetActiveConnectionContext();
                if ( context==null )
                    return;

                context.Encoding = value.Clone() as Encoding;
            }
        }


        /// <summary>
        /// Get or set the logger.
        /// </summary>
        public ILogger Logger
        {
            get;
            set;
        }


        /// <summary>
        /// Get the binary file extension for the active connection.
        /// </summary>
        public string BinaryFileExt
        {
            get
            {
                ConnectionContext context = GetActiveConnectionContext();
                if ( context==null )
                    return string.Empty;

                return context.BinaryFileExt;
            }
        }


        /// <summary>
        /// Get the flag indicating if connected to any viewer.
        /// </summary>
        public bool IsConnected
        {
            get
            {
                if ( m_activeConnecter==null )
                    return false;

                return m_activeConnecter.IsConnected;
            }
        }


        /// <summary>
        /// Get the flag indicating whether auto connect is enabled.
        /// </summary>
        public bool IsAutoConnectEnabled
        {
            get { return m_bAutoConnect; }
        }


        /// <summary>
        /// Get or set the flag indicating whether this is enabled or not.
        /// </summary>
        public bool IsEnabled
        {
            get { return m_bEnabled; }
            set { m_bEnabled = value; }
        }


        /// <summary>
        /// Check if there are any queued messages.
        /// </summary>
        public bool HasAnyMessage
        {
            get
            {
                return m_iMessageCount>0;
            }
        }


        /// <summary>
        /// Get the number of queued messages.
        /// </summary>
        public int MessageCount
        {
            get { return m_iMessageCount; }
        }


        /// <summary>
        /// Get the flag indicating the platform of the active connection.
        /// </summary>
        public PlatformType Platform
        {
            get
            {
                ConnectionContext context = GetActiveConnectionContext();
                if ( context==null )
                    return PlatformType.Windows;

                return context.Platform;
            }
        }


        /// <summary>
        /// Get the flag indicating if the connection uses big endian.
        /// </summary>
        public bool IsBigEndian
        {
            get
            {
                ConnectionContext context = GetActiveConnectionContext();
                if ( context==null )
                    return false;

                return ( ((int)context.Platform & (int)PlatformType.EndianMask)!=0 );
            }
        }


        /// <summary>
        /// Get the flag indicating if the connected target uses native formats.
        /// </summary>
        public bool UseNativeFormat
        {
            get
            {
                ConnectionContext context = GetActiveConnectionContext();
                if ( context==null )
                    return false;

                return (((int)context.Platform & (int)PlatformType.NativeFormatMask)!=0);
            }
        }

        #endregion

        #region Connecter related

        /// <summary>
        /// Get connecter by its identifier.
        /// </summary>
        /// <param name="identifier">The identifier.</param>
        /// <returns>
        /// The connecter or null if no connecter with the given identifier exists.
        /// </returns>
        public IConnecter GetConnecterByIdentifier( string identifier )
        {
            foreach ( IConnecter connecter in m_connecters )
            {
                if ( connecter.Identifier==identifier )
                    return connecter;
            }

            return null;
        }


        /// <summary>
        /// Get the active connecter.
        /// </summary>
        /// <returns>The active connecter.</returns>
        public IConnecter GetActiveConnecter()
        {
            return m_activeConnecter;
        }


        /// <summary>
        /// Get the connection context from the active connecter.
        /// </summary>
        /// <returns></returns>
        public ConnectionContext GetActiveConnectionContext()
        {
            if ( m_activeConnecter==null )
                return null;

            return m_activeConnecter.GetConnectionContext();
        }


        /// <summary>
        /// Set the active connection context.
        /// </summary>
        /// <param name="context">The connection context.</param>
        public void SetActiveConnectionContext( ConnectionContext context )
        {
            bool bConnected = false;
            if ( this.IsConnected==true )
            {
                // If the connection context is different from the current one,
                // close the connection first.
                ConnectionContext currContect = GetActiveConnectionContext();
                if ( context.Identifier!=currContect.Identifier )
                    this.Close();

                bConnected = true;
            }

            // Connect with the new connection context.
            if ( this.IsAutoConnectEnabled==true )
                AutoConnect( true, context );
            else if ( bConnected==true )
                Connect( context );
        }

        #endregion

        #region Frame processor

        /// <summary>
        /// ルーティン処理（毎フレーム呼び出し必要があります）
        /// もし呼び出さないときは、ビューア終了で自動切断されない
        /// </summary>
        public void FrameProcess(ref Message m)
        {
            if ( this.IsConnected==false )
                return;

            if ( m_bRequestCloseConn==true &&
                 m_bEnableCallBack==true )
            {
                m_bEnableCallBack = false;
                Close();
                m_bEnableCallBack = true;
            }
            else if ( m.Msg==NWCore.Win32.WM.WM_COPYDATA )
            {
                bool bResult = true;

                // Is the message for this process?
                var process = Process.GetCurrentProcess();
                if ( m.HWnd==process.MainWindowHandle )
                {
                    // Retrieve the message data
                    Win32.COPYDATASTRUCT copyData =
                        (Win32.COPYDATASTRUCT)Marshal.PtrToStructure( m.LParam,
                                                                      typeof(Win32.COPYDATASTRUCT) );
                    // Is data size greater than zero?
                    if ( copyData.cbData>0 )
                    {
                        try
                        {
                            // Get the global data buffer
                            byte[] buffer = new byte[copyData.cbData + Header.StructSize];
                            Marshal.Copy( copyData.lpData,
                                          buffer,
                                          0,
                                          copyData.cbData );

                            // Push the data to connector for later access.
                            TcpViewerConnecter connector = m_activeConnecter as TcpViewerConnecter;
                            if ( connector!=null )
                                connector.WinMsgPushReceivedData(buffer);
                        }
                        catch ( Exception ex )
                        {
                            bResult = false;

                            // Failed acquiring global buffer.
                            string output = String.Format( "  Viewer -> EM4F: Failed acquiring message data." );
                            DebugConsole.DebugConsole.Instance.WriteLine(ex.Message);
                        }
                    }

                    // Log received message
                    if ( bResult==true )
                    {
                        string output = String.Format( "  Viewer -> EM4F: {0}, size: {1}",
                                                       copyData.dwData.ToInt32(),
                                                       copyData.cbData.ToString() );
                        DebugConsole.DebugConsole.Instance.WriteLine( output );
                    }
                }
            }
        }

        #endregion

        #region Stream reader/writer

        /// <summary>
        /// Create stream writer with the platform and encoding settings
        /// of the active connection.
        /// </summary>
        public Protocol4FWriter CreateWriter( Stream stream )
        {
            if ( stream==null )
                return null;

            ConnectionContext context = GetActiveConnectionContext();
            if ( context==null )
                return null;

            if ( context.Platform==PlatformType.Windows )
            {
                return new LittleEndianProtocolWriter( stream, context.Encoding );
            }
            else if ( context.Platform==PlatformType.Cafe )
            {
                return new BigEndianProtocolWriter( stream, context.Encoding );
            }
            else if ( context.Platform==PlatformType.CTR )
            {
                return new LittleEndianProtocolWriter( stream, context.Encoding );
            }

            return null;
        }


        /// <summary>
        /// Create stream reader with the platform and encoding settings
        /// of the active connection.
        /// </summary>
        public Protocol4FReader CreateReader( Stream stream )
        {
            if ( stream==null )
                return null;

            ConnectionContext context = GetActiveConnectionContext();
            if ( context==null )
                return null;

            if ( context.Platform==PlatformType.Windows )
            {
                return new LittleEndianProtocolReader( stream, context.Encoding );
            }
            else if ( context.Platform==PlatformType.Cafe )
            {
                return new BigEndianProtocolReader( stream, context.Encoding );
            }
            else if ( context.Platform==PlatformType.CTR )
            {
                return new LittleEndianProtocolReader( stream, context.Encoding );
            }

            return null;
        }

        #endregion

        #region Connection

        /// <summary>
        /// Check if the target of the given connection exists.
        /// </summary>
        public bool CheckTargetExistence( ConnectionContext context )
        {
            if ( context==null || context.IsReady==false )
                return false;

            IConnecter connecter = GetConnecterByIdentifier( context.ConnecterIdentifier );
            if ( connecter==null )
                return false;

            return connecter.CheckTargetExistence( context );
        }


        /// <summary>
        /// Check if the connection target of the given context is same as
        /// that of the active context.
        /// </summary>
        /// <param name="context">The connection context to check.</param>
        /// <returns>True if the targets are the same.</returns>
        public bool IsSameConnectionTarget( ConnectionContext context )
        {
            ConnectionContext activeContext = GetActiveConnectionContext();
            if ( activeContext==null )
                return false;

            return activeContext.IsSameTarget( context );
        }


        /// <summary>
        /// Request disconnect from the active connection.
        /// </summary>
        internal void RequestDisconnect()
        {
            m_bRequestCloseConn = true;
        }


        /// <summary>
        /// Schedule auto connect with the given connection context.
        /// </summary>
        /// <param name="bEnable">Enable / disable auto connect.</param>
        /// <param name="context">The connection context.</param>
        public void AutoConnect( bool bEnable,
                                 ConnectionContext context = null )
        {
            if ( bEnable==false )
            {
                m_bAutoConnect    = false;
                m_autoConnContext = null;
            }
            else
            {
                if ( context==null || context.IsReady==false )
                    return;

                // Already connected?
                if ( this.IsConnected==true )
                {
                    // Same connection target?
                    if ( IsSameConnectionTarget( context )==true )
                        return;
                    else
                        Close();
                }

                m_bAutoConnect    = true;
                m_autoConnContext = context;
            }
        }


        /// <summary>
        /// Connect to the target with the given connection context.
        /// </summary>
        public bool Connect( ConnectionContext context )
        {
            if ( context==null || context.IsReady==false )
                return false;

            IConnecter connecter = GetConnecterByIdentifier( context.ConnecterIdentifier );
            if ( connecter==null )
                return false;

            // Set the active connector.
            m_activeConnecter = connecter;

            if ( m_bConnecting==true )
                return true;

            m_bConnecting = true;

            // Connect
            if ( connecter.Connect( context )==false )
            {
                m_bConnecting = false;
                return false;
            }

            // Trigger connection changed event
            TriggerConnectionChangedEvent( true );

            // Log
            DebugConsole.DebugConsole.Instance.WriteLine( "> Connected to viewer : " + context.Name );
            if ( this.Logger!=null )
            {
                string output = String.Format( "{0} \"{1}\"",
                                               res.Strings.INFO_CONNECTED,
                                               context.Name );
                this.Logger.Info.AddMessage( output );
            }

            m_bConnected  = true;
            m_bConnecting = false;

            return true;
        }


        /// <summary>
        /// Disconnect from the viewer.
        /// </summary>
        public void Close()
        {
            if ( m_bClosing==true )
                return;

            m_bClosing = true;

            // Clear the queue anyway.
            m_messageQueue.PopMessages();
            m_iMessageCount = 0;

            ConnectionContext context  = GetActiveConnectionContext();
            string            connName = "Unknown";
            if ( context!=null )
                connName = context.Name;

            bool bConnected = this.IsConnected;

            // Disconnect the active connector.
            if ( m_activeConnecter!=null )
            {
                m_activeConnecter.Close();
                m_activeConnecter = null;
            }

            if ( bConnected==true )
            {
                // Log
                DebugConsole.DebugConsole.Instance.WriteLine( "> Disconnected from viewer : " +
                                                              connName );
                if ( this.Logger!=null )
                    this.Logger.Info.AddMessage( res.Strings.INFO_DISCONNECTED );

                // Trigger connection changed event.
                TriggerConnectionChangedEvent( false );
            }

            m_bRequestCloseConn = false;
            m_bClosing          = false;
            m_bConnected        = false;
        }

        #endregion

        #region Access message queue

        /// <summary>
        /// メッセージを登録します。
        /// </summary>
        public void PushMessage( BaseMessage message )
        {
            if ( this.IsConnected==false || m_thread==null )
                return;

            lock ( m_messageQueue )
            {
                // 新規に追加
                m_messageQueue.PushMessage( message );
                m_iMessageCount = m_messageQueue.Count;
            }
        }


        /// <summary>
        /// メッセージを取り出します
        /// </summary>
        private BaseMessage PopMessage()
        {
            lock (this.m_messageQueue)
            {
                // We don't update m_iMessageCount now.
                // It's only updated after the process loop.
                return m_messageQueue.PopMessage();
            }
        }

        /// <summary>
        /// メッセージを取り出します
        /// </summary>
        private BaseMessage[] PopMessages()
        {
            lock ( m_messageQueue )
            {
                // We don't update m_iMessageCount now.
                // It's only updated after the process loop.
                return m_messageQueue.PopMessages();
            }
        }

        /// <summary>
        /// Set message processor. The incoming will be handed to the processor
        /// for further process, eg. Dispatch message received events, filter
        /// messages...
        /// </summary>
        /// <param name="processor">The message processor.</param>
        public void SetMessageProcessor(IMessageProcessor processor)
        {
            m_processor = processor;
        }

        #endregion

        #region Communication thread process

        /// <summary>
        /// Process for the communication thread.
        /// </summary>
        private void ThreadProcess()
        {
            try
            {
                while ( m_bFinalizing==false )
                {
                    ExecuteThreadLoop();

                    // Yield CPU for other threads
                    if ( this.IsConnected==false && m_bAutoConnect==true )
                        Thread.Sleep( AUTO_CONNECT_TIME );
                    else
                    {
                        if ( m_activeConnecter==null ||
                             m_activeConnecter.IsIncomingBufferFull==false )
                        {
                            Thread.Sleep( SLEEP_TIME );
                        }
                    }
                }

                ExecuteThreadLoop();
            }
            catch ( ThreadAbortException )
            {
                // Do nothing on thread abortion
                return;
            }
            catch ( Exception exception )
            {
                if ( m_bFinalizing==false )
                {
                    string output = res.Strings.VIEWER_EXCEPTION_MSG + "\n\n" + exception.Message;

                    #if DEBUG
                        output += "\n" + exception.StackTrace;
                    #endif

                    DebugConsole.DebugConsole.Instance.WriteLine( output );

                    if ( this.Logger!=null )
                    {
                        this.Logger.Fatal.AddMessage( output );
                        this.Logger.Fatal.AddMessage( res.Strings.ERROR_COMMUNICATE );
                    }
                }

                RequestDisconnect();
            }
        }


        /// <summary>
        /// Execute the tasks for one thread loop.
        /// </summary>
        private void ExecuteThreadLoop()
        {
            // Already connected to the viewer?
            if ( this.IsConnected==false )
            {
                if (m_bConnecting)
                {
                    return;
                }

                // Connection lost?
                if ( m_bConnected==true )
                {
                    Close();

                    // Log
                    DebugConsole.DebugConsole.Instance.WriteLine( "> Connection lost" );
                    if ( this.Logger!=null )
                        this.Logger.Info.AddMessage( res.Strings.INFO_DISCONNECTED );

                    // Trigger connection changed event.
                    TriggerConnectionChangedEvent( false );
                }

                if ( m_bAutoConnect==false ||
                     m_bClosing==true )
                {
                    return;
                }

                if ( Connect( m_autoConnContext )==false )
                    return;
            }
            else if ( m_bClosing==false )
            {
                // Read driver buffer for new incoming messages
                ReadStream();

                // Send out the contents of the stream
                SendStream();

                // Ping the viewer to make sure it is still there and vice versa.
                PingViewer();

                // The message count does not being cleared until now.
                // This is to prevent the MCSTransmissionDialog to
                // exit too early.
                // In SendStream(), if the messages are popped out but
                // haven't been sent out, Close() might crash the message
                // manager.
                m_iMessageCount = m_messageQueue.Count;
            }
        }


        /// <summary>
        /// Send out the output stream through the active connector.
        /// </summary>
        private void SendStream()
        {
            if ( this.IsConnected==false )
                return;

            // Skip execution if this is disabled or the message queue
            if ( this.HasAnyMessage==false || this.IsEnabled==false )
                return;

            int i;
            BaseMessage[] messages = PopMessages();
            BaseMessage   msg;

            // Create the stream
            MemoryStream stream = new MemoryStream();

            Protocol4FWriter writer = CreateWriter( stream );

            // Binarize the messages and fill them into the stream.
            for ( i=0;i<messages.Length;++i )
            {
                msg = messages[i];
                if ( msg!=null )
                {
                    msg.Write( writer );
                }
            }

            long currPos = stream.Position;
            if ( currPos<=0 )
                return;


                {
                    #region Output the binary stream for debugging

                    #if DUMP_OUTGOING_MESSAGE

                    Console.WriteLine( "-------------------- Begin outgoing NetStream --------------------" );
                    Console.WriteLine( "Buffer Size : " + stream.Length );
                    int    iNumBytes = 0;
                    string output    = string.Empty;
                    foreach ( byte data in stream.GetBuffer() )
                    {
                        output += data.ToString( "X2" ) + " ";
                        if ( iNumBytes % 16 == 15 )
                        {
                            Console.WriteLine( "{0} | 0x{1:X4}({1})", output, iNumBytes );
                            output = string.Empty;
                        }
                        ++iNumBytes;
                        if ( iNumBytes>=stream.Length )
                            break;
                    }
                    if ( output.Length>0 )
                        Console.WriteLine( "{0} | 0x{1:X4}({1})", output, (iNumBytes-1) );
                    Console.WriteLine( "--------------------  End outgoing NetStream  --------------------" );

                    #endif

                    #endregion
                }

            // Write the messages to console for debug
            TraceWriteStream( stream );

            stream.Position = currPos;

            // Append the ending packet
            {
                var header = new Header( MessageType.File_EndOfPacket,
                                         MESSAGEPACKET_File_EndOfPacket.StructSize,
                                         0 );
                var end = new MESSAGEPACKET_File_EndOfPacket();

                header.Write( writer );
                end.Write( writer );
            }

            // Reset the stream position.
            stream.Position = 0;

            bool bResult = m_activeConnecter.Send( stream.GetBuffer(),
                                                   (int)stream.Length );

            if ( bResult==false)
            {
                DebugConsole.DebugConsole.Instance.WriteLine("SendStream : *** error ***");
            }

            // Clean up
            stream.Close();
        }


        /// <summary>
        /// Read the received messages from our stream.
        /// </summary>
        private void ReadStream()
        {
            if ( this.IsConnected==false )
                return;

            try
            {
                #region Read stream into our buffer

                // The size of the buffer left over from previous read.
                int iLeftOverSize = 0;
                if ( m_leftOverBuffer!=null )
                    iLeftOverSize = m_leftOverBuffer.Length;

                // The buffer just received
                byte[] buffer;
                int iBufferSize = m_activeConnecter.Receive( out buffer );
                if ( iBufferSize<=0 || buffer==null || buffer.Length<=0 )
                    iBufferSize = 0;

                byte[] combinedBuffer;
                if ( iLeftOverSize<=0 && iBufferSize>0 )
                {
                    // Nothing left over on the previous read.
                    combinedBuffer = buffer;
                }
                else if ( iLeftOverSize>0 && iBufferSize>0 )
                {
                    // There is left over data and also received something this time.
                    int iIndex = 0;

                    combinedBuffer = new byte[iLeftOverSize + iBufferSize];

                    foreach ( byte value in m_leftOverBuffer )
                        combinedBuffer[iIndex++] = value;

                    foreach ( byte value in buffer )
                        combinedBuffer[iIndex++] = value;
                }
                else if ( iLeftOverSize>0 && iBufferSize<=0 )
                {
                    // Nothing received this time, just process the left overs.
                    combinedBuffer = m_leftOverBuffer;
                }
                else
                {
                    // Nothing unprocessed.
                    return;
                }

                {
                    #region Output the combined binary stream for debugging

                    #if DUMP_RECEIVED_MESSAGE

                    // This outputs the messages with the left over buffers,
                    // so every dump of the message would be combined with the
                    // part that has not been processed in the previous receiving.

                    Console.WriteLine( "-------------------- Begin received NetStream --------------------" );
                    Console.WriteLine( "Buffer Size : " + combinedBuffer.Length );
                    int    iNumBytes = 0;
                    string output    = string.Empty;
                    foreach ( byte data in combinedBuffer )
                    {
                        output += data.ToString( "X2" ) + " ";
                        if ( iNumBytes % 16 == 15 )
                        {
                            Console.WriteLine( output );// + " <= " + iNumBytes );
                            output = string.Empty;
                        }
                        ++iNumBytes;
                    }
                    if ( output.Length>0 )
                        Console.WriteLine( output );// + " <= " + (iNumBytes - 1) );
                    Console.WriteLine( "--------------------  End received NetStream  --------------------" );

                    #endif

                    #if DUMP_RECEIVED_MESSAGE_TO_FILE

                    // Create the buffer
                    if ( m_debugMsgBuffer==null )
                    {
                        m_iDebugBufferIndex = 0;
                        m_debugMsgBuffer    = new byte[8192];
                    }

                    // Copy the received message to buffer
                    if ( (m_iDebugBufferIndex+buffer.Length)<m_debugMsgBuffer.Length )
                    {
                        Array.Copy( buffer, 0,
                                    m_debugMsgBuffer, m_iDebugBufferIndex,
                                    buffer.Length );
                        m_iDebugBufferIndex += buffer.Length;
                    }
                    else
                    {
                        int iCopyLength = m_debugMsgBuffer.Length - m_iDebugBufferIndex;
                        Array.Copy( buffer, 0,
                                    m_debugMsgBuffer, m_iDebugBufferIndex,
                                    iCopyLength );

                        FileStream debugStream = new FileStream( m_msgDumpFilePath,
                                                                 FileMode.OpenOrCreate );
                        debugStream.Write( m_debugMsgBuffer, 0, m_debugMsgBuffer.Length );
                        debugStream.Close();

                        int iLeftLength = buffer.Length - iCopyLength;
                        Array.Copy( buffer, iCopyLength,
                                    m_debugMsgBuffer, 0,
                                    iLeftLength );

                        m_iDebugBufferIndex = iLeftLength;
                    }

                    #endif

                    #endregion
                }

                #endregion

                MemoryStream     stream = new MemoryStream(combinedBuffer);
                Protocol4FReader reader = CreateReader(stream);

                MessageType msgType;

                byte[] leftOver = null;
                do
                {
                    long iStreamPos = stream.Position;

                    #region Check if there's still enough data for header

                    // Is the data still enough for reading the header?
                    int iLength = (int)(stream.Length - stream.Position);
                    if ( iLength<Header.StructSize )
                    {
                        if ( iLength<4 )
                        {
                            leftOver = reader.ReadBytes( iLength );
                            break;
                        }
                        else
                        {
                            long iPos       = stream.Position;
                            msgType         = (MessageType)reader.ReadInt32();
                            stream.Position = iPos;
                            if ( msgType!=MessageType.File_EndOfPacket )
                            {
                                leftOver = reader.ReadBytes( iLength );
                                break;
                            }
                        }
                    }

                    #endregion

                    // Read the header of the packet.
                    Header header = new Header();
                    header.Read(reader);

                    msgType = header.Type;
                    if ( msgType==MessageType.File_EndOfPacket && m_bEoPReceived==true )
                        continue;

                    m_bEoPReceived = ( msgType==MessageType.File_EndOfPacket );

                    #region Check if there's still enough data for packet contents

                    // Is the data still enough for reading the packet contents?
                    int iPacketSize = header.Size + header.ArraySize;
                    if ( (stream.Length-stream.Position)<iPacketSize )
                    {
                        // Rewind to the beginning of the packet.
                        stream.Position = iStreamPos;
                        leftOver = reader.ReadBytes( (int)(stream.Length-stream.Position) );
                        break;
                    }

                    #endregion

                    // Log to debug console
                    string output = " - Received type: 0x" + header.Type.ToString("X") +
                                    " size:" + header.Size.ToString() +
                                    "( " + header.Type.ToString() + " )";

                    DebugConsole.DebugConsole.Instance.WriteLine( output );
                    Trace.WriteLine( output );

                    // Process the received message
                    if (this.m_processor != null)
                    {
                        if ( this.m_processor.FilterMessage(header.Type)==true )
                            this.m_processor.ProcessMessage( header, reader );
                    }

                    // Force the stream to the position right after the processed packet,
                    // in case the packet processor didn't read through the whole packet.
                    iStreamPos += (long)Header.StructSize +
                                  (long)header.Size +
                                  (long)header.ArraySize;
                    if ( stream.Position!=iStreamPos )
                        stream.Position = iStreamPos;

                } while ( stream.Position<stream.Length );

                // Save the left overs for next time.
                m_leftOverBuffer = leftOver;

                // Clean up
                stream.Close();
            }
            catch (Exception ex)
            {
                DebugConsole.DebugConsole.Instance.WriteLine(ex.Message);
                Trace.Write(ex.Message);
            }
        }


        /// <summary>
        /// Ping the viewer to make sure it is still there and vice versa.
        /// </summary>
        private void PingViewer()
        {
            #if ENABLE_VIEWER_PING_MSG

            if ( m_activeConnecter==null )
                return;

            ConnectionContext context = m_activeConnecter.GetConnectionContext();
            if ( context==null )
                return;

            // Get the tick count ( in milliseconds ) since the system started.
            int iCurrTickCount = Environment.TickCount;
            if ( (iCurrTickCount - m_iLastPingTime)<PING_PERIOD )
                return;

            // Reset the timer
            m_iLastPingTime = iCurrTickCount;

        #if BUILD_FOR_CTR
            // CTR実機の場合は, Pingの応答を返さないためPingメッセージは送らない.
            if ( context.Platform == PlatformType.CTR )
            { return; }
        #endif//BUILD_FOR_CTR

            // Create buffer for the ping message
            byte[] pingMsgBuffer = new byte[] { 0x00, 0x00, 0x00, 0x00,
                                                0x00, 0x00, 0x00, 0x00,
                                                0x00, 0x00, 0x00, 0x00 };

            // Send out the ping message.
            bool bResult = m_activeConnecter.Send( pingMsgBuffer,
                                                   pingMsgBuffer.Length );
            #endif
        }


        /// <summary>
        /// Log the out going messages to debug console.
        /// </summary>
        [Conditional("DEBUG")]
        private void TraceWriteStream( MemoryStream stream )
        {
            stream.Position = 0;

            var    reader = CreateReader( stream );
            string output = String.Empty;

            DebugConsole.DebugConsole.Instance.WriteLine( "---> size:" + stream.Length.ToString() );
            while ( stream.Position<stream.Length )
            {
                output = String.Empty;

                // Read the header
                Header header = new Header();
                {
                    header.Read( reader );
                    output += " EM4F → : " + header.ToString();
                }

                // Read the message body
                BaseMessagePacket messagePacket =
                    MessageTypeAttribute.CreateMessagePacket( header.Type );
                {
                    messagePacket.Read( reader );
                    output += " : " + messagePacket.ToString();
                }

                // Output the array data size.
                if ( messagePacket.ArrayLength>0 )
                {
                    output += String.Format( " : Array Size : {0} bytes", messagePacket.ArrayLength );
                    stream.Position += messagePacket.ArrayLength;
                }

                // Log to debug console
                DebugConsole.DebugConsole.Instance.WriteLine( output );
            }

            DebugConsole.DebugConsole.Instance.WriteLine("<---");
        }

        #endregion

        #region Event

        /// <summary>
        /// Triggers when the connection is changed.
        /// </summary>
        public event ConnectionChangedEventHandler ConnectionChanged;

        /// <summary>
        /// Connection changed event invoker.
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="args">Event arguments.</param>
        private delegate void ConnectionChangedEventInvoker( object sender,
                                                             MCSConnectionChangedEventArgs args );

        /// <summary>
        /// 接続状態が変更されたときにイベントをコールします。
        /// </summary>
        internal void TriggerConnectionChangedEvent( bool bConnected )
        {
            m_syncContext.Post( ( s ) =>
            {
                if ( this.ConnectionChanged!=null )
                {
                    this.ConnectionChanged( this, new MCSConnectionChangedEventArgs(bConnected) );
                }
            }, null );
        }

        #endregion

        #region Member variables

        private const int AUTO_CONNECT_TIME = 5000; // 5秒に一回接続チェック
        private const int TIMEOUT_MS        = 500;
        private const int SLEEP_TIME        = 200;  // 0.2sec に一回パケットを送ります。
        private const int PING_PERIOD       = 1000;
        private const int BUFFER_SIZE       = 524288; // 512K Bytes, 2 times as big as the buffer for the socket.

        private MessageQueue      m_messageQueue      = new MessageQueue();
        private List<IConnecter>  m_connecters        = new List<IConnecter>();
        private IConnecter        m_activeConnecter   = null;
        private ConnectionContext m_autoConnContext   = null;
        private IMessageProcessor m_processor         = null;
        private Thread            m_thread            = null;

        private SynchronizationContext m_syncContext  = null;

        private byte[]            m_leftOverBuffer    = null;
        private bool              m_bEoPReceived      = false;
        private int               m_iLastPingTime     = 0;

        private volatile int      m_iMessageCount     = 0;

        // Flags
        private          bool     m_bAutoConnect      = false;
        private volatile bool     m_bEnabled          = true;
        private volatile bool     m_bEnableCallBack   = false;
        private volatile bool     m_bConnected        = false;
        private volatile bool     m_bRequestCloseConn = false;
        private volatile bool     m_bFinalizing       = false;
        private volatile bool     m_bConnecting       = false;
        private volatile bool     m_bClosing          = false;

        #endregion
    }

    /// <summary>
    /// コネクタ変更時のイベント用
    /// </summary>
    /// <param name="sender">送信者</param>
    /// <param name="newConnecter">変更される新しいコネクタ</param>
    public delegate void ChangedConnecterEventHandler(object sender, IConnecter newConnecter);
}
