﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.ComponentModel;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;
using NintendoWare.ToolDevelopmentKit.Conversion;
using NintendoWare.ToolDevelopmentKit.Xml;
using NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0;

namespace NWCore.Serializer.Major_1.Minor_9.Build_0.Revision_0
{
    /// <summary>
    /// Class for serializer for the user functions.
    /// </summary>
    [Serializable]
    public class UserFunctionXML
    {
        #region Class for value information

        /// <summary>
        /// Value information.
        /// </summary>
        [Serializable]
        public class ValueInfo
        {
            #region Constructors

            /// <summary>
            /// Default constructor.
            /// </summary>
            public ValueInfo()
            {
                this.Name           = string.Empty;
                this.NameCRC        = 0;
                this.Value          = null;
                this.ValueType      = null;
                this.SerializedType = null;
            }

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="name">Name of the field.</param>
            /// <param name="iNameCRC">CRC hash code of the name.</param>
            /// <param name="value">Value.</param>
            public ValueInfo( string name,
                              uint iNameCRC,
                              object value )
            {
                this.Name    = string.Copy( name );
                this.NameCRC = iNameCRC;
                this.Value   = value;
                if ( value!=null )
                {
                    this.ValueType      = value.GetType();
                    this.SerializedType = this.ValueType;
                }
                else
                {
                    this.ValueType      = null;
                    this.SerializedType = null;
                }
            }


            /// <summary>
            /// Constructor for manually assign the value type.
            /// </summary>
            /// <param name="name">Name of the field.</param>
            /// <param name="iNameCRC">CRC hash code of the name.</param>
            /// <param name="value">Value.</param>
            /// <param name="valueType">Value type.</param>
            public ValueInfo( string name,
                              uint iNameCRC,
                              object value,
                              Type valueType )
            {
                this.Name           = string.Copy( name );
                this.NameCRC        = iNameCRC;
                this.Value          = value;
                this.ValueType      = valueType;
                this.SerializedType = value.GetType();
            }

            #endregion

            #region Properties

            /// <summary>The field name.</summary>
            [XmlAttribute( "Name" )]
            public string Name { get; set; }

            /// <summary>CRC hash code for the field name.</summary>
            [XmlAttribute( "NameCRC" )]
            public uint NameCRC { get; set; }

            /// <summary>The converted string representation of the value type.</summary>
            [XmlAttribute( "Type" )]
            public string ValueTypeStr { get; set; }

            /// <summary>The converted string representation of the actual value type.</summary>
            [XmlAttribute( "SerializeType" )]
            public string SerializedTypeStr { get; set; }

            /// <summary>The converted string representation of the value.</summary>
            [XmlElement( "Value" )]
            public string ValueStr { get; set; }

            /// <summary>The actual value type.</summary>
            [System.Xml.Serialization.XmlIgnore]
            public Type SerializedType { get; set; }

            /// <summary>The value type.</summary>
            [System.Xml.Serialization.XmlIgnore]
            public Type ValueType { get; set; }

            /// <summary>The value of the field.</summary>
            [System.Xml.Serialization.XmlIgnore]
            public object Value { get; set; }

            #endregion

            #region Pre serialize / post deserialize

            /// <summary>
            /// Pre-serialization. This method should be called before serialize.
            /// </summary>
            public void PreSerialize()
            {
                this.ValueTypeStr      = this.ValueType.AssemblyQualifiedName;
                this.SerializedTypeStr = this.SerializedType.AssemblyQualifiedName;

                // Is the value an array?
                if ( this.SerializedType.IsArray==true )
                {
                    // Convert the array to coma delimited string.
                    this.ValueStr = ConvertArrayToString( (this.Value as Array) );
                }
                else
                {
                    // Convert the value to a string.
                    this.ValueStr = this.Value.ToString();
                }
            }


            /// <summary>
            /// Post-deserialization. This method should be called after deserialize.
            /// </summary>
            public void PostDeserialize()
            {
                this.ValueType      = Type.GetType( this.ValueTypeStr );
                this.SerializedType = Type.GetType( this.SerializedTypeStr );

                if ( this.SerializedType.IsArray==true )
                {
                    // Convert the value string to array
                    this.Value = ConvertStringToArray( this.ValueStr );
                }
                else
                {
                    // Convert the value string to the actual data
                    TypeConverter converter =
                        TypeDescriptor.GetConverter( this.SerializedType );
                    if ( converter!=null )
                        this.Value = converter.ConvertFromString( this.ValueStr );
                }
            }

            #endregion

            #region Utilities

            /// <summary>
            /// Convert arrays to coma delimited strings.
            /// </summary>
            /// <param name="array">The array in any generic data type.</param>
            /// <returns>The converted string.</returns>
            private string ConvertArrayToString( Array array )
            {
                string output = "";
                for ( int i=0;i<array.Length;++i )
                {
                    if ( i>0 )
                        output += ",";

                    output += array.GetValue(i).ToString();
                }
                return output;
            }


            /// <summary>
            /// Convert coma delimited strings to generic data type data array.
            /// </summary>
            /// <param name="str">The coma delimited string.</param>
            /// <returns>The result array.</returns>
            private object ConvertStringToArray( string str )
            {
                char[] delimiters = new char[] { ',' };

                // Split the string into tokens.
                string[] tokens = str.Split( delimiters,
                                             StringSplitOptions.RemoveEmptyEntries );

                // Get the type of the array elements.
                Type elementType = this.SerializedType.GetElementType();
                if ( elementType==null )
                    return null;

                // Get type converter of the element type.
                TypeConverter converter =
                    TypeDescriptor.GetConverter( elementType );
                if ( converter==null )
                    return null;

                // Convert the tokens.
                System.Collections.ArrayList list =
                    new System.Collections.ArrayList( tokens.Length );
                foreach ( string token in tokens )
                {
                    list.Add( converter.ConvertFromString( token ) );
                }

                // Convert ArrayList to a generic data type array.
                return list.ToArray( elementType );
            }

            #endregion
        }

        #endregion

        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserFunctionXML()
        {
            this.UserFuncName = string.Empty;
            this.UserFuncID   = 0;
            this.Fields       = new List<ValueInfo>();
        }


        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="name">Name of the user function.</param>
        /// <param name="iID">ID of the user function.</param>
        public UserFunctionXML( string name,
                                uint iID )
        {
            this.UserFuncName = string.Copy( name );
            this.UserFuncID   = iID;
            this.Fields       = new List<ValueInfo>();
        }

        #endregion

        #region Properties

        /// <summary>The name of the user function.</summary>
        [XmlAttribute("Name")]
        public string UserFuncName { get; set; }


        /// <summary>The ID of the user function.</summary>
        [XmlAttribute( "ID" )]
        public uint UserFuncID { get; set; }


        /// <summary>The value fields for the user function.</summary>
        [XmlArrayItem("Field")]
        public List<ValueInfo> Fields { get; set; }

        #endregion

        #region Add / get value

        /// <summary>
        /// Get the value with the given name.
        /// </summary>
        /// <param name="iNameCRC">Name CRC hash code of the value to get.</param>
        /// <returns>
        /// The value found with the given name, or null if the
        /// given name is not found.
        /// </returns>
        public object Get( uint iNameCRC )
        {
            // Try to get the value.
            ValueInfo info;
            bool bResult = m_dictionary.TryGetValue( iNameCRC, out info );
            if ( bResult==false )
                return null;

            if ( info.ValueType!=null &&
                 info.ValueType.GetInterface( "NintendoWare.ToolDevelopmentKit.IVector" )!=null )
            {
                float[] array = info.Value as float[];
                if ( array.Length==1 )
                    return new NintendoWare.ToolDevelopmentKit.Vector1( array );
                else if ( array.Length==2 )
                    return new NintendoWare.ToolDevelopmentKit.Vector2( array );
                else if ( array.Length==3 )
                    return new NintendoWare.ToolDevelopmentKit.Vector3( array );
                else if ( array.Length==4 )
                    return new NintendoWare.ToolDevelopmentKit.Vector4( array );
            }
            else if ( info.ValueType!=null &&
                      info.ValueType.GetInterface( "NWCore.DataModel.IVectori" )!=null )
            {
                int[] array = info.Value as int[];
                if ( array.Length==1 )
                    return new NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0.Vector1i( array );
                else if ( array.Length==2 )
                    return new NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0.Vector2i( array );
                else if ( array.Length==3 )
                    return new NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0.Vector3i( array );
                else if ( array.Length==4 )
                    return new NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0.Vector4i( array );
            }
            else if ( info.ValueType!=null &&
                      info.ValueType.Equals( typeof( NintendoWare.ToolDevelopmentKit.RgbaColor ) )==true )
            {
                float[] array = info.Value as float[];
                if ( array.Length==4 )
                {
                    return new NintendoWare.ToolDevelopmentKit.RgbaColor( array[0],
                                                                          array[1],
                                                                          array[2],
                                                                          array[3] );
                }
            }
            else if ( info.ValueType!=null &&
                      info.ValueType.Equals( typeof( System.Drawing.Color ) )==true )
            {
                byte[] array = info.Value as byte[];
                if ( array.Length==4 )
                {
                    return System.Drawing.Color.FromArgb( array[0],
                                                          array[1],
                                                          array[2],
                                                          array[3] );
                }
            }

            return info.Value;
        }


        /// <summary>
        /// Add a name/value pair to the user function.
        /// </summary>
        /// <param name="name">Name of the field.</param>
        /// <param name="iNameCRC">Name CRC hash code.</param>
        /// <param name="value">Value.</param>
        /// <returns>True on success.</returns>
        public bool Add( string name,
                         uint iNameCRC,
                         object value )
        {
            // Is the value serializable?
            if ( value!=null )
            {
                Type valueType = value.GetType();
                if ( value is NintendoWare.ToolDevelopmentKit.IVector )
                {
                    return AddVector( name,
                                      iNameCRC,
                                      value as NintendoWare.ToolDevelopmentKit.IVector );
                }
                else if ( value is NWCore.DataModel.IVectori )
                {
                    return AddVector( name,
                                      iNameCRC,
                                      value as NWCore.DataModel.IVectori );
                }
                else if ( value is NintendoWare.ToolDevelopmentKit.RgbaColor )
                {
                    return AddRgbaColor( name,
                                         iNameCRC,
                                         value as NintendoWare.ToolDevelopmentKit.RgbaColor );
                }
                else if ( value is System.Drawing.Color )
                {
                    return AddColor( name,
                                     iNameCRC,
                                     (System.Drawing.Color)value );
                }
                if ( valueType.IsSerializable==false )
                {
                    Debug.Assert( false,
                                  "UserFunctionXML.Add : The value (" +
                                  valueType.ToString() +
                                  ") is not serializable." );
                    return false;
                }
            }

            // Does the name already exist?
            if ( m_dictionary.ContainsKey( iNameCRC )==true )
            {
                Debug.Assert( false,
                              "UserFunctionXML.Add : The name \"" +
                              name +
                              "\" already exists." );
                return false;
            }

            // Add the name/value pair
            m_dictionary.Add( iNameCRC, new ValueInfo( name, iNameCRC, value ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name of the field.</param>
        /// <param name="iNameCRC">Name CRC hash code.</param>
        /// <param name="vec">Vector of float values.</param>
        /// <returns>True on success.</returns>
        private bool AddVector( string name,
                                uint iNameCRC,
                                NintendoWare.ToolDevelopmentKit.IVector vec )
        {
            // Does the name already exist?
            if ( m_dictionary.ContainsKey( iNameCRC )==true )
            {
                Debug.Assert( false,
                              "SerializableNameValueGroup.AddVector : The name \"" +
                              iNameCRC +
                              "\" already exists." );
                return false;
            }

            // Add the converted float array
            m_dictionary.Add( iNameCRC, new ValueInfo( name,
                                                       iNameCRC,
                                                       vec.ToArray(),
                                                       vec.GetType() ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name of the field.</param>
        /// <param name="iNameCRC">Name CRC hash code.</param>
        /// <param name="vec">Vector of integer values.</param>
        /// <returns>True on success.</returns>
        private bool AddVector( string name,
                                uint iNameCRC,
                                NWCore.DataModel.IVectori vec )
        {
            // Does the name already exist?
            if ( m_dictionary.ContainsKey( iNameCRC )==true )
            {
                Debug.Assert( false,
                              "SerializableNameValueGroup.AddVector : The name \"" +
                              iNameCRC +
                              "\" already exists." );
                return false;
            }

            // Add the converted float array
            m_dictionary.Add( iNameCRC, new ValueInfo( name,
                                                       iNameCRC,
                                                       vec.ToArray(),
                                                       vec.GetType() ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name of the field.</param>
        /// <param name="iNameCRC">Name CRC hash code.</param>
        /// <param name="color">Color object to add.</param>
        /// <returns>True on success.</returns>
        private bool AddRgbaColor( string name,
                                   uint iNameCRC,
                                   NintendoWare.ToolDevelopmentKit.RgbaColor color )
        {
            // Does the name already exist?
            if ( m_dictionary.ContainsKey( iNameCRC )==true )
            {
                Debug.Assert( false,
                              "SerializableNameValueGroup.AddVector : The name \"" +
                              iNameCRC +
                              "\" already exists." );
                return false;
            }

            // Store the color channels in a float array
            float[] array = new float[] { color.R,
                                          color.G,
                                          color.B,
                                          color.A };

            // Add the converted float array
            m_dictionary.Add( iNameCRC, new ValueInfo( name,
                                                       iNameCRC,
                                                       array,
                                                       color.GetType() ) );

            return true;
        }


        /// <summary>
        /// Add a name/value pair to the group.
        /// </summary>
        /// <param name="name">Name of the field.</param>
        /// <param name="iNameCRC">Name CRC hash code.</param>
        /// <param name="color">Color object to add.</param>
        /// <returns>True on success.</returns>
        private bool AddColor( string name,
                               uint iNameCRC,
                               System.Drawing.Color color )
        {
            // Does the name already exist?
            if ( m_dictionary.ContainsKey( iNameCRC )==true )
            {
                Debug.Assert( false,
                              "SerializableNameValueGroup.AddVector : The name \"" +
                              iNameCRC +
                              "\" already exists." );
                return false;
            }

            // Store the color channels in a float array
            byte[] array = new byte[] { color.A,
                                        color.R,
                                        color.G,
                                        color.B };

            // Add the converted float array
            m_dictionary.Add( iNameCRC, new ValueInfo( name,
                                                       iNameCRC,
                                                       array,
                                                       color.GetType() ) );

            return true;
        }


        #endregion

        #region Pre serialize / post deserialize

        /// <summary>
        /// Pre-serialization. This method should be called before serialize.
        /// </summary>
        public void PreSerialize()
        {
            this.Fields.Clear();

            // Dictionaries cannot be serialized directly, so we need to
            // fill the data into the list before this can be serialized.
            foreach ( ValueInfo info in m_dictionary.Values )
            {
                if ( info==null )
                    continue;

                info.PreSerialize();

                this.Fields.Add( info );
            }
        }


        /// <summary>
        /// Post-deserialization. This method should be called after deserialize.
        /// </summary>
        public void PostDeserialize()
        {
            m_dictionary.Clear();
            foreach ( ValueInfo info in Fields )
            {
                if ( info==null )
                    continue;

                m_dictionary.Add( info.NameCRC, info );

                info.PostDeserialize();
            }
        }

        #endregion

        #region Utility methods

        public void Copy( UserFunctionXML src )
        {
            this.m_dictionary.Clear();
            foreach ( ValueInfo field in src.m_dictionary.Values )
            {
                this.Add( field.Name, field.NameCRC, field.Value );
            }
        }

        #endregion

        #region Member variables

        private Dictionary<uint, ValueInfo> m_dictionary =
            new Dictionary<uint, ValueInfo>();

        #endregion
    }


    /// <summary>
    /// Class for the serializer of the game config data.
    /// </summary>
    //[XmlRoot( "root" )]
    public class GameConfigDataXml
    {
        #region Construtors

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GameConfigDataXml()
        {
            this.Name             = string.Empty;
            this.Comment          = string.Empty;
            this.EmitterSetPath   = string.Empty;
            this.UserFunctionList = new List<UserFunctionXML>();

            m_data = new GameConfigData();
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public GameConfigDataXml( string esetFilePath,
                                  GameConfigData data ) :
            this()
        {
            this.Name             = string.Empty;
            this.Comment          = string.Empty;
            this.EmitterSetPath   = string.Copy( esetFilePath );
            this.UserFunctionList = new List<UserFunctionXML>();

            m_data.Set( data );
        }

        #endregion

        #region Properties : EmitterSet

        /// <summary>
        /// File path to the owner emitter set.
        /// </summary>
        public string EmitterSetPath { get; set; }

        #endregion

        #region Properties : GameConfigData

        /// <summary>
        /// Get the game config data.
        /// </summary>
        [System.Xml.Serialization.XmlIgnore]
        public GameConfigData GameConfigData
        {
            get { return m_data; }
        }

        /// <summary>
        /// プレビューの名前
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// プレビューの名前
        /// </summary>
        public string Comment { get; set; }

        /// <summary>
        /// プレビューモデルのインデックス
        /// </summary>
        public int PreviewModelIndex
        {
            get { return m_data.PreviewModelIndex; }
            set { m_data.PreviewModelIndex = value; }
        }

        /// <summary>
        /// 接続先のモデル
        /// </summary>
        public string ModelName
        {
            get { return m_data.ModelName; }
            set { m_data.ModelName = value; }
        }

        /// <summary>
        /// 接続先のボーンのインデックス
        /// </summary>
        public int BoneIndex
        {
            get { return m_data.BoneIndex; }
            set { m_data.BoneIndex = value; }
        }

        /// <summary>
        /// 接続先のボーン
        /// </summary>
        public string BoneName
        {
            get { return m_data.BoneName; }
            set { m_data.BoneName = value; }
        }

//         /// <summary>
//         /// DEBUG:エミッタセットが有効か？
//         /// </summary>
//         public bool DispEmitterSet
//         {
//             get { return m_data.DispEmitterSet; }
//             set { m_data.DispEmitterSet = value; }
//         }
//
        /// <summary>
        /// エミッタセットの名前
        /// </summary>
        public string EmitterSetName
        {
            get { return m_data.EmitterSetName; }
            set { m_data.EmitterSetName = value; }
        }

        /// <summary>
        /// 呼び出し開始フレーム
        /// </summary>
        public int StartFrame
        {
            get { return m_data.StartFrame; }
            set { m_data.StartFrame = value; }
        }

        /// <summary>
        /// 再生開始フレーム
        /// </summary>
        public int PlaybackStartFrame
        {
            get { return m_data.PlaybackStartFrame; }
            set { m_data.PlaybackStartFrame = value; }
        }

        /// <summary>
        /// 強制開始 フレーム
        /// </summary>
        public int ForceStartFrame
        {
            get { return m_data.ForceStartFrame; }
            set { m_data.ForceStartFrame = value; }
        }

        /// <summary>
        /// 色（乗算値）
        /// </summary>
        public Vector4Xml MultColor
        {
            get { return UtilityXml.ToVector4Xml( m_data.MultColor ); }
            set { m_data.MultColor = UtilityXml.ToVector4( value ); }
        }

        /// <summary>
        /// パーティクルの大きさ
        /// </summary>
        public Vector2Xml ParticleScale
        {
            get { return UtilityXml.ToVector2Xml( m_data.ParticleScale ); }
            set { m_data.ParticleScale = UtilityXml.ToVector2( value ); }
        }

        /// <summary>
        /// 放出時のみ作用するパーティクルの大きさ
        /// </summary>
        public Vector2Xml EmissionParticleScale
        {
            get { return UtilityXml.ToVector2Xml( m_data.EmissionParticleScale ); }
            set { m_data.EmissionParticleScale = UtilityXml.ToVector2( value ); }
        }

        /// <summary>
        /// 行列設定タイプ
        /// </summary>
        public MatrixSetType MartixSetType
        {
            get { return m_data.MartixSetType; }
            set { m_data.MartixSetType = value; }
        }

        /// <summary>
        /// 行列適用モード
        /// </summary>
        public MtxApplyMode MtxApplyMode
        {
            get { return m_data.MtxApplyMode; }
            set { m_data.MtxApplyMode = value; }
        }

        /// <summary>
        /// オフセット行列 - スケール
        /// </summary>
        public Vector3Xml OffsetScale
        {
            get { return UtilityXml.ToVector3Xml( m_data.OffsetScale ); }
            set { m_data.OffsetScale = UtilityXml.ToVector3( value ); }
        }

        /// <summary>
        /// オフセット行列 - エミッタビルボード
        /// </summary>
        public bool EmitterBillboard
        {
            get { return m_data.EmitterBillboard; }
            set { m_data.EmitterBillboard = value; }
        }

        /// <summary>
        /// オフセット行列 - 回転X
        /// </summary>
        public float OffsetRotateX
        {
            get { return m_data.OffsetRotateX; }
            set { m_data.OffsetRotateX = value; }
        }

        /// <summary>
        /// オフセット行列 - 回転Y
        /// </summary>
        public float OffsetRotateY
        {
            get { return m_data.OffsetRotateY; }
            set { m_data.OffsetRotateY = value; }
        }

        /// <summary>
        /// オフセット行列 - 回転Z
        /// </summary>
        public float OffsetRotateZ
        {
            get { return m_data.OffsetRotateZ; }
            set { m_data.OffsetRotateZ = value; }
        }

        /// <summary>
        /// オフセット行列 - 座標
        /// </summary>
        public Vector3Xml OffsetPosition
        {
            get { return UtilityXml.ToVector3Xml( m_data.OffsetPosition ); }
            set { m_data.OffsetPosition = UtilityXml.ToVector3( value ); }
        }

        /// <summary>
        /// エミッタスケール
        /// </summary>
        public Vector3Xml EmitterScale
        {
            get { return UtilityXml.ToVector3Xml( m_data.EmitterScale ); }
            set { m_data.EmitterScale = UtilityXml.ToVector3( value ); }
        }

        /// <summary>
        /// 初速度 - 全方向速度のスケール値
        /// </summary>
        public float AllDirectionalVelScale
        {
            get { return m_data.AllDirectionalVelScale; }
            set { m_data.AllDirectionalVelScale = value; }
        }

        /// <summary>
        /// 初速度 - 指定方向速度のスケール値
        /// </summary>
        public float DirectionalVelScale
        {
            get { return m_data.DirectionalVelScale; }
            set { m_data.DirectionalVelScale = value; }
        }

        /// <summary>
        /// 初速度 - ランダムスケール値
        /// </summary>
        public float RandomVel
        {
            get { return m_data.RandomVel; }
            set { m_data.RandomVel = value; }
        }

        /// <summary>
        /// 初速度 - 最終的に出てきた速度に加算する値
        /// </summary>
        public Vector3Xml AdditionalVel
        {
            get { return UtilityXml.ToVector3Xml( m_data.AdditionalVel ); }
            set { m_data.AdditionalVel = UtilityXml.ToVector3( value ); }
        }

        /// <summary>
        /// 初速度 - 指定方向の方向を指定するか
        /// </summary>
        public bool SpecifyDirection
        {
            get { return m_data.SpecifyDirection; }
            set { m_data.SpecifyDirection = value; }
        }

        /// <summary>
        /// 初速度 - 指定方向の方向
        /// </summary>
        public Vector3Xml DirectionalVel
        {
            get { return UtilityXml.ToVector3Xml( m_data.DirectionalVel ); }
            set { m_data.DirectionalVel = UtilityXml.ToVector3( value ); }
        }

        #endregion

        #region Properties : UserFunction

        /// <summary>
        /// The list of the user functions.
        /// </summary>
        [XmlArrayItem( "UserFunction" )]
        public List<UserFunctionXML> UserFunctionList { get; set; }

        /// <summary>
        /// The flag indicating if the model linkage page is enabled.
        /// </summary>
        public bool ModelLinkagePageEnabled { get; set; }

        #endregion

        #region Pre serialize / post deserialize

        /// <summary>
        /// Pre-serialization. This method should be called before serialize.
        /// </summary>
        public void PreSerialize()
        {
            foreach ( UserFunctionXML userFunc in this.UserFunctionList )
            {
                if ( userFunc==null )
                    continue;

                userFunc.PreSerialize();
            }
        }


        /// <summary>
        /// Post-deserialization. This method should be called after deserialize.
        /// </summary>
        public void PostDeserialize()
        {
            foreach ( UserFunctionXML userFunc in this.UserFunctionList )
            {
                if ( userFunc==null )
                    continue;

                userFunc.PostDeserialize();
            }
        }

        #endregion

        #region Member variables

        private GameConfigData m_data = null;

        #endregion
    }
}
