﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading;

using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Core;

namespace EffectMaker.Foundation.Log
{
    /// <summary>
    /// Enum for log levels.
    /// </summary>
    public enum LogLevels
    {
        /// <summary>
        /// Debug log.
        /// </summary>
        Debug,

        /// <summary>
        /// Information log.
        /// </summary>
        Information,

        /// <summary>
        /// Warning log.
        /// </summary>
        Warning,

        /// <summary>
        /// Error log.
        /// </summary>
        Error,

        /// <summary>
        /// Fatal error log.
        /// </summary>
        Fatal
    }

    /// <summary>
    /// ログを管理するクラスです。
    /// </summary>
    public static class Logger
    {
        /// <summary>The delimiters for splitting the destination string.</summary>
        private static char[] destinationDelimiters = new char[] { ',' };

        /// <summary>
        /// ログハンドラー
        /// </summary>
        private static Dictionary<string, ILogHandler> logHandlers =
            new Dictionary<string, ILogHandler>();

        /// <summary>
        /// The synchronization context for dispatching log messages when called from other threads.
        /// </summary>
        private static SynchronizationContext synchContext = null;

        /// <summary>
        /// Default log level.
        /// </summary>
        public static LogLevels DefaultLogLevel { get; set; }

        /// <summary>
        /// Default log destination.
        /// </summary>
        public static string DefaultLogDestination { get; set; }

        /// <summary>
        /// Register the log handler.
        /// </summary>
        /// <param name="handler">The log handler.</param>
        public static void RegisterLogHandler(ILogHandler handler)
        {
            Type handlerType = handler.GetType();

            // Get our alias attribute from the type of the handler.
            var attributes = handlerType.GetCustomAttributes(typeof(AliasAttribute), false);
            if ((attributes.Length <= 0) || ((attributes[0] is AliasAttribute) == false))
            {
                Debug.Assert(false, "Logger.RegisterLogHandler : The registering handler has no Alias attribute.");
            }

            // Get the alias name from the attribute of the log handler.
            var alias = ((AliasAttribute)attributes[0]).Alias;
            if (Logger.logHandlers.ContainsKey(alias) == true)
            {
                Log(
                    LogLevels.Warning,
                    "Logger.RegisterLogHandler : The registering handler name '" + alias + "' has already been registered.");

                return;
            }

            // Save the handler to our dictionary.
            Logger.logHandlers.Add(alias, handler);
        }

        /// <summary>
        /// Set synchronization context for dispatching log messages when called from other threads.
        /// </summary>
        /// <param name="context">The synchronization context.</param>
        public static void SetSynchronizationContext(SynchronizationContext context)
        {
            synchContext = context;
        }

        /// <summary>
        /// Output log message.
        /// </summary>
        /// <param name="destination">The output destination.</param>
        /// <param name="level">The log level.</param>
        /// <param name="format">The format or message to output.</param>
        /// <param name="args">The objects to output with the format.</param>
        public static void Log(
                               string destination,
                               LogLevels level,
                               string format,
                               params object[] args)
        {
            var handlers = new List<ILogHandler>();

            IEnumerable<string> destinations = null;

            // Find the specified log destinations.
            if (string.IsNullOrEmpty(destination) == true)
            {
                // The destination is an empty string, just broadcast the log to everyone.
                handlers.Capacity = Logger.logHandlers.Count;
                foreach (var item in Logger.logHandlers)
                {
                    handlers.Add(item.Value);
                }
            }
            else
            {
                // Parse the comma-delimited string into destinations.
                string[] tokens =
                    destination.Split(destinationDelimiters, StringSplitOptions.RemoveEmptyEntries);

                destinations = tokens;

                // Find the destinations in the registered handler dictionary.
                foreach (string token in tokens)
                {
                    if (Logger.logHandlers.ContainsKey(token) == false)
                    {
                        Debug.WriteLine("Logger.Log : The specified log destination '" + token + "' is not found or not yet registered.");
                        return;
                    }

                    ILogHandler handler = Logger.logHandlers[token];
                    if (handler == null)
                    {
                        Debug.Assert(false, "Logger.Log : The assigned destination '" + token + "' has no longer existed.");
                        return;
                    }

                    handlers.Add(handler);
                }
            }

            // No handlers found, bail out.
            if (handlers.Count <= 0)
            {
                return;
            }

            // 出力メッセージを書式変換。
            string message;
            if (args == null || args.Length <= 0)
            {
                message = format;
            }
            else
            {
                message = string.Format(format, args);
            }

            if (destination == null)
            {
                destinations = Enumerable.Empty<string>();
            }

            // The destination is an empty string, just broadcast the log to everyone.
            foreach (var handler in handlers)
            {
                DispatchLog(handler, destinations, level, message);
            }
        }

        /// <summary>
        /// Output log message to the default destination.
        /// </summary>
        /// <param name="level">The log level.</param>
        /// <param name="format">The format or message to output.</param>
        /// <param name="args">The objects to output with the format.</param>
        public static void Log(
                               LogLevels level,
                               string format,
                               params object[] args)
        {
            Logger.Log(
                Logger.DefaultLogDestination,
                level,
                string.Format(format, args));
        }

        /// <summary>
        /// Output log message with default log level to the default destination.
        /// </summary>
        /// <param name="format">The format or message to output.</param>
        /// <param name="args">The objects to output with the format.</param>
        public static void Log(string format, params object[] args)
        {
            Logger.Log(
                Logger.DefaultLogDestination,
                Logger.DefaultLogLevel,
                string.Format(format, args));
        }

        /// <summary>
        /// Helper method for dispatching the log message to the specified log handler.
        /// </summary>
        /// <param name="handler">The log handler.</param>
        /// <param name="destination">The output destination.</param>
        /// <param name="level">The log level.</param>
        /// <param name="message">The log message.</param>
        private static void DispatchLog(
            ILogHandler handler, IEnumerable<string> destination, LogLevels level, string message)
        {
            if (synchContext == null)
            {
                handler.Log(destination, level, message);
            }
            else
            {
                synchContext.Post(_ => handler.Log(destination, level, message), null);
            }
        }
    }
}
