﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.Foundation.Extensions
{
    /// <summary>
    /// Class containing extension methods for IPrimitive.
    /// </summary>
    public static class PrimitiveExtensions
    {
        /// <summary>
        /// Checks whether an IPrimitive is a PrimitiveBase class or not.
        /// </summary>
        /// <param name="primitive">The IPrimitive instance to check against.</param>
        /// <returns>Returns true if the IPrimitive is a PrimitiveBase class,
        /// false otherwise.</returns>
        public static bool IsPrimitiveBase(this IPrimitive primitive)
        {
            Type dummy;
            return IsPrimitiveBase(primitive, out dummy);
        }

        /// <summary>
        /// Checks whether an IPrimitive is a PrimitiveBase class or not,
        /// and retrive the primitive data type.
        /// </summary>
        /// <param name="primitive">The IPrimitive instance to check against.</param>
        /// <param name="dataType">The type of underlying data of the PrimitiveBase class.</param>
        /// <returns>Returns true if the IPrimitive is a PrimitiveBase class,
        /// false otherwise.</returns>
        public static bool IsPrimitiveBase(this IPrimitive primitive, out Type dataType)
        {
            dataType = null;

            if (primitive == null)
            {
                throw new ArgumentNullException("primitive");
            }

            Type[] genericArguments;
            bool isMatching = primitive.GetType().IsAssignableFromGenericTypeDefinition(
                typeof(PrimitiveBase<>),
                out genericArguments);

            if (isMatching == false || genericArguments.Length != 1)
            {
                return false;
            }

            dataType = genericArguments[0];

            return true;
        }

        /// <summary>
        /// Throws an ArgumentException telling the primitive is of invalid type.
        /// </summary>
        /// <param name="primitive">The primitive of invalid type.</param>
        /// <param name="parameterName">Optional, the name of parameter
        /// given to the ArgumentException.</param>
        public static void ThrowInvalidPrimitiveTypeException(
            IPrimitive primitive,
            string parameterName = null)
        {
            if (primitive == null)
            {
                throw new ArgumentNullException("primitive");
            }

            string message = string.Format(
                "Invalid primitive (type: {0})",
                primitive.GetType().FullName);

            if (string.IsNullOrWhiteSpace(parameterName))
            {
                throw new ArgumentException(message);
            }
            else
            {
                throw new ArgumentException(message, parameterName);
            }
        }

        /// <summary>
        /// Sets values to a primitive from an enumerable.
        /// </summary>
        /// <param name="primitive">The primitive to set.</param>
        /// <param name="source">The source enumerable.</param>
        public static void FromEnumerable(
            this IPrimitive primitive,
            System.Collections.IEnumerable source)
        {
            if (primitive == null)
            {
                throw new ArgumentNullException("primitive");
            }

            if (source == null)
            {
                throw new ArgumentNullException("source");
            }

            System.Collections.IEnumerator e = source.GetEnumerator();

            for (int i = 0; i < primitive.Count; i++)
            {
                if (e.MoveNext() == false)
                {
                    throw new ArgumentOutOfRangeException("source");
                }

                primitive[i] = e.Current;
            }
        }

        /// <summary>
        /// Sets values to a primitive from an enumerable.
        /// </summary>
        /// <typeparam name="T">The type of value.</typeparam>
        /// <param name="primitive">The primitive to set.</param>
        /// <param name="source">The source enumerable.</param>
        public static void FromEnumerable<T>(this IPrimitive primitive, IEnumerable<T> source)
        {
            FromEnumerable(primitive, source as System.Collections.IEnumerable);
        }

        /// <summary>
        /// Creates an enumerable from the values of a primitive.
        /// </summary>
        /// <param name="primitive">The primitive to create values from.</param>
        /// <returns>The sequence of values from the primitive.</returns>
        public static System.Collections.IEnumerable ToEnumerable(this IPrimitive primitive)
        {
            if (primitive == null)
            {
                throw new ArgumentNullException("primitive");
            }

            for (int i = 0; i < primitive.Count; i++)
            {
                yield return primitive[i];
            }
        }

        /// <summary>
        /// Creates an enumerable from the values of a primitive.
        /// </summary>
        /// <typeparam name="T">The type of value.</typeparam>
        /// <param name="primitive">The primitive to create values from.</param>
        /// <returns>The sequence of values from the primitive.</returns>
        public static IEnumerable<T> ToEnumerable<T>(this IPrimitive primitive)
        {
            return ToEnumerable(primitive).Cast<T>();
        }
    }
}
