﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.CodeDom.Compiler;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Xml.Serialization;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;
using Microsoft.CSharp;

namespace EffectMaker.DataModel.Manager
{
    /// <summary>
    /// CustomActionSettingManager
    /// </summary>
    public class CustomActionSettingManager
    {
        /// <summary>
        /// シングルトンインスタンス
        /// </summary>
        public static readonly CustomActionSettingManager Instance = new CustomActionSettingManager();

        /// <summary>
        /// 最大セッティング数
        /// </summary>
        private const int MaxSettingCount = 8;

        /// <summary>
        /// セッチングの型リスト
        /// </summary>
        private readonly List<Type> settingTypes = new List<Type>();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        private CustomActionSettingManager()
        {
        }

        /// <summary>
        /// カスタムアクションセッティング
        /// </summary>
        public List<CustomActionSettingInfo> Settings
        {
            get;
            private set;
        }

        /// <summary>
        /// インスタンスの作成.
        /// </summary>
        public static void Initialize()
        {
            Instance.InitializeInternal();
        }

        /// <summary>
        /// ユーザーデータモデルをロードする.
        /// </summary>
        public void LoadUserDataModels()
        {
            this.settingTypes.Clear();

            foreach (var setting in this.Settings)
            {
                // データモデルを生成.
                var asmInfo = UserDataModelManager.LoadDataModelSourceCode(Path.Combine(UserDataModelManager.ExeDirPath, "Addins", setting.DataModelFileName));
                if (asmInfo == null)
                {
                    this.settingTypes.Add(null);
                    continue;
                }

                var type = asmInfo.Assembly.ExportedTypes.FirstOrDefault();

                this.settingTypes.Add(type);
            }

            while (this.settingTypes.Count() < MaxSettingCount)
            {
                this.settingTypes.Add(null);
            }
        }

        /// <summary>
        /// Helper method for a property getter which returns a data model.
        /// When the backing field is null, this method create a new instance
        /// of the data model to the backing field.
        /// </summary>
        /// <param name="self">The extending data model.</param>
        /// <param name="settingIndex">セッティングインデックス</param>
        /// <param name="field">The backing field of the property.</param>
        /// <returns>The data model stored in the backing field.</returns>
        public CustomActionSettingData GetDataModelFromField(
            CustomActionData self,
            int settingIndex,
            ref CustomActionSettingData field)
        {
            if (field == null)
            {
                field = this.CreateSetting(settingIndex);
                field.Parent = self;
            }

            return field;
        }

        /// <summary>
        /// エクストラデータを取得する.
        /// </summary>
        /// <returns>エクストラデータの配列.</returns>
        public IEnumerable<Type> GetExtraTypes()
        {
            return this.settingTypes.Where(x => x != null);
        }

        /// <summary>
        /// インスタンスの作成.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool InitializeInternal()
        {
            string filePath =
                Path.Combine(UserDataModelManager.ExeDirPath, "Addins\\CustomActionSettings.xml");

            // ファイルからユーザーページ情報をロードする.
            this.Settings = SerializationHelper.Load<List<CustomActionSettingInfo>>(filePath);
            if (this.Settings == null)
            {
                return false;
            }

            this.LoadUserDataModels();

            return true;
        }

        /// <summary>
        /// セッティングオブジェクトの作成
        /// </summary>
        /// <param name="index">セッティングインデックス</param>
        /// <returns>セッティングオブジェクト</returns>
        private CustomActionSettingData CreateSetting(int index)
        {
            if (this.settingTypes[index] == null)
            {
                return new CustomActionSettingData();
            }
            else
            {
                return Activator.CreateInstance(this.settingTypes[index]) as CustomActionSettingData;
            }
        }
    }
}
