﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Xml.Serialization;

using EffectMaker.DataModel.Attributes;

using EffectMaker.Foundation.Collections;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.DataModel.DataModels
{
    /// <summary>
    /// The base data model class.
    /// </summary>
    public abstract class DataModelBase : ICloneable, ISettable
    {
        /////// <summary>The flag indicating whether should set the data model Guid.</summary>
        ////private static bool shouldSetGuid = false;

        /// <summary>
        /// Children data model.
        /// </summary>
        private List<DataModelBase> children;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public DataModelBase()
        {
            ////this.Guid = Guid.NewGuid();
            this.Parent = null;
            this.children = new List<DataModelBase>();
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source data model to copy from.</param>
        public DataModelBase(DataModelBase src) :
            this()
        {
            this.Set(src);
        }

        /////// <summary>
        /////// Get or set the Guid of the data model.
        /////// </summary>
        ////public Guid Guid { get; set; }

        /// <summary>
        /// Get or set the parent data model.
        /// </summary>
        [XmlIgnore]
        public DataModelBase Parent { get; set; }

        /// <summary>
        /// Get or set the children data model.
        /// </summary>
        [XmlIgnore]
        public List<DataModelBase> Children
        {
            get
            {
                return this.children;
            }

            set
            {
                this.children = value;

                foreach (DataModelBase child in this.children)
                {
                    child.Parent = this;
                }
            }
        }

        /// <summary>
        /// Set data from the source data model.
        /// </summary>
        /// <param name="src">The source data model.</param>
        /// <returns>True on success.</returns>
        public virtual bool Set(object src)
        {
            var srcDataModel = src as DataModelBase;
            if (srcDataModel == null)
            {
                return false;
            }

            ////if (shouldSetGuid == true)
            ////{
            ////    this.Guid = srcDataModel.Guid;
            ////}

            return true;
        }

        /// <summary>
        /// Set data from the source data model.
        /// </summary>
        /// <param name="src">The source data model.</param>
        /// <returns>True on success.</returns>
        public virtual bool SetWithoutGuid(object src)
        {
            ////shouldSetGuid = false;
            bool result = this.Set(src);
            ////shouldSetGuid = true;

            return result;
        }

        /// <summary>
        /// Clone this data model and return a new instance.
        /// </summary>
        /// <returns>The created instance.</returns>
        public virtual object Clone()
        {
            var newInstance = Activator.CreateInstance(this.GetType()) as DataModelBase;

            newInstance.Set(this);

            return newInstance;
        }

        /// <summary>
        /// Clone this data model and return a new instance.
        /// </summary>
        /// <returns>The created instance.</returns>
        public virtual object CloneWithoutGuid()
        {
            var newInstance = Activator.CreateInstance(this.GetType()) as DataModelBase;

            newInstance.SetWithoutGuid(this);

            return newInstance;
        }

        /// <summary>
        /// Add data model to the child list.
        /// </summary>
        /// <param name="child">The data model to add.</param>
        public void AddChild(DataModelBase child)
        {
            child.Parent = this;
            this.children.Add(child);
        }

        /// <summary>
        ///  Add a batch of data models to the child list.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public void AddChildren(IEnumerable<DataModelBase> children)
        {
            foreach (DataModelBase child in children)
            {
                child.Parent = this;
            }

            this.children.AddRange(children);
        }

        /// <summary>
        ///  Insert to children data.
        /// </summary>
        /// <param name="index">Index.</param>
        /// <param name="child">Adding data.</param>
        public void InsertChild(int index, DataModelBase child)
        {
            child.Parent = this;
            this.children.Insert(index, child);
        }

        /// <summary>
        ///  Remove from children data.
        /// </summary>
        /// <param name="child">Remove data.</param>
        public void RemoveChild(DataModelBase child)
        {
            if (this.children.Remove(child) == true)
            {
                child.Parent = null;
            }
        }

        /// <summary>
        /// Get property Guid.
        /// </summary>
        /// <param name="propertyName">The name of the property.</param>
        /// <param name="guid">The guid.</param>
        /// <returns>
        /// True on success.
        /// False if the property is not found or the DataModelPropertyGuidAttribute
        /// is not assigned to the property.
        /// </returns>
        public bool GetPropertyGuid(string propertyName, out Guid guid)
        {
            // Find the property.
            PropertyInfo info = this.GetType().GetProperty(propertyName);
            if (info == null)
            {
                guid = Guid.Empty;
                return false;
            }

            // Get the data model property Guid attribute.
            var attr = info.GetCustomAttribute(typeof(DataModelPropertyGuidAttribute), false) as DataModelPropertyGuidAttribute;
            if (attr == null)
            {
                guid = Guid.Empty;
                return false;
            }

            guid = attr.Guid;

            return true;
        }

        /// <summary>
        /// Get property value (including properties in child data models)
        /// with the property's path.
        /// </summary>
        /// <param name="path">The path to the property.</param>
        /// <param name="value">The property value.</param>
        /// <returns>True if the data model has the property information.</returns>
        public bool GetPropertyValueWithPath(PathNodeList path, out object value)
        {
            value = null;

            object obj = this;
            foreach (string token in path)
            {
                if (obj == null)
                {
                    return false;
                }

                PropertyInfo propertyInfo = obj.GetType().GetProperty(token);
                if (propertyInfo == null)
                {
                    return false;
                }

                obj = propertyInfo.GetValue(obj);
            }

            value = obj;

            return true;
        }
    }
}
