﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Xml.Serialization;
using System.Globalization;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_5.Build_0.Revision_0
{
    /// <summary>
    /// Class for the data model for the animation table data.
    /// </summary>
    public class AnimTableData : BaseDataModel<AnimTableData>
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public AnimTableData()
        {
            this.KeyFrames = new List<AnimTableKeyFrameData>();
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="source">The source to copy from.</param>
        public AnimTableData( AnimTableData source ) :
            this()
        {
            this.Set(source);
        }


        /// <summary>
        /// Initialize the data.
        /// </summary>
        public override void Initialize()
        {
            this.TimeBaseType      = AnimTimeBaseTypes.Emitter;
            this.Name              = AnimTableTargetTypes.unknown;
            this.IsEnabled         = false;
            this.IsLoop            = false;
            this.InterpolationType = AnimInterpolationTypes.Linear;

            if ( this.KeyFrames==null )
                this.KeyFrames = new List<AnimTableKeyFrameData>();

            this.KeyFrames.Clear();
        }

        #endregion

        #region Properties

        /// <summary>
        /// The type of the animation playback timing.
        /// </summary>
        public AnimTimeBaseTypes TimeBaseType { get; set; }


        /// <summary>
        /// The applied target type name of the animation table.
        /// </summary>
        public AnimTableTargetTypes Name { get; set; }


        /// <summary>
        /// Is the animation table enabled?
        /// </summary>
        public bool IsEnabled { get; set; }


        /// <summary>
        /// Is loop enabled for this animation.
        /// </summary>
        public bool IsLoop { get; set; }


        /// <summary>
        /// The type of the interpolation mode between key frames.
        /// </summary>
        public AnimInterpolationTypes InterpolationType { get; set; }


        /// <summary>
        /// The list of animation key frames.
        /// </summary>
        public List<AnimTableKeyFrameData> KeyFrames
        {
            get { return m_keyFrames; }
            set
            {
                if ( value==null )
                {
                    if ( m_keyFrames!=null )
                        m_keyFrames.Clear();
                    return;
                }

                if ( m_keyFrames==null )
                    m_keyFrames = new List<AnimTableKeyFrameData>( value.Count );

                m_keyFrames.Clear();
                foreach ( AnimTableKeyFrameData data in value )
                {
                    m_keyFrames.Add( data.Clone() );
                }

                SortKeyFrames();
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableData source )
        {
            Ensure.Argument.NotNull(source);

            this.TimeBaseType      = source.TimeBaseType;
            this.Name              = source.Name;
            this.IsEnabled         = source.IsEnabled;
            this.IsLoop            = source.IsLoop;
            this.InterpolationType = source.InterpolationType;
            this.KeyFrames         = source.KeyFrames;
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableData Clone()
        {
            return new AnimTableData( this );
        }


        /// <summary>
        /// Sort the key frames by the key frame number.
        /// </summary>
        public void SortKeyFrames()
        {
            if ( m_keyFrames==null ||
                 m_keyFrames.Count<=1 )
            {
                return;
            }

            QuickSortInternal( 0, m_keyFrames.Count-1 );
        }


        /// <summary>
        /// Internal key frame sorting method implementing the quick sort algorithm.
        /// </summary>
        /// <param name="iLeft">The left pivot index.</param>
        /// <param name="iRight">The right pivot index.</param>
        private void QuickSortInternal( int iLeft,
                                        int iRight )
        {
            int l     = iLeft;
            int r     = iRight;
            int pivot = m_keyFrames[(iLeft+iRight)/2].KeyFrame;

            do
            {
                while ( m_keyFrames[l].KeyFrame<pivot &&
                        l<iRight )
                {
                    ++l;
                }

                while ( m_keyFrames[r].KeyFrame>pivot &&
                        iLeft<r )
                {
                    --r;
                }

                if ( l<=r )
                {
                    AnimTableKeyFrameData tmp = m_keyFrames[l];
                    m_keyFrames[l] = m_keyFrames[r];
                    m_keyFrames[r] = tmp;

                    ++l;
                    --r;
                }
            } while ( l<=r );

            if ( iLeft<r )
                QuickSortInternal( iLeft, r );

            if ( l<iRight )
                QuickSortInternal( l, iRight );
        }

        #endregion

        #region Member variables

        private List<AnimTableKeyFrameData> m_keyFrames = null;

        #endregion
    }

    #region Base key frame data

    /// <summary>
    /// Class for storing the key frame data of the animation table.
    /// </summary>
    public abstract class AnimTableKeyFrameData : BaseDataModel<AnimTableKeyFrameData>
    {
        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        public override void Initialize()
        {
            this.KeyFrame = 0;
        }

        #endregion

        #region Properties

        /// <summary>
        /// The frame number of the key frame data.
        /// </summary>
        public int KeyFrame { get; set; }


        /// <summary>
        /// The key frame value in string formats.
        /// </summary>
        public abstract string StrValue { get; set; }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableKeyFrameData source )
        {
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableKeyFrameData Clone()
        {
            return null;
        }

        #endregion
    }

    #endregion

    #region Integer value key frame data

    /// <summary>
    /// Class for storing the key frame data of the animation table.
    /// </summary>
    public class IntAnimTableKeyFrameData : AnimTableKeyFrameData
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public IntAnimTableKeyFrameData()
        {
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="source">The source to copy from.</param>
        public IntAnimTableKeyFrameData( IntAnimTableKeyFrameData source ) :
            this()
        {
            this.Set( source );
        }


        /// <summary>
        /// Initialize the data.
        /// </summary>
        public override void Initialize()
        {
            base.Initialize();

            this.Value = 0;
        }

        #endregion

        #region Properties

        /// <summary>
        /// The animation value for the key frame.
        /// </summary>
        public int Value { get; set; }


        /// <summary>
        /// The key frame value in string formats.
        /// </summary>
        public override string StrValue
        {
            get { return this.Value.ToString( CultureInfo.InvariantCulture ); }
            set
            {
                int iValue;
                if ( int.TryParse( value,
                                   NumberStyles.Number,
                                   CultureInfo.InvariantCulture,
                                   out iValue )==true )
                    this.Value = iValue;
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableKeyFrameData source )
        {
            Ensure.Argument.NotNull( source );

            this.KeyFrame = source.KeyFrame;
            if ( source is IntAnimTableKeyFrameData )
                this.Value = (source as IntAnimTableKeyFrameData).Value;
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableKeyFrameData Clone()
        {
            return new IntAnimTableKeyFrameData( this );
        }

        #endregion
    }

    #endregion

    #region Float value key frame data

    /// <summary>
    /// Class for storing the key frame data of the animation table.
    /// </summary>
    public class FloatAnimTableKeyFrameData : AnimTableKeyFrameData
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public FloatAnimTableKeyFrameData()
        {
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="source">The source to copy from.</param>
        public FloatAnimTableKeyFrameData( FloatAnimTableKeyFrameData source ) :
            this()
        {
            this.Set( source );
        }


        /// <summary>
        /// Initialize the data.
        /// </summary>
        public override void Initialize()
        {
            base.Initialize();

            this.Value = 0.0f;
        }

        #endregion

        #region Properties

        /// <summary>
        /// The animation value for the key frame.
        /// </summary>
        public float Value { get; set; }


        /// <summary>
        /// The key frame value in string formats.
        /// </summary>
        public override string StrValue
        {
            get
            {
                // By default, float.ToString() does not preserve the maximum precision
                // of a float, so we have to specify ToString() with that precision.
                return this.Value.ToString( "G9", CultureInfo.InvariantCulture );
            }
            set
            {
                float fValue;
                if ( float.TryParse( value,
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.Value = fValue;
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set the data from the source.
        /// </summary>
        /// <param name="source">The source to set data from.</param>
        public override void Set( AnimTableKeyFrameData source )
        {
            Ensure.Argument.NotNull( source );

            this.KeyFrame = source.KeyFrame;
            if ( source is FloatAnimTableKeyFrameData )
                this.Value = (source as FloatAnimTableKeyFrameData).Value;
        }


        /// <summary>
        /// Clone the data model.
        /// </summary>
        /// <returns>The new data model.</returns>
        public override AnimTableKeyFrameData Clone()
        {
            return new FloatAnimTableKeyFrameData( this );
        }

        #endregion
    }

    #endregion
}
