﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_0.Minor_0.Build_0.Revision_0
{
    /// <summary>
    /// プロジェクトヘッダー
    /// </summary>
    public sealed class TextureRes : BaseDataModel<TextureRes>
    {
        Bitmap[] m_colorBitmaps;
        Bitmap[] m_alphaBitmaps;

        #region Properties

        /// <summary>
        /// Color Bitmap
        /// </summary>
        public Bitmap[] ColorBitmap
        {
            get { return m_colorBitmaps; }
        }

        /// <summary>
        /// Alpha Bitmap
        /// </summary>
        public Bitmap[] AlphaBitmap
        {
            get { return m_alphaBitmaps; }
        }

        /// <summary>
        /// テクスチャー幅
        /// </summary>
        public int Width
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャー高さ
        /// </summary>
        public int Height
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャーフォーマット
        /// </summary>
        public int Format
        {
            get;
            set;
        }

        /// <summary>
        /// 名前（名前テーブル先頭からのオフセット
        /// </summary>
        public int NumMipLevel
        {
            get;
            set;
        }

        /// <summary>
        /// ラップモード（上位:U 下位：V）
        /// </summary>
        public int WrapMode
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャフィルタ
        /// </summary>
        public FilterMode FilterMode
        {
            get;
            set;
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public int DataSize
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャーのデータ位置（テクスチャーテーブルからのオフセット）
        /// </summary>
        public int DataPos
        {
            get;
            set;
        }

        /// <summary>
        /// // テクスチャーのハンドル
        /// </summary>
        public uint Handle
        {
            get;
            set;
        }

        #endregion

        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public TextureRes()
        {
            m_colorBitmaps = null;
            m_alphaBitmaps = null;
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public TextureRes(TextureRes source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// テクスチャーの設定
        /// </summary>
        /// <param name="colorBmp">RGB Bitmap</param>
        /// <param name="alphaBmp">Alpha Bitmap</param>
        public void SetTexture(
            Bitmap[] colorBmps,
            Bitmap[] alphaBmps,
            int mipmapLevel,
            int dataSize,
            int format)
        {
            m_colorBitmaps = colorBmps;
            m_alphaBitmaps = alphaBmps;

            if ( colorBmps==null )
            {
                this.Width       =  0;
                this.Height      =  0;
                this.Format      = -1;
                this.NumMipLevel =  0;
                this.WrapMode    =  0;
                this.DataSize    =  0;
                this.DataPos     =  0;
            }
            else if ( colorBmps.Length>0 )
            {
                this.Width = colorBmps[0].Width;
                this.Height = colorBmps[0].Height;
//TODO:Format はまだ未完成
                this.Format = format;
                this.NumMipLevel = mipmapLevel;
                this.WrapMode = 0;
                this.DataSize = dataSize;
                this.DataPos = 0;
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(TextureRes source)
        {
            Ensure.Argument.NotNull(source);

            this.Width = source.Width;
            this.Height = source.Height;
            this.Format = source.Format;
            this.NumMipLevel = source.NumMipLevel;
            this.WrapMode = source.WrapMode;
            this.DataSize = source.DataSize;
            this.DataPos = source.DataPos;
            this.Handle = source.Handle;

            this.m_colorBitmaps =
                (source.m_colorBitmaps != null) ?
                    (source.m_colorBitmaps.Clone() as Bitmap[]) : null;

            this.m_alphaBitmaps =
                (source.m_alphaBitmaps != null) ?
                    (source.m_alphaBitmaps.Clone() as Bitmap[]) : null;
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override TextureRes Clone()
        {
            return new TextureRes(this);
        }

        #endregion
    }
}
