﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Threading;
using System.Windows.Forms;

namespace App
{
    /// <summary>
    /// デバッグコンソールクラス。
    /// </summary>
    public sealed partial class DebugConsole : Form
    {
        // インスタンス
        private static DebugConsole s_instance = null;
        // 出力カウンタ
        private static int s_outputCounter = 0;
        // メインスレッドのＩＤ
        private static int s_mainThreadId = 0;
        // オーナー
        private static IDebugConsoleOwner s_owner;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private DebugConsole()
        {
            InitializeComponent();

            // 初期化
            s_mainThreadId = Thread.CurrentThread.ManagedThreadId;

            // タイトルを変更
            string title = Application.ProductName + " : Debug Console";
            this.Text = title;
        }

        /// <summary>
        /// 初期化処理。
        /// </summary>
        [Conditional("DEBUG")]
        public static void Initialize(IDebugConsoleOwner owner)
        {
            if (s_instance != null)
            {
                s_instance.Close();
                s_instance = null;
            }

            s_owner = owner;
            if (s_instance == null)
            {
                s_instance = new DebugConsole();
                s_instance.Show();
                s_instance.Update();
            }
        }

        /// <summary>
        /// 終了処理。
        /// </summary>
        [Conditional("DEBUG")]
        public static void Terminate()
        {
            if (s_instance != null)
            {
                if (!s_instance.IsHandleCreated)
                {
                    s_instance.Close();
                }
            }
        }

        // 書き込みデリゲート
        private delegate void WriteDelegate(string message);

        // Invoke用書き込みメソッド
        private static void _Write(string message) { Write(message); }

        /// <summary>
        /// 書き込み。
        /// </summary>
        [Conditional("DEBUG")]
        public static void Write(string message)
        {
            if (s_instance != null && s_instance.IsHandleCreated)
            {
                // 別スレッドからの呼び出し用
                if (Thread.CurrentThread.ManagedThreadId != s_mainThreadId)
                {
                    s_instance.BeginInvoke(new WriteDelegate(_Write), new object[] { message });
                    return;
                }

                // 通常の呼び出し用
                TextBox tbx    = s_instance.tbxOutput;
                string  output = string.Format("[{0:d3}]{1}", s_outputCounter, message);
                if (tbx.TextLength + output.Length > tbx.MaxLength)
                {
                    tbx.Clear();
                }
                tbx.AppendText(output);

                // 出力カウンタ更新
                if (++s_outputCounter >= 1000)
                {
                    s_outputCounter = 0;
                }
            }
        }

        /// <summary>
        /// 書式指定付き書き込み。
        /// </summary>
        [Conditional("DEBUG")]
        public static void Write(string format, params object[] args)
        {
            Write(string.Format(format, args));
        }

        /// <summary>
        /// 書き込み＋改行。
        /// </summary>
        [Conditional("DEBUG")]
        public static void WriteLine(string message)
        {
            Write(message + "\r\n");
        }

        /// <summary>
        /// 書式指定付き書き込み＋改行。
        /// </summary>
        [Conditional("DEBUG")]
        public static void WriteLine(string format, params object[] args)
        {
            Write(string.Format(format, args) + "\r\n");
        }

        /// <summary>
        /// 条件付き書き込み。
        /// </summary>
        [Conditional("DEBUG")]
        public static void WriteIf(bool condition, string message)
        {
            if (condition)
            {
                Write(message);
            }
        }

        /// <summary>
        /// 条件付き書式指定付き書き込み。
        /// </summary>
        [Conditional("DEBUG")]
        public static void WriteIf(bool condition, string format, params object[] args)
        {
            if (condition)
            {
                Write(format, args);
            }
        }

        /// <summary>
        /// 条件付き書き込み＋改行。
        /// </summary>
        [Conditional("DEBUG")]
        public static void WriteLineIf(bool condition, string message)
        {
            if (condition)
            {
                WriteLine(message);
            }
        }

        /// <summary>
        /// 条件付き書式指定付き書き込み＋改行。
        /// </summary>
        [Conditional("DEBUG")]
        public static void WriteLineIf(bool condition, string format, params object[] args)
        {
            if (condition)
            {
                WriteLine(format, args);
            }
        }

        #region オーバーライド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            // 位置、サイズをデスクトップ隅に設定
            Rectangle desktop = Screen.PrimaryScreen.WorkingArea;
            Size      size    = new Size(desktop.Width / 2, desktop.Height / 2);
            this.Location = new Point(desktop.X, desktop.Bottom - size.Height);
            this.Size     = size;

            base.OnLoad(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            // ESCキーでバッファのクリア
            if (e.KeyCode == Keys.Escape)
            {
                s_instance.tbxOutput.Clear();
                s_outputCounter = 0;
            }
            base.OnKeyDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // 既定の処理
            if (base.ProcessCmdKey(ref msg, keyData))
            {
                return true;
            }

            // メインフレームのショートカット処理
            if (s_owner != null)
            {
                return s_owner.ProcessCommandKey(ref msg, keyData);
            }
#if true    //無限ループになるので削除
            return false;
#else
            else
            {
                return ProcessCmdKey(ref msg, keyData);
            }
#endif
        }
        #endregion
    }
}
