﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Drawing;
using System.Diagnostics;
using System.Xml.Serialization;
using NintendoWare.ToolDevelopmentKit;

namespace App.Data
{
    /// <summary>
    /// ＲＧＢＡカラークラス。
    /// </summary>
    public sealed class Rgba
    {
        private int _r = 255; // -1024 - 1023 も OK
        private int _g = 255; // -1024 - 1023 も OK
        private int _b = 255; // -1024 - 1023 も OK
        private int _a = 255; // -1024 - 1023 も OK

        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Rgba() { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Rgba(Rgba src) { Set(src); }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Rgba(Color src) { Set(src); }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Rgba(int r, int g, int b) { Set(r, g, b, 255); }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Rgba(int r, int g, int b, int a) { Set(r, g, b, a); }

        //---------------------------------------------------------------------
        /// <summary>
        /// Color型から作成。
        /// </summary>
        public static Rgba FromColor(Color src)
        {
            return new Rgba(src);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// クランプ。
        /// 範囲外の値を切り詰めて 0-255 にします。
        /// </summary>
        public static int Clamp(int value)
        {
            if (value < 0) { return 0; }
            if (value > 255) { return 255; }
            return value;
        }

        /// <summary>
        /// ラップ。
        /// 範囲外の値をシフトして 0-255 にします。
        /// </summary>
        public static int Wrap(int value)
        {
            while (value < 0) { value += 256; }
            while (value > 255) { value -= 256; }
            return value;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// Ｒ成分。
        /// </summary>
        [XmlIgnore]
        public int R
        {
            get { return _r; }
            set { _r = value; }
        }

        /// <summary>
        /// 暫定対応
        /// </summary>
        [XmlElement("R")]
        public float RXML
        {
            get { return _r / 255F; }
            set { _r = (int)value * 255; }
        }

        /// <summary>
        /// Ｇ成分。
        /// </summary>
        [XmlIgnore]
        public int G
        {
            get { return _g; }
            set { _g = value; }
        }

        /// <summary>
        /// 暫定対応
        /// </summary>
        [XmlElement("G")]
        public float GXML
        {
            get { return _g / 255F; }
            set { _g = (int)value * 255; }
        }

        /// <summary>
        /// Ｂ成分。
        /// </summary>
        public int B
        {
            get { return _b; }
            set { _b = value; }
        }

        /// <summary>
        /// 暫定対応
        /// </summary>
        [XmlElement("B")]
        public float BXML
        {
            get { return _b / 255F; }
            set { _b = (int)value * 255; }
        }

        /// <summary>
        /// Ａ成分。
        /// </summary>
        public int A
        {
            get { return _a; }
            set { _a = value; }
        }

        /// <summary>
        /// 暫定対応
        /// </summary>
        [XmlElement("A")]
        public float AXML
        {
            get { return _a / 255F; }
            set { _a = (int)value * 255; }
        }

        /// <summary>
        /// 成分（インデクサ）。
        /// 各成分をインデクス指定でアクセスします。
        /// </summary>
        [XmlIgnore]
        public int this[int index]
        {
            get
            {
                Debug.Assert(0 <= index && index <= 3);
                switch (index)
                {
                    case 0: return _r;
                    case 1: return _g;
                    case 2: return _b;
                    case 3: return _a;
                    default: return 0;
                }
            }
            set
            {
                Debug.Assert(0 <= index && index <= 3);
                switch (index)
                {
                    case 0: _r = value; break;
                    case 1: _g = value; break;
                    case 2: _b = value; break;
                    case 3: _a = value; break;
                    default: break;
                }
            }
        }

        /// <summary>
        /// 黒色かどうか（描画色などで特別処理する場合が多いので）。
        /// </summary>
        [XmlIgnore]
        public bool IsBlack
        {
            get { return _r == 0 && _g == 0 && _b == 0; }
        }

        /// <summary>
        /// HDR範囲の値かどうか。
        /// </summary>
        [XmlIgnore]
        public bool IsHdrRange
        {
            get
            {
                if (_r < 0 || _r > 255) { return true; }
                if (_g < 0 || _g > 255) { return true; }
                if (_b < 0 || _b > 255) { return true; }
                if (_a < 0 || _a > 255) { return true; }
                return false;
            }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(Rgba src)
        {
            _r = src._r;
            _g = src._g;
            _b = src._b;
            _a = src._a;
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(Color src)
        {
            _r = src.R;
            _g = src.G;
            _b = src.B;
            _a = src.A;
        }

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(int r, int g, int b, int a)
        {
            _r = r;
            _g = g;
            _b = b;
            _a = a;
        }

        /// <summary>
        /// クランプして設定。
        /// </summary>
        public void SetWithClamp(int r, int g, int b, int a)
        {
            _r = Clamp(r);
            _g = Clamp(g);
            _b = Clamp(b);
            _a = Clamp(a);
        }

        /// <summary>
        /// ラップして設定。
        /// </summary>
        public void SetWithWrap(int r, int g, int b, int a)
        {
            _r = Wrap(r);
            _g = Wrap(g);
            _b = Wrap(b);
            _a = Wrap(a);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// Color型に変換。
        /// </summary>
        public Color ToColor()
        {
            // 範囲外にならないようにする
            return Color.FromArgb(Clamp(_a), Clamp(_r), Clamp(_g), Clamp(_b));
        }

        /// <summary>
        /// Color型に変換（アルファ無視）。
        /// </summary>
        public Color ToColorNoAlpha()
        {
            // 範囲外にならないようにする
            return Color.FromArgb(Clamp(_r), Clamp(_g), Clamp(_b));
        }

        /// <summary>
        /// RgbaColorへ変換します。
        /// </summary>
        /// <returns></returns>
        public RgbaColor ToRgbaColor()
        {
            var rgbaColor = new RgbaColor();
            rgbaColor.R = _r;
            rgbaColor.G = _g;
            rgbaColor.B = _b;
            rgbaColor.A = _a;
            return rgbaColor;
        }

        /// <summary>
        /// Hsv型に変換。
        /// </summary>
        public Hsv ToHsv()
        {
            int h = 0;
            int s = 0;
            int v = 0;

            int r = Wrap(_r);
            int g = Wrap(_g);
            int b = Wrap(_b);

            // 最大値と最小値
            int max = 0;
            int min = 0;
            if (r >= g && r >= b) { max = r; }
            else if (g >= b) { max = g; }
            else { max = b; }
            if (r <= g && r <= b) { min = r; }
            else if (g <= b) { min = g; }
            else { min = b; }

            // 差分
            int dif = max - min;

            // Ｈ成分
            if (dif > 0)
            {
                if (max == g) { h = (int)((b - r) / (float)dif * 60 + 120); }
                else if (max == b) { h = (int)((r - g) / (float)dif * 60 + 240); }
                else if (b > g) { h = (int)((g - b) / (float)dif * 60 + 360); }
                else { h = (int)((g - b) / (float)dif * 60); }

                if (h < 0)
                {
                    h += 360;
                }
            }
            else
            {
                h = 0;
            }

            // Ｓ成分
            if (max > 0)
            {
                s = (int)(dif / (float)max * 255);
            }
            else
            {
                s = 0;
            }

            // Ｖ成分
            v = max;

            return new Hsv(h, s, v);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 等値比較。
        /// </summary>
        public bool Equals(Rgba src)
        {
            if (src == null) { return false; }
            if (src == this) { return true; }

            return
                _r == src._r &&
                _g == src._g &&
                _b == src._b &&
                _a == src._a;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override bool Equals(object obj)
        {
            return Equals(obj as Rgba);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override int GetHashCode()
        {
            return
                _r.GetHashCode() ^
                _g.GetHashCode() ^
                _b.GetHashCode() ^
                _a.GetHashCode();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override string ToString()
        {
            return string.Format("R={0},G={1},B={2},A={3}", _r, _g, _b, _a);
        }
    }
}
